/*!
  \file
  \brief URG センサ制御

  \author Satofumi KAMIMURA

  $Id: UrgCtrl.cpp 299 2008-10-26 12:15:43Z satofumi $

  \todo スレッド動作中はの受信処理と、それ以外の送受信を排他制御させる
*/

#include "UrgCtrl.h"
#include "CaptureSettings.h"
#include "SerialCtrl.h"
#include "ScipHandler.h"
#include "RangeSensorParameter.h"
#include "CaptureSettings.h"
#include "getTicks.h"
#include "Thread.h"
#include "LockGuard.h"
#include "Lock.h"
#include "MathUtils.h"
#include "DetectOS.h"
#include <deque>

#ifdef MSC
#define snprintf _snprintf
#endif

using namespace qrk;


struct UrgCtrl::pImpl
{
  struct ScanData
  {
    std::vector<long> length_data;
    int timestamp;

    ScanData(void) : timestamp(-1)
    {
    }
  };


  class Capture
  {
  public:
    virtual ~Capture(void)
    {
    }

    virtual std::string createCaptureCommand(void) = 0;
    virtual int capture(std::vector<long>& data, int* timestamp) = 0;
    virtual void setCapturesSize(size_t size) = 0;
    virtual size_t capturesSize(void) = 0;
    virtual size_t remainCaptureTimes(void) = 0;
  };


  class RawManualCapture : public Capture
  {
    pImpl* pimpl_;

  public:
    RawManualCapture(pImpl* pimpl) : pimpl_(pimpl)
    {
    }


    ~RawManualCapture(void)
    {
    }


    std::string createCaptureCommand(void)
    {
      // !!! parameter 受信直後に、送受信パラメータへの代入を行う
      // !!! ここでの送受信には、送受信パラメータの内容を用いる

      char buffer[] = "GDbbbbeeeegg\r";
      snprintf(buffer, strlen(buffer) + 1, "GD%04d%04d%02d\r",
               pimpl_->capture_begin_, pimpl_->capture_end_,
               pimpl_->capture_skip_lines_);

      return buffer;
    }


    int capture(std::vector<long>& data, int* timestamp)
    {
      // レーザを点灯させておく
      pimpl_->scip_.setLaserOutput(ScipHandler::On);

      // データ取得コマンドの送信
      std::string command = createCaptureCommand();
      int n = pimpl_->scip_.send(command.c_str(),
                                 static_cast<int>(command.size()));
      if (n != static_cast<int>(command.size())) {
        pimpl_->error_message_ = "Send command:" + command + " fail.";
        return -1;
      }
      pimpl_->scip_.receiveCaptureData(data, timestamp, NULL);
      return data.size();
    }


    void setCapturesSize(size_t size)
    {
      // 何もしない
    }


    size_t capturesSize(void)
    {
      return 1;
    }


    size_t remainCaptureTimes(void)
    {
      return 0;
    }
  };


  class RawAutoCapture : public Capture
  {
    pImpl* pimpl_;
    size_t captures_size_;


  public:
    RawAutoCapture(pImpl* pimpl) : pimpl_(pimpl), captures_size_(1)
    {
    }


    ~RawAutoCapture(void)
    {
    }


    std::string createCaptureCommand(void)
    {
      char buffer[] = "MDbbbbeeeeggstt\r";
      snprintf(buffer, strlen(buffer) + 1, "MD%04d%04d%02d%01d%02d\r",
               pimpl_->capture_begin_, pimpl_->capture_end_,
               pimpl_->capture_skip_lines_,
               pimpl_->capture_frame_interval_,
               pimpl_->capture_times_);

      return buffer;
    }


    int capture(std::vector<long>& data, int* timestamp)
    {
      // スレッドを起動
      if (! pimpl_->thread_.isRunning()) {
        pimpl_->thread_.run(1);
      }

      // 取得済みデータがなければ、戻る
      LockGuard guard(pimpl_->mutex_);
      if (pimpl_->data_buffer_.empty()) {
        return 0;
      }

      std::swap(data, pimpl_->data_buffer_.front().length_data);
      if (timestamp) {
        *timestamp = pimpl_->data_buffer_.front().timestamp;
      }
      pimpl_->data_buffer_.pop_front();

      return data.size();
    }


    void setCapturesSize(size_t size)
    {
      captures_size_ = size;
    }


    size_t capturesSize(void)
    {
      return captures_size_;
    }


    size_t remainCaptureTimes(void)
    {
      LockGuard guard(pimpl_->mutex_);
      return pimpl_->remain_times_;
    }
  };


  class RawIntensityCapture : public Capture
  {
    pImpl* pimpl_;
    size_t captures_size_;


  public:
    RawIntensityCapture(pImpl* pimpl) : pimpl_(pimpl), captures_size_(1)
    {
    }


    ~RawIntensityCapture(void)
    {
    }


    std::string createCaptureCommand(void)
    {
      char buffer[] = "MEbbbbeeeeggstt\r";
      snprintf(buffer, strlen(buffer) + 1, "ME%04d%04d%02d%01d%02d\r",
               0, 1, 2, 0, 1);
      return buffer;
    }


    int capture(std::vector<long>& data, int* timestamp)
    {
      // !!!
      return -1;
    }


    void setCapturesSize(size_t size)
    {
      captures_size_ = size;
    }


    size_t capturesSize(void)
    {
      return captures_size_;
    }

    size_t remainCaptureTimes(void)
    {
      LockGuard guard(pimpl_->mutex_);
      return pimpl_->remain_times_;
    }
  };


  std::string error_message_;
  Connection* con_;
  SerialCtrl* serial_;          //!< 有効な接続オブジェクトがないときに用いる
  ScipHandler scip_;
  RangeSensorParameter parameters_;
  int timestamp_offset_;

  RangeCaptureMode capture_mode_;
  RawManualCapture manual_capture_;
  RawAutoCapture auto_capture_;
  RawIntensityCapture intensity_capture_;
  Capture* capture_;
  Thread thread_;
  Lock mutex_;

  std::deque<ScanData> data_buffer_;

  int capture_begin_;
  int capture_end_;
  int capture_skip_lines_;
  int capture_skip_frames_;

  size_t capture_frame_interval_;
  size_t capture_times_;

  size_t remain_times_;


  pImpl(void) : error_message_("no error."), con_(NULL), serial_(NULL),
                timestamp_offset_(0), capture_mode_(ManualCapture),
                manual_capture_(this), auto_capture_(this),
                intensity_capture_(this), capture_(&manual_capture_),
                thread_(&capture_thread, this),
                capture_begin_(0), capture_end_(0),
                capture_skip_lines_(1), capture_skip_frames_(0),
                capture_frame_interval_(0), capture_times_(0),
                remain_times_(0)
  {
  }


  ~pImpl(void)
  {
    // con_ は外部からセットされる可能性があるため、このクラスでは解放しない
    // serial_ のみ解放する
    delete serial_;
  }


  void initializeSerial(void)
  {
    if (! serial_) {
      serial_ = new SerialCtrl;
    }
    con_ = serial_;
  }


  bool connect(const char* device, long baudrate)
  {
    disconnect();
    if (! con_) {
      initializeSerial();
    }
    scip_.setConnection(con_);

    // ボーレートを検出した上でのデバイスとの接続
    if (! scip_.connect(device, baudrate)) {
      error_message_ = scip_.what();
      return false;
    }

    if (! loadParameter()) {
      return false;
    }
    updateCaptureParameters();

    return true;
  }


  bool loadParameter(void)
  {
    // URG パラメータの取得
    RangeSensorParameter parameters;
    if (! scip_.loadParameter(parameters)) {
      error_message_ = scip_.what();
      return false;
    }
    std::swap(parameters_, parameters);

    // !!! capture_begin_, capture_end_ との調整をすべき
    // !!! std::min(), std::max() を使うこと

    return true;
  }


  void disconnect(void)
  {
    stop();
    if (con_) {
      con_->disconnect();
    }
  }


  void updateCaptureParameters(void)
  {
    capture_begin_ = parameters_.area_min;
    capture_end_ = parameters_.area_max;
  }


  // AutoCapture, IntensityCapture のデータ取得を行う
  static int capture_thread(void* args)
  {
    pImpl* obj = static_cast<pImpl*>(args);

    // 設定に基づいて、データ受信コマンドを作成して発行
    std::string capture_command = obj->capture_->createCaptureCommand();
    int n = obj->scip_.send(capture_command.c_str(), capture_command.size());
    if (n != static_cast<int>(capture_command.size())) {
      obj->error_message_ = capture_command + " fail.";
      return -1;
    }

    // 受信待ち
    ScanData data;
    int remain_times = 100;
    while (1) {
      // 受信完了、およびエラーで抜ける
      CaptureType type =
        obj->scip_.receiveCaptureData(data.length_data, &data.timestamp,
                                      &remain_times);
      if (type == Mx_Reply) {
        // MS/MD の応答パケットの場合、次のデータを待つ
        continue;
      }

      if (! ((type == MD) || (type == MS) || (type == ME))) {
        // 受信データでなければ、スレッド処理を中断する
        break;
      }

      LockGuard guard(obj->mutex_);
      std::deque<ScanData>& data_buffer = obj->data_buffer_;

      // 古くなったデータを取り除く
      size_t current_size = data_buffer.size();
      int erase_size = current_size - obj->capture_->capturesSize();
      if (erase_size > 0) {
        data_buffer.erase(data_buffer.begin(),
                          data_buffer.begin() + erase_size);
      }

      // 今回のデータを追加
      ScanData dummy_data;
      data_buffer.push_back(dummy_data);
      std::swap(data_buffer.back(), data);
      obj->remain_times_ = remain_times;
    }

    return 0;
  }


  int capture(std::vector<long>& data, int* timestamp)
  {
    int n = capture_->capture(data, timestamp);
    if (n < 0) {
      error_message_ = scip_.what();
      return n;
    }

    if (timestamp) {
      *timestamp = *timestamp - timestamp_offset_;
    }
    return n;
  }


  int captureIntensity(std::vector<long>& data, int* timestamp)
  {
    // !!!
    if (timestamp) {
      *timestamp = *timestamp - timestamp_offset_;
    }

    return -1;
  }


  bool setTimestamp(int ticks)
  {
    // TM0 モードに遷移
    // !!! true, false をキーワードで置換すること
    if (! scip_.setRawTimestampMode(true)) {
      return false;
    }

    // TM1 のタイムスタンプを取得し、通信遅延と URG のタイムスタンプを取得する
    int urg_timestamp = 0;
    int first_ticks = getTicks();
    if (scip_.rawTimestamp(&urg_timestamp)) {
      int delay = getTicks() - first_ticks;
      timestamp_offset_ =  urg_timestamp - ticks + delay;
    }

    // URG タイムスタンプ用のオフセット時間を計算し、TM2 で抜ける
    if (! scip_.setRawTimestampMode(false)) {
      return false;
    }

    return true;
  }


  int rad2index(const double radian) const
  {
    int area_total = parameters_.area_total;
    int index =
      static_cast<int>((radian * area_total) / (2.0 * M_PI))
      + parameters_.area_front;

    if (index < 0) {
      index = 0;
    } else if (index > parameters_.area_max) {
      index = parameters_.area_max;
    }
    return index;
  }


  void stop(void)
  {
    if (capture_mode_ == ManualCapture) {
      return;
    }

    // QT コマンドの発行
    scip_.setLaserOutput(ScipHandler::Off);

    // 応答を待つ
    if (thread_.isRunning()) {
      thread_.wait();
    }
  }
};


UrgCtrl::UrgCtrl(void) : pimpl(new pImpl)
{
}


UrgCtrl::~UrgCtrl(void)
{
}


const char* UrgCtrl::what(void) const
{
  return pimpl->error_message_.c_str();
}


bool UrgCtrl::connect(const char* device, long baudrate)
{
  return pimpl->connect(device, baudrate);
}


void UrgCtrl::setConnection(Connection* con)
{
  pimpl->con_ = con;
}


Connection* UrgCtrl::connection(void)
{
  return pimpl->con_;
}


void UrgCtrl::disconnect(void)
{
  pimpl->disconnect();
}


bool UrgCtrl::isConnected(void)
{
  return (pimpl->con_ == NULL) ? false : pimpl->con_->isConnected();
}


long UrgCtrl::minDistance(void) const
{
  return pimpl->parameters_.distance_min;
}


long UrgCtrl::maxDistance(void) const
{
  return pimpl->parameters_.distance_max;
}


int UrgCtrl::maxScanLines(void) const
{
  // +1 は、parameters_ が未初期化のときに new long [0] しないための処置
  return pimpl->parameters_.area_max + 1;
}


void UrgCtrl::setCapturesSize(size_t size)
{
  pimpl->capture_->setCapturesSize(size);
}


size_t UrgCtrl::remainCaptureTimes(void)
{
  return pimpl->capture_->remainCaptureTimes();
}


int UrgCtrl::scanMsec(void) const
{
  int scan_rpm = pimpl->parameters_.scan_rpm;
  return (scan_rpm <= 0) ? 1 : (1000 * 60 / scan_rpm);

}


void UrgCtrl::setCaptureMode(RangeCaptureMode mode)
{
  // capture を停止し、モード変更を行う。capture の再開は行わない
  stop();
  if (mode == ManualCapture) {
    pimpl->capture_ = &pimpl->manual_capture_;

  } else if (mode == AutoCapture) {
    pimpl->capture_ = &pimpl->auto_capture_;

  } else if (mode == IntensityCapture) {
    pimpl->capture_ = &pimpl->intensity_capture_;
  }

  pimpl->capture_mode_ = mode;
}


RangeCaptureMode UrgCtrl::captureMode(void)
{
  return pimpl->capture_mode_;
}


void UrgCtrl::setCaptureRange(int begin_index, int end_index)
{
  // !!! 排他制御

  pimpl->capture_begin_ = begin_index;
  pimpl->capture_end_ = end_index;
}


void UrgCtrl::setCaptureFrameInterval(size_t interval)
{
  // !!! 排他制御

  pimpl->capture_frame_interval_ = interval;
}


void UrgCtrl::setCaptureTimes(size_t times)
{
  // !!! 排他制御

  // !!! 範囲制限の判定処理
  pimpl->capture_times_ = times;
}



void UrgCtrl::setCaptureSkipLines(size_t skip_lines)
{
  // !!!
}


int UrgCtrl::capture(std::vector<long>& data, int* timestamp)
{
  // !!! 未接続ならば、戻す
  return pimpl->capture(data, timestamp);
}


int UrgCtrl::captureIntensity(std::vector<long>& data, int* timestamp)
{
  // !!! 未接続ならば、戻す
  return pimpl->captureIntensity(data, timestamp);
}


void UrgCtrl::stop(void)
{
  pimpl->stop();
}


bool UrgCtrl::setTimestamp(int ticks)
{
  // !!! 未接続ならば、戻す

  return pimpl->setTimestamp(ticks);
}


bool UrgCtrl::setLaserOutput(bool on)
{
  // !!! 未接続ならば、戻す

  return pimpl->scip_.setLaserOutput(on, ScipHandler::Force);
}


double UrgCtrl::index2rad(const int index) const
{
  int index_from_front = index - pimpl->parameters_.area_front;
  return index_from_front * (2.0 * M_PI) / pimpl->parameters_.area_total;
}


int UrgCtrl::rad2index(const double radian) const
{
  return pimpl->rad2index(radian);
}


void UrgCtrl::setParameter(const RangeSensorParameter& parameter)
{
  pimpl->parameters_ = parameter;
  pimpl->updateCaptureParameters();
}


RangeSensorParameter UrgCtrl::parameter(void)
{
  return pimpl->parameters_;
}


bool UrgCtrl::loadParameter(void)
{
  return pimpl->loadParameter();
}


bool UrgCtrl::versionLines(std::vector<std::string>& lines)
{
  // !!! 未接続ならば、戻す

  return pimpl->scip_.versionLines(lines);
}
