#ifndef RANGE_SENSOR_H
#define RANGE_SENSOR_H

/*!
  \file
  \brief レンジセンサのインターフェース定義

  \author Satofumi KAMIMURA

  $Id: RangeSensor.h 290 2008-10-22 15:35:54Z satofumi $
*/

#include "stddef.h"
#include <vector>
#include <string>


namespace qrk
{
  class Connection;

  /*!
    \brief 測域センサ・インターフェース
  */
  class RangeSensor
  {
  public:
    virtual ~RangeSensor(void) {}


    /*!
      \brief 内部状態を返す

      \return 内部状態を示す文字列

      <strong>Example</strong>
\code
UrgCtrl sensor;
if (! sensor.connect(device, baudrate)) {
  printf("connect: %s\n", sensor.what());
  exit(1);
} \endcode
    */
    virtual const char* what(void) const = 0;


    /*!
      \brief 接続

      \param[in] device 接続デバイス名
      \param[in] baudrate 接続ボーレート

      \retval true 成功
      \retval false 失敗

      <strong>Example</strong>
\code
const char device[] = "/dev/ttyACM0";
const long baudrate = 115200;
UrgCtrl sensor;
if (! sensor.connect(device, baudrate)) {
  printf("connect: %s\n", sensor.what());
  exit(1);
} \endcode
    */
    virtual bool connect(const char* device, long baudrate) = 0;


    /*!
      \brief 接続オブジェクトの設定

      \param[in] con 接続オブジェクト
    */
    virtual void setConnection(Connection* con) = 0;


    /*!
      \brief 接続オブジェクトの取得

      \return 接続オブジェクト
    */
    virtual Connection* connection(void) = 0;


    /*!
      \brief 切断
    */
    virtual void disconnect(void) = 0;


    /*!
      \brief 接続状態を返す

      \retval true 接続中
      \retval false 切断中
    */
    virtual bool isConnected(void) = 0;


    /*!
      \brief 有効距離の最小値を取得

      \return 有効距離の最小値
    */
    virtual long minDistance(void) const = 0;


    /*!
      \brief 有効距離の最大値を取得

      \return 有効距離の最大値
    */
    virtual long maxDistance(void) const = 0;


    /*!
      \brief １周の最大スキャン数

      \return １周の最大スキャン数
    */
    virtual int maxScanLines(void) const = 0;

    virtual int scanMsec(void) const = 0;


    /*!
      \brief データ取得

      データ取得を行い、バッファに格納する

      \param[out] data 取得データ用バッファ
      \param[in] max_size 取得データ用バッファの最大サイズ
      \param[out] timestamp タイムスタンプ

      \return 取得データ数
      \retval < 0 取得失敗
    */
    virtual int capture(std::vector<long>& data, int* timestamp) = 0;


    virtual int captureIntensity(std::vector<long>& data,
                                 int* timestamp) = 0;


    /*!
      \brief センサが返すタイムスタンプ値の設定

      capture() で取得できるタイムスタンプを設定する

      \param[in] ticks その時刻のタイムスタンプの設定値

      \retval true 成功
      \retval false 失敗
    */
    virtual bool setTimestamp(int ticks = 0) = 0;


    virtual bool setLaserOutput(bool on) = 0;


    /*!
      \brief データインデックスのラジアン角度変換

      センサ前面が、0.0 [rad] に相当する。

      \param[in] index データインデックス

      \return ラジアン角度

      \image html sensor_radian.png センサ上面からの角度対応図
    */
    virtual double index2rad(const int index) const = 0;


    /*!
      \brief ラジアン角度のデータインデックス変換

      センサ前面が、0.0 [rad] に相当する。

      \param[in] radian ラジアン角度

      \return データインデックス

      \image html sensor_radian.png センサ上面からの角度対応図
    */
    virtual int rad2index(const double radian) const = 0;
  };
};

#endif /* !RANGE_SENSOR_H */
