/*!
  \file
  \brief SCIP 通信処理

  \author Satofumi KAMIMURA

  $Id: ScipHandler.cpp 304 2008-10-27 00:06:51Z satofumi $
*/

#include "ScipHandler.h"
#include "RangeSensorParameter.h"
#include "CaptureSettings.h"
#include "Connection.h"
#include "ConnectionUtils.h"
#include "DetectOS.h"

#ifdef MSC
#define snprintf _snprintf
#endif


using namespace qrk;

struct ScipHandler::pImpl
{
  enum {
    ContinuousTimeout = 100,
    FirstTimeout = 1000,

    BufferSize = 64 + 1 + 16, // データ長 + チェックサム + 改行 + 予備

    ResponseTimeout = -1,
    MismatchResponse = -2,
    Scip11Response = -14,
  };

  typedef enum {
    LaserUnknown = 0,
    LaserOn,
    LaserOff,
  } LaserState;

  std::string error_message_;
  Connection* con_;
  LaserState laser_state_;
  bool mx_capturing_;

  std::vector<long> intensity_data_;


  pImpl(void)
    : error_message_("no error."), con_(NULL), laser_state_(LaserUnknown),
      mx_capturing_(false)
  {
  }


  bool connect(const char* device, long baudrate)
  {
    if (! con_->connect(device, baudrate)) {
      error_message_ = con_->what();
      return false;
    }

    long try_baudrates[] = { 115200, 19200, 38400, };
    size_t try_size = sizeof(try_baudrates) / sizeof(try_baudrates[0]);

    // 接続したいボーレートを配列の先頭と入れ換える
    for (size_t i = 1; i < try_size; ++i) {
      if (baudrate == try_baudrates[i]) {
        std::swap(try_baudrates[0], try_baudrates[i]);
        break;
      }
    }

    // 指定のボーレートで接続し、応答が返されるかどうか試す
    for (size_t i = 0; i < try_size; ++i) {

      // ホスト側のボーレートを変更
      if (! con_->setBaudrate(try_baudrates[i])) {
        error_message_ = con_->what();
        return false;
      }

      // QT の発行
      int return_code = -1;
      char qt_expected_response[] = { 0, -1 };
      if (response(return_code, "QT\r", qt_expected_response)) {
        laser_state_ = LaserOff;
        return changeBothBaudrate(baudrate);

      } else if (return_code == ResponseTimeout) {
        // ボーレートが違っていて、通信できなかったとみなす
        error_message_ = "baudrate is not detected.";
        continue;

      } else if (return_code == MismatchResponse) {
        // MD/MS の応答とみなし、受信データを読み飛ばす
        skip(con_, ContinuousTimeout);
        return changeBothBaudrate(baudrate);

      } else if (return_code == Scip11Response) {
        // SCIP1.1 プロトコルの場合のみ、SCIP2.0 を投げる
        // !!! SCIP2.0 の送信を行い、ボーレートを変更する
        char scip20_expected_response[] = { 0, -1 };
        if (! response(return_code, "SCIP2.0\r", scip20_expected_response)) {
          error_message_ =
            "SCIP1.1 protocol is not supported. Please update URG firmware.";
          return false;
        }
        return changeBothBaudrate(baudrate);

      } else if (return_code == 0xE) {
        // TM モードとみなし、TM2 を発行する
        char tm2_expected_response[] = { 0, -1 };
        if (response(return_code, "TM2\r", tm2_expected_response)) {
          return changeBothBaudrate(baudrate);
        }
      }
    }
    return false;
  }


  bool changeBothBaudrate(long baudrate)
  {
    // 既に目標対象のボーレート値ならば、成功とみなす
    // この関数は、ScipHandler::connect() か、それ以後でないと呼ばれないため
    if (con_->baudrate() == baudrate) {
      return true;
    }

    // URG 側のボーレートを変更
    if (! changeBaudrate(baudrate)) {
      error_message_ = "SS command fail.";
      return false;
    }

    // ホスト側のボーレートを変更
    return con_->setBaudrate(baudrate);
  }


  bool changeBaudrate(long baudrate)
  {
    if (! ((baudrate == 19200) || (baudrate == 38400) ||
           (baudrate == 57600) || (baudrate == 115200))) {
      error_message_ = "Invalid baudrate value.";
      return false;
    }

    // SS を送信し、URG 側のボーレートを変更する
    char send_buffer[] = "SSxxxxxx\r";
    snprintf(send_buffer, 10, "SS%06ld\r", baudrate);
    int return_code = -1;
    // !!! 既に設定対象のボーレート、の場合の戻り値を ss_expected... に追加する
    char ss_expected_response[] = { 0, -1 };
    if (! response(return_code, send_buffer, ss_expected_response)) {
      error_message_ = "Baudrate change fail.";
      return false;
    }

    return true;
  }


  bool loadParameter(RangeSensorParameter& parameters)
  {
    // PP の送信とデータの受信
    int return_code = -1;
    char pp_expected_response[] = { 0, -1 };
    std::vector<std::string> lines;
    if (! response(return_code, "PP\r", pp_expected_response, &lines)) {
      error_message_ = "PP fail.";
      return false;
    }

    // PP 応答内容の格納
    if (lines.size() != 8) {
      error_message_ = "Invalid PP response.";
      return false;
    }

    // !!! チェックサムの評価を行うべき

    int modl_length = lines[RangeSensorParameter::MODL].size();
    // 最初のタグと、チェックサムを除いた文字列を返す
    if (modl_length > (5 + 2)) {
      modl_length -= (5 + 2);
    }
    parameters.model = lines[RangeSensorParameter::MODL].substr(5, modl_length);

    parameters.distance_min = substr2int(lines[RangeSensorParameter::DMIN], 5);
    parameters.distance_max = substr2int(lines[RangeSensorParameter::DMAX], 5);
    parameters.area_total = substr2int(lines[RangeSensorParameter::ARES], 5);
    parameters.area_min = substr2int(lines[RangeSensorParameter::AMIN], 5);
    parameters.area_max = substr2int(lines[RangeSensorParameter::AMAX], 5);
    parameters.area_front = substr2int(lines[RangeSensorParameter::AFRT], 5);
    parameters.scan_rpm = substr2int(lines[RangeSensorParameter::SCAN], 5);

    return true;
  }


  int substr2int(const std::string& line, int from_n,
                 int length = std::string::npos)
  {
    return atoi(line.substr(from_n, length).c_str());
  }


  bool response(int& return_code, const char send_command[],
                char expected_response[],
                std::vector<std::string>* lines = NULL)
  {
    return_code = -1;
    if (! con_) {
      error_message_ = "no connection.";
      return false;
    }

    int send_size = strlen(send_command);
    con_->send(send_command, send_size);

    // エコーバックの受信
    char buffer[BufferSize];
    int recv_size = readline(con_, buffer, BufferSize, FirstTimeout);
    if (recv_size < 0) {
      error_message_ = "response timeout.";
      return_code = ResponseTimeout;
      return false;
    }
    if ((recv_size != (send_size - 1)) ||
        (strncmp(buffer, send_command, recv_size))) {
      error_message_ = "mismatch response: " + std::string(buffer);
      return_code = MismatchResponse;
      return false;
    }

    // 応答の受信
    // !!! 上記の処理となるべく共通にする
    // !!! SCIP1.1 プロトコルの応答は、負号を付加した上で return_code に格納する
    recv_size = readline(con_, buffer, BufferSize, ContinuousTimeout);
    if (recv_size < 0) {
      // !!! この処理をくくる
      error_message_ = "response timeout.";
      return_code = ResponseTimeout;
      return false;
    }
    if (recv_size == 3) {
      // ３文字ならば、SCIP2.0 とみなしてチェックサムを確認する
      // !!! チェックサムの確認
      buffer[2] = '\0';
      return_code = strtol(buffer, NULL, 16);

    } else if (recv_size == 1) {
      // １文字ならば、SCIP1.1 とみなして 16進変換した値に負号をつけて返す
      // !!!
      buffer[1] = '\0';
      return_code = -strtol(buffer, NULL, 16);
    }

    // データ領域の受信
    // １行読み出し、改行のみならば終了とみなす
    do {
      recv_size = readline(con_, buffer, BufferSize, ContinuousTimeout);
      if (lines && (recv_size > 0)) {
        lines->push_back(buffer);
      }
    } while (recv_size > 0);

    // !!! 実際にはループにする
    // !!! かつ、受信データをポインタが NULL でなければ格納する

    for (int i = 0; expected_response[i] != -1; ++i) {
      if (return_code == expected_response[i]) {
        return true;
      }
    }
    return false;
  }


  bool setLaserOutput(bool on, bool force)
  {
    if (((on == true) && (laser_state_ == LaserOn)) ||
        ((on == false) && (laser_state_ == LaserOff))) {
      if (! force) {
        // レーザ出力が現在の状態と同じならば戻る
        // 強制設定フラグが true のときは戻らずに設定を行う
        return true;
      }
    }

    if (on) {
      int return_code = -1;
      char expected_response[] = { 0, -1 };
      if (! response(return_code, "BM\r", expected_response)) {
        error_message_ = "BM fail.";
        return false;
      }
      laser_state_ = LaserOn;
      return true;

    } else {
      // "QT"
      con_->send("QT\r", 3);
      if (! mx_capturing_) {
        // BM を打ち消すための QT では、応答を待つべき
        int return_code = -1;
        char qt_expected_response[] = { 0, -1 };
        if (! response(return_code, "QT\r", qt_expected_response)) {
          return false;
        }
        laser_state_ = LaserOff;
        return true;

      } else {
        // MD を中断するための QT では、応答を待ってはならない
        // 応答は、受信スレッド内で処理される
      }

      return true;
    }
  }


  CaptureType receiveCaptureData(std::vector<long>& data, int* timestamp,
                                 int* remain_times)
  {
    int line_count = 0;
    intensity_data_.clear();
    data.clear();

    std::string remain_string;

    CaptureSettings settings;
    std::string left_packet_data;
    char buffer[BufferSize];

    // !!! 以下の処理が長い。関数化すべき
    CaptureType type = TypeUnknown;
    int timeout = FirstTimeout;
    int line_size = 0;
    while ((line_size = readline(con_, buffer, BufferSize, timeout)) > 0) {
      //fprintf(stderr, "% 3d: %s\n", line_count, buffer);

      // チェックサムの確認
      if (line_count != 0) {
        if (! checkSum(buffer, line_size - 1, buffer[line_size - 1])) {
          return InvalidData;
        }
      }

      if (line_count == 0) {

        // エコーバック
        std::string line = buffer;
        if ((! line.compare(0, 2, "GD")) || (! line.compare(0, 2, "GS"))) {
          if (! parseGdEchoback(settings, line)) {
            break;
          }
          type = (line[1] = 'D') ? GD : GS;
          data.reserve(settings.capture_last + 1);
          intensity_data_.reserve(settings.capture_last + 1);

        } else if ((! line.compare(0, 2, "MD")) ||
                   (! line.compare(0, 2, "MS"))) {
          if (! parseMdEchoback(settings, line)) {
            break;
          }
          type = (line[1] = 'D') ? MD : MS;
          laser_state_ = LaserOn;
          data.reserve(settings.capture_last + 1);
          intensity_data_.reserve(settings.capture_last + 1);

        } else if (! line.compare(0, 2, "ME")) {
          if (! parseMeEchoback(settings, line)) {
            break;
          }
          type = ME;
          laser_state_ = LaserOn;
          data.reserve(settings.capture_last + 1);
          intensity_data_.reserve(settings.capture_last + 1);

        } else if (! line.compare(0, 2, "QT")) {
          settings.remain_times = 0;
          laser_state_ = LaserOff;
          mx_capturing_ = false;

        } else {
          //fprintf(stderr, "invalid data: %s\n", buffer);
          return InvalidData;
        }

      } else if (line_count == 1) {
        // 応答コード
        // !!! 長さが 2 + 1 かのチェックをすべき
        buffer[2] = '\0';
        settings.error_code = atoi(buffer);

        // !!! "00P" との比較をすべき
        if ((settings.error_code == 0) && ((type == MD) || (type == MS))) {
          type = Mx_Reply;
        }

      } else if (line_count == 2) {
        // タイムスタンプ
        if (timestamp) {
          *timestamp = encode(buffer, 4);
        }
      } else {
        if (line_count == 3) {
          // 受信データがない先頭からの領域を、ダミーデータで埋める
          for (int i = 0; i < settings.capture_first; ++i) {
            data.push_back(0);
            intensity_data_.push_back(0);
          }
        }
        // 距離データ
        left_packet_data =
          addLengthData(data, std::string(buffer),
                        left_packet_data, settings.data_byte,
                        settings.skip_lines);
      }
      ++line_count;
      timeout = ContinuousTimeout;
    }

    // !!! type が距離データ取得のときは、正常に受信が完了したか、を確認すべき

    if (remain_times) {
      *remain_times = settings.remain_times;
    }
    return type;
  }


  bool parseGdEchoback(CaptureSettings& settings, const std::string& line)
  {
    if (line.size() != 12) {
      error_message_ = "Invalid Gx packet has arrived.";
      return false;
    }

    settings.capture_first = substr2int(line, 2, 4);
    settings.capture_last = substr2int(line, 6, 4);
    settings.skip_lines = substr2int(line, 10, 2);
    settings.data_byte = (line[1] == 'D') ? 3 : 2;

    return true;
  }


  bool parseMdEchoback(CaptureSettings& settings, const std::string& line)
  {
    if (line.size() != 15) {
      error_message_ = "Invalid Mx packet has arrived.";
      return false;
    }

    settings.capture_first = substr2int(line, 2, 4);
    settings.capture_last = substr2int(line, 6, 4);
    settings.skip_lines = substr2int(line, 10, 2);
    settings.skip_frames = substr2int(line, 12, 1);
    settings.remain_times = substr2int(line, 13, 2);
    settings.data_byte = (line[1] == 'D') ? 3 : 2;

    if (settings.remain_times == 1) {
      // 最後のデータ取得で、レーザを消灯扱いにする
      mx_capturing_ = false;

    } else {
      mx_capturing_ = true;
    }

    return true;
  }


  bool parseMeEchoback(CaptureSettings& settings, const std::string& line)
  {
    if (line.size() != 15) {
      error_message_ = "Invalid ME packet has arrived.";
      return false;
    }

    settings.capture_first = substr2int(line, 2, 4);
    settings.capture_last = substr2int(line, 6, 4);
    settings.skip_lines = substr2int(line, 10, 2);
    settings.skip_frames = substr2int(line, 12, 1);
    settings.remain_times = substr2int(line, 13, 2);
    settings.data_byte = 3;

    return true;
  }


  std::string addLengthData(std::vector<long>& data,
                            const std::string& line,
                            const std::string& left_packet_data,
                            const size_t data_byte, const int skip_lines = 1)
  {
    if (line.empty()) {
      // 空行の場合、戻る
      return left_packet_data;
    }

    // 端数。次回に処理する分
    std::string left_byte = left_packet_data;

    size_t data_size = (left_byte.size() + (line.size() - 1)) / data_byte;
    size_t n = data_size * data_byte - left_byte.size();
    for (size_t i = 0; i < n; ++i) {
      left_byte.push_back(line[i]);
      if (left_byte.size() >= data_byte) {
        // データを距離に変換して、格納
        long length = encode(&left_byte[0], data_byte);
        for (int j = 0; j < skip_lines; ++j) {
          data.push_back(length);
        }
        left_byte.clear();
      }
    }
    left_byte += line.substr(n, (line.size() - n) - 1);

    return left_byte;
  }
};


ScipHandler::ScipHandler(void) : pimpl(new pImpl)
{
}


ScipHandler::~ScipHandler(void)
{
}


const char* ScipHandler::what(void) const
{
  return pimpl->error_message_.c_str();
}


long ScipHandler::encode(const char* data, int size)
{
  int value = 0;

  for (int i = 0; i < size; ++i) {
    value <<= 6;
    value &= ~0x3f;
    value |= data[i] - 0x30;
  }
  return value;
}


bool ScipHandler::checkSum(char* buffer, int size, char actual_sum)
{
  char expected_sum = 0x00;
  for (int i = 0; i < size; ++i) {
    expected_sum += buffer[i];
  }
  expected_sum = (expected_sum & 0x3f) + 0x30;

#if 0
  if (expected_sum != actual_sum) {
    fprintf(stderr, "checksum error!\n");
    for (int i = 0; i < size; ++i) {
      fprintf(stderr, "%c", buffer[i]);
    }
    fprintf(stderr, ", %d\n\n", size);
  }
#endif
  return (expected_sum == actual_sum) ? true : false;
}


void ScipHandler::setConnection(Connection* con)
{
  pimpl->con_ = con;
}


Connection* ScipHandler::connection(void)
{
  return pimpl->con_;
}


bool ScipHandler::connect(const char* device, long baudrate)
{
  return pimpl->connect(device, baudrate);
}


int ScipHandler::send(const char data[], int size)
{
  return pimpl->con_->send(data, size);
}


int ScipHandler::recv(char data[], int size, int timeout)
{
  return pimpl->con_->recv(data, size, timeout);
}


bool ScipHandler::loadParameter(RangeSensorParameter& parameters)
{
  return pimpl->loadParameter(parameters);
}


bool ScipHandler::versionLines(std::vector<std::string>& lines)
{
  int return_code = -1;
  char expected_response[] = { 0, -1 };
  if (! pimpl->response(return_code, "VV\r", expected_response, &lines)) {
    return false;
  }
  return true;
}


bool ScipHandler::setRawTimestampMode(bool on)
{
  char send_command[] = "TMx\r";
  send_command[2] = (on) ? '0' : '2';

  // TM0 or TM2 の送信
  int return_code = -1;
  char expected_response[] = { 0, -1 };
  if (! pimpl->response(return_code, send_command, expected_response)) {
    pimpl->error_message_ = (on) ? "TM1 fail." : "TM2 fail.";
    return false;
  }
  if (on) {
    pimpl->laser_state_ = pImpl::LaserOff;
  }
  return true;
}


bool ScipHandler::rawTimestamp(int* timestamp)
{
  // TM1 の値を返す
  int return_code = -1;
  char expected_response[] = { 0, -1 };
  std::vector<std::string> lines;
  if (! pimpl->response(return_code, "TM1\r", expected_response, &lines)) {
    pimpl->error_message_ = "TM1 fail.";
    return false;
  }

  if ((lines.size() != 1) || (lines[0].size() != 5)) {
    pimpl->error_message_ = "response mismatch.";
    return false;
  }

  *timestamp = encode(lines[0].c_str(), 4);
  return true;
}


bool ScipHandler::setLaserOutput(bool on, bool force)
{
  return pimpl->setLaserOutput(on, force);
}


CaptureType ScipHandler::receiveCaptureData(std::vector<long>& data,
                                            int* timestamp, int* remain_times)
{
  return pimpl->receiveCaptureData(data, timestamp, remain_times);
}
