/*!
  \file
  \brief モニタ対応の通信インターフェース

  \author Satofumi KAMIMURA

  $Id: mConnection.cpp 293 2008-10-23 14:33:32Z satofumi $

  \todo 接続自体を記録する必要があるか、を確認する
  \todo what() の処理をどう記録するか、を確認する
*/

#include "mConnection.h"
#include "MonitorDataHandler.h"
#include "LogNameHolder.h"
#include <string>

using namespace qrk;


struct mConnection::pImpl
{
  Connection* con_;
  MonitorDataHandler* handler_;
  MonitorModeManager::MonitorMode mode_;

  long baudrate_;


  pImpl(Connection* connection)
    : con_(connection), handler_(NULL), mode_(MonitorModeManager::Undefined),
      baudrate_(0)
  {
  }


  bool connect(const char* device, long baudrate)
  {
    if (! handler_) {
      std::string log_name = LogNameHolder::singleton()->name("mConnection");
      mode_ = MonitorModeManager::singleton()->mode();
      handler_ = new MonitorDataHandler(log_name, mode_);
    }

    // connect() については、return 情報のみ管理する
    if (mode_ == MonitorModeManager::Record) {
      bool ret = con_->connect(device, baudrate);
      handler_->fetch(ret);
      return ret;

    } else if (mode_ == MonitorModeManager::Play) {
      bool ret = false;
      if (handler_->fetch(ret) < 0) {
        disconnect();
        return false;
      }
      return ret;

    } else if (mode_ == MonitorModeManager::Simulation) {
      // 対応するポートに対して接続を行う
      // !!!
      return false;

    } else {
      return con_->connect(device, baudrate);
    }
  }


  void disconnect(void)
  {
    con_->disconnect();
  }


  const char* what(void)
  {
    // エラーメッセージも記録と再生の対象とする
    if ((mode_ == MonitorModeManager::Record) ||
        (mode_ == MonitorModeManager::Play)) {
      std::string error_message = std::string(con_->what());
      handler_->fetch(error_message);
      return error_message.c_str();

    } else if (mode_ == MonitorModeManager::Simulation) {
      // !!!
      return "Not implemented.";

    } else {
      return con_->what();
    }
  }


  bool setBaudrate(long baudrate)
  {
    // ボーレート値と戻り値を記録対象とする
    if ((mode_ == MonitorModeManager::Record) ||
        (mode_ == MonitorModeManager::Play)) {
      bool ret = con_->setBaudrate(baudrate);
      handler_->fetch(ret);
      handler_->fetch(baudrate);
      baudrate_ = baudrate;
      return ret;

    } else if (mode_ == MonitorModeManager::Simulation) {
      // !!!
      return false;

    } else {
      return con_->setBaudrate(baudrate);
    }
  }


  long baudrate(void)
  {
    if (mode_ == MonitorModeManager::Play) {
      return baudrate_;

    } else {
      return con_->baudrate();
    }
  }


  bool isConnected(void)
  {
    // !!! 戻り値のみを記録対象とする
    return false;

    if ((mode_ == MonitorModeManager::Record) ||
        (mode_ == MonitorModeManager::Play)) {
      bool ret = con_->isConnected();
      handler_->fetch(ret);
      return ret;

    } else if (mode_ == MonitorModeManager::Simulation) {
      // !!!
      return false;

    } else {
      return con_->isConnected();
    }
  }


  int send(const char* data, size_t count)
  {
    if (mode_ == MonitorModeManager::Record) {
      // 記録する必要はないが、デバッグ情報として使うために残しておく
      int send_size = con_->send(data, count);
      return handler_->fetch(const_cast<char*>(data), send_size, "send()");

    } else if (mode_ == MonitorModeManager::Play) {
      std::string line;
      int ret = handler_->fetch(line);
      return ret;

    } else if (mode_ == MonitorModeManager::Simulation) {
      // 対応するポートに対して接続を行う
      // !!!
      return false;

    } else {
      return con_->send(data, count);
    }
  }


  int recv(char* data, size_t count, int timeout)
  {
    if (mode_ == MonitorModeManager::Record) {
      int recv_size = con_->recv(data, count, timeout);
      return handler_->fetch(data, recv_size, "recv()");

    } else if (mode_ == MonitorModeManager::Play) {
      int recv_size = handler_->fetch(data, count);
      if (recv_size < 0) {
        disconnect();
      }
      return recv_size;

    } else if (mode_ == MonitorModeManager::Simulation) {
      // 対応するポートに対して接続を行う
      // !!!
      return false;

    } else {
      return con_->recv(data, count, timeout);
    }
  }


  size_t size(void)
  {
    // !!! 記録対象
    return con_->size();

    if ((mode_ == MonitorModeManager::Record) ||
        (mode_ == MonitorModeManager::Play)) {
      int data_size = con_->size();
      return handler_->fetch(data_size);

    } else if (mode_ == MonitorModeManager::Simulation) {
      // 対応するポートに対して接続を行う
      // !!!
      return 0;

    } else {
      return con_->size();
    }
  }
};


mConnection::mConnection(Connection* connection) : pimpl(new pImpl(connection))
{
}


mConnection::~mConnection(void)
{
}


const char* mConnection::what(void)
{
  return pimpl->what();
}


bool mConnection::connect(const char* device, long baudrate)
{
  return pimpl->connect(device, baudrate);
}


void mConnection::disconnect(void)
{
  pimpl->disconnect();
}


bool mConnection::setBaudrate(long baudrate)
{
  return pimpl->setBaudrate(baudrate);
}


long mConnection::baudrate(void)
{
  return pimpl->baudrate();
}


bool mConnection::isConnected(void)
{
  return pimpl->isConnected();
}


int mConnection::send(const char* data, size_t count)
{
  return pimpl->send(data, count);
}


int mConnection::recv(char* data, size_t count, int timeout)
{
  return pimpl->recv(data, count, timeout);
}


size_t mConnection::size(void)
{
  return pimpl->size();
}


void mConnection::flush(void)
{
  pimpl->con_->flush();
}


void mConnection::clear(void)
{
  pimpl->con_->clear();
}


void mConnection::ungetc(const char ch)
{
  pimpl->con_->ungetc(ch);
}
