/*!
  \file
  \brief シリアル通信 (Linux, Mac 実装)

  \author Satofumi KAMIMURA

  $Id: SerialCtrl_lin.cpp 301 2008-10-26 16:18:00Z satofumi $

  \todo rfds_ を、使用するサイズになるように調整する
  \todo エラーメッセージを調整する
*/

#include "SerialCtrl.h"
#include "RingBuffer.h"
#include "delay.h"
#include <fcntl.h>
#include <termios.h>
#include <errno.h>
#include <string>

using namespace qrk;


struct SerialCtrl::pImpl
{
  enum {
    InvalidFd = -1,
  };

  std::string error_message_;   //!< エラー状態
  int fd_;                      //!< ファイルディスクリプタ
  struct termios sio_;          //!< ターミナル制御
  fd_set rfds_;                 //!< タイムアウト制御
  RingBuffer<char> ring_buffer_; //!< 受信バッファ
  long baudrate_;                //!< 接続中のボーレート


  pImpl(void) : error_message_("no error."), fd_(InvalidFd), baudrate_(0)
  {
  }


  ~pImpl(void)
  {
    disconnect();
  }


  bool connect(const char* device, long baudrate)
  {
#ifndef MAC_OS
    enum { O_EXLOCK = 0x0 }; // Linux では使えないのでダミーを作成しておく
#endif
    fd_ = open(device, O_RDWR | O_EXLOCK | O_NONBLOCK | O_NOCTTY);
    if (fd_ < 0) {
      // 接続に失敗
      error_message_ = std::string(device) + ": " + strerror(errno);
      return false;
    }
    int flags = fcntl(fd_, F_GETFL, 0);
    fcntl(fd_, F_SETFL, flags & ~O_NONBLOCK);

    // シリアル通信の初期化
    tcgetattr(fd_, &sio_);
    sio_.c_iflag = 0;
    sio_.c_oflag = 0;
    sio_.c_cflag &= ~(CSIZE | PARENB | CSTOPB);
    sio_.c_cflag |= CS8 | CREAD | CLOCAL;
    sio_.c_lflag &= ~(ICANON | ECHO | ISIG | IEXTEN);

    sio_.c_cc[VMIN] = 0;
    sio_.c_cc[VTIME] = 0;

    // ボーレートの変更
    if (! setBaudrate(baudrate)) {
      return false;
    }
    return true;
  }


  void disconnect(void)
  {
    if (fd_ >= 0) {
      close(fd_);
      fd_ = InvalidFd;
    }
  }


  bool setBaudrate(long baudrate)
  {
    long baudrate_value = -1;
    enum { ErrorMessageSize = 256 };
    char error_message[ErrorMessageSize];

    switch (baudrate) {

    case 4800:
      baudrate_value = B4800;
      break;

    case 9600:
      baudrate_value = B9600;
      break;

    case 19200:
      baudrate_value = B19200;
      break;

    case 38400:
      baudrate_value = B38400;
      break;

    case 57600:
      baudrate_value = B57600;
      break;

    case 115200:
      baudrate_value = B115200;
      break;

    default:
      sprintf(error_message, "No handle baudrate value: %ld", baudrate);
      error_message_ = std::string(error_message);
      baudrate_ = 0;
      return false;
    }

    /* ボーレート変更 */
    cfsetospeed(&sio_, baudrate_value);
    cfsetispeed(&sio_, 0);
    tcsetattr(fd_, TCSANOW, &sio_);
    baudrate_ = baudrate;

    return true;
  }


  int recv(char* data, size_t count, int timeout) {

    if (fd_ < 0) {
      // 未接続ならば、戻る
      return 0;
    }

    size_t filled = 0;
    while (filled < count) {

      if ((timeout > 0) && (! waitReceive(0))) {
        // スレッド切替えを促す
        delay(1);
      }
      if (! waitReceive(timeout)) {
        break;
      }

      size_t required_n = count - filled;
      int read_n = read(fd_, &data[filled], required_n);
      if (read_n <= 0) {
        /* 読み出しエラー。現在までの受信内容で戻る */
        break;
      }
      filled += read_n;
    }
    return filled;
  }


  bool waitReceive(int timeout)
  {
    // タイムアウト設定
    FD_ZERO(&rfds_);
    FD_SET(fd_, &rfds_);

    struct timeval tv;
    tv.tv_sec = timeout / 1000;
    tv.tv_usec = (timeout % 1000) * 1000;

    if (select(fd_ + 1, &rfds_, NULL, NULL, &tv) <= 0) {
      /* タイムアウト発生 */
      return false;
    }
    return true;
  }


  void flush(void)
  {
    tcdrain(fd_);
    tcflush(fd_, TCIOFLUSH);
  }


  void updateRecvBuffer(size_t require_n, int timeout)
  {
    enum { BufferSize = 4096 };
    char buffer[BufferSize];

    if ((require_n == 0) && (timeout == 0)) {
      // 受信サイズのチェックとみなし、バッファ更新後に戻る
      int n = recv(buffer, BufferSize, 0);
      if (n > 0) {
        ring_buffer_.put(buffer, n);
      }
      return;
    }

    size_t readable_n = ring_buffer_.size();
    if (readable_n >= require_n) {
      // 必要なだけバッファにデータがあるならば、戻る
      return;
    }

    // 要求サイズ分、受信バッファのタイムアウト付き読み出しを行う
    while (1) {
      int require_n_actual = require_n - ring_buffer_.size();
      if (require_n_actual <= 0) {
        // 受信完了
        break;
      }
      // １回の読み出しは、バッファサイズで制限する
      require_n_actual =
        std::min(require_n_actual, static_cast<int>(BufferSize));

      int n = recv(buffer, require_n_actual, timeout);
      if (n > 0) {
        ring_buffer_.put(buffer, n);
      }
      if (n < require_n_actual) {
        // タイムアウト
        break;
      }
    }
  }
};


SerialCtrl::SerialCtrl(void) : pimpl(new pImpl)
{
}


SerialCtrl::~SerialCtrl(void)
{
}


const char* SerialCtrl::what(void)
{
  return pimpl->error_message_.c_str();
}


bool SerialCtrl::connect(const char* device, long baudrate)
{
  if (isConnected()) {
    // 接続されていれば、切断する
    disconnect();
  }
  return pimpl->connect(device, baudrate);
}


void SerialCtrl::disconnect(void)
{
  if (! isConnected()) {
    return;
  }
  return pimpl->disconnect();
}


bool SerialCtrl::setBaudrate(long baudrate)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return false;
  }
  return pimpl->setBaudrate(baudrate);
}


long SerialCtrl::baudrate(void)
{
  return pimpl->baudrate_;
}


bool SerialCtrl::isConnected(void)
{
  return (pimpl->fd_ >= 0) ? true : false;
}


int SerialCtrl::send(const char* data, size_t count)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return 0;
  }
  return write(pimpl->fd_, data, count);
}


int SerialCtrl::recv(char* data, size_t count, int timeout)
{

  if ((! isConnected()) && pimpl->ring_buffer_.empty()) {
    pimpl->error_message_ = "no connection.";
    return 0;
  }

  // 受信バッファの更新
  pimpl->updateRecvBuffer(count, timeout);

  // 受信バッファの内容を取得
  size_t readable_n = pimpl->ring_buffer_.size();
  size_t require_n = std::min(count, readable_n);

  return pimpl->ring_buffer_.get(data, require_n);
}


size_t SerialCtrl::size(void)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return 0;
  }

  // リングバッファのサイズを更新
  pimpl->updateRecvBuffer(0, 0);
  return pimpl->ring_buffer_.size();
}


void SerialCtrl::flush(void)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return;
  }
  pimpl->flush();
}


void SerialCtrl::clear(void)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return;
  }

  flush();
  pimpl->updateRecvBuffer(0, 0);
  pimpl->ring_buffer_.clear();
}


void SerialCtrl::ungetc(const char ch)
{
  if (! isConnected()) {
    pimpl->error_message_ = "no connection.";
    return;
  }
  pimpl->ring_buffer_.ungetc(ch);
}
