/*!
  \file
  \brief COM ポート一覧の取得

  \author Satofumi KAMIMURA

  $Id: FindComPorts.cpp 202 2008-09-03 01:13:35Z satofumi $
*/

#include "FindComPorts.h"
#include "IsUsbCom.h"
#include "DetectOS.h"
#ifdef WINDOWS_OS
#include <windows.h>
#else
#include "FindFiles.h"
using namespace boost::xpressive;
#endif

using namespace qrk;


struct FindComPorts::pImpl
{
  std::vector<std::string> base_names_;
  IsUsbCom* is_usb_;


  pImpl(IsUsbCom* is_usb) : is_usb_(is_usb)
  {
  // Windows はレジストリ情報だけから COM 一覧の探索を行う
  // よって、以降の設定は Linux, MacOS のみで使われる
#ifdef LINUX_OS
    base_names_.push_back("/dev/ttyUSB");
    base_names_.push_back("/dev/usb/ttyUSB");
#else
    // !!! 不明
    // !!! 何か、具体的に取得する方法があるのかも
    // !!!base_names_.push_back("unknown");
#endif
  }


  void addBaseName(const char* base_name)
  {
    base_names_.insert(base_names_.begin(), base_name);
  }


  void addBaseNames(void)
  {
    if (! is_usb_) {
      return;
    }

    std::vector<std::string> additional_ports = is_usb_->setBaseNames();
    // !!! マージするように変更する
    for (std::vector<std::string>::iterator it = additional_ports.begin();
         it != additional_ports.end(); ++it) {
      addBaseName(it->c_str());
    }
  }


  void orderByDriver(std::vector<std::string>& ports)
  {
    if (! is_usb_) {
      return;
    }

    // COM 探索結果に対して、IsUsbCom を適用し、該当する場合は先頭に配置する
    size_t last_index = 0;
    for (std::vector<std::string>::iterator it = ports.begin();
         it != ports.end(); ++it) {
      if (is_usb_->isUsbCom(it->c_str())) {
        std::swap(ports[last_index], *it);
        ++last_index;
      }
    }
  }


  void addFoundPorts(std::vector<std::string>& found_ports,
		     const std::string& port)
  {
    for (std::vector<std::string>::iterator it = found_ports.begin();
	 it != found_ports.end(); ++it) {

	// !!! データ構造を map にすべきかも
	// !!! 検索のアルゴリズムを用いるべき

      if (! port.compare(*it)) {
	return;
      }
    }
    found_ports.push_back(port);
  }
};


FindComPorts::FindComPorts(IsUsbCom* is_usb) : pimpl(new pImpl(is_usb))
{
}


FindComPorts::~FindComPorts(void)
{
}


void FindComPorts::clear(void)
{
  pimpl->base_names_.clear();
}


void FindComPorts::addBaseName(const char* base_name)
{
  pimpl->addBaseName(base_name);
}


std::vector<std::string> FindComPorts::baseNames(void)
{
  return pimpl->base_names_;
}


#ifdef WINDOWS_OS
// Windows の場合
std::vector<std::string> FindComPorts::find(void)
{
  std::vector<std::string> found_ports;
  HKEY hkey;
  if (RegOpenKeyExA(HKEY_LOCAL_MACHINE, "HARDWARE\\DEVICEMAP\\SERIALCOMM",
                    0, KEY_READ, &hkey) != ERROR_SUCCESS) {
    return found_ports;
  }
  enum { MaxLength = 255 };
  CHAR device[MaxLength];
  char name[MaxLength];

  DWORD ret = ERROR_SUCCESS;
  for (int i = 0; ret == ERROR_SUCCESS; ++i) {
    DWORD dl = MaxLength, nl = MaxLength;
    ret = RegEnumValueA(hkey, i, device, &dl, NULL, NULL, (BYTE*)name, &nl);
    if (ret == ERROR_SUCCESS) {
      pimpl->addFoundPorts(found_ports, std::string(name));
    }
  }
  RegCloseKey(hkey);

  pimpl->orderByDriver(found_ports);
  return found_ports;
}

#else
// Linux, Mac の場合
std::vector<std::string> FindComPorts::find(void)
{
  std::vector<std::string> found_ports;

  pimpl->addBaseNames();

  // 登録ベース名毎に、ファイル名がないかの探索を行う
  for (std::vector<std::string>::iterator it = pimpl->base_names_.begin();
       it != pimpl->base_names_.end(); ++it) {

    // ディレクトリ名とファイル名との切り分け
    const char* path = it->c_str();
    const char* last_slash = strrchr(path, '/') + 1;

    std::string dir_name = it->substr(0, last_slash - path);
    std::string file_name = it->substr(last_slash - path, std::string::npos);

    std::vector<std::string> ports;
    findFiles(ports, dir_name.c_str(),
	      sregex::compile("[a-zA-Z]+[0-9]+"));

    size_t n = it->size();
    for (std::vector<std::string>::iterator fit = ports.begin();
	 fit != ports.end(); ++fit) {
      if (! fit->compare(0, n, *it)) {
	// マッチしたら、登録を行う
	pimpl->addFoundPorts(found_ports, *fit);
      }
    }
  }
  pimpl->orderByDriver(found_ports);

  return found_ports;
}
#endif
