#ifndef RING_BUFFER_H
#define RING_BUFFER_H

/*!
  \file
  \brief リングバッファ

  \author Satofumi KAMIMURA

  $Id: RingBuffer.h 276 2008-10-14 15:19:23Z satofumi $
*/

#include <vector>


namespace qrk
{
  /*!
    \brief リングバッファ
  */
  template<class T> class RingBuffer
  {
    RingBuffer(const RingBuffer& rhs);
    RingBuffer& operator = (const RingBuffer& rhs);

    std::vector<T> ring_buffer_;
    size_t buffer_size_;
    size_t first_;
    size_t last_;


    size_t free_size(void)
    {
      return buffer_size_ -1 - size();
    }


    void extendBuffer(long need_size)
    {
      size_t extend_size = (buffer_size_ == 0) ? 1 : buffer_size_;
      while (need_size > 0) {
        need_size -= extend_size;
        extend_size *= 2;
      }
      ring_buffer_.resize(extend_size);

      // 拡張サイズに合わせてデータを移動
      if (first_ > last_) {
        // 0 から last までのデータを、buffer_size 以降に移動させる
        memmove(&ring_buffer_[buffer_size_],
                &ring_buffer_[0], last_ * sizeof(T));
        last_ += buffer_size_;
      }
      buffer_size_ = extend_size;
    }

  public:
    /*!
      \brief 初期バッファサイズ指定のコンストラクタ

      ここで指定したサイズ以上のバッファが確保される

      \param[in] defaultSize バッファサイズ
    */
    explicit RingBuffer(size_t defaultSize = 0)
      : buffer_size_(1), first_(0), last_(0)
    {
      while (buffer_size_ < defaultSize) {
        buffer_size_ *= 2;
      }
      ring_buffer_.resize(buffer_size_);
    }


    /*!
      \brief バッファサイズの取得
    */
    size_t size(void)
    {
      return (last_ >= first_) ?
        (last_ - first_) : (buffer_size_ - (first_ - last_));
    }


    /*!
      \brief バッファが空か

      \retval true データなし
      \retval false データあり
    */
    bool empty(void)
    {
      return (first_ == last_) ? true : false;
    }


    /*!
      \brief データの格納

      \param[in] data データ
      \param[in] size データ個数
    */
    void put(const T* data, size_t size)
    {
      long need_size = static_cast<long>(size - free_size());
      if (need_size > 0) {
        // バッファに入りきらない場合、拡張
        extendBuffer(need_size);
      }
      // データ配置
      if (first_ <= last_) {
        // last から buffer_size 終端までに配置
        size_t to_end = buffer_size_ - last_;
        size_t move_size = (to_end > size) ? size : to_end;
        memmove(&ring_buffer_[last_], data, move_size * sizeof(T));
        last_ += move_size;
        last_ &= (buffer_size_ -1);

        long left_size = static_cast<long>(size - move_size);
        if (left_size > 0) {
          // 0 から first の前までを配置
          memmove(&ring_buffer_[0], &data[move_size], left_size * sizeof(T));
          last_ = left_size;
        }
      } else {
        // last から first の前まで配置
        memmove(&ring_buffer_[last_], data, size * sizeof(T));
        last_ += size;
      }
    }


    /*!
      \brief データの取り出し

      \param[out] data データ取り出し用バッファ
      \param[in] size 取り出すデータの最大個数

      \return 取り出したデータ個数
    */
    size_t get(T* data, size_t size)
    {
      // データ取得
      size_t return_size = (size <= this->size()) ? size : this->size();

      if (first_ <= last_) {
        memmove(data, &ring_buffer_[first_], return_size * sizeof(T));
        first_ += return_size;
      } else {
        // first から buffer_size 終端までを配置
        size_t to_end = buffer_size_ - first_;
        size_t move_size = (to_end > return_size) ? return_size : to_end;
        memmove(data, &ring_buffer_[first_], move_size * sizeof(T));
        first_ += move_size;
        first_ &= (buffer_size_ -1);

        long left_size = static_cast<long>(return_size - move_size);
        if (left_size > 0) {
          // 0 から last の前までを配置
          memmove(&data[move_size], &ring_buffer_[0], left_size * sizeof(T));
          first_ = left_size;
        }
      }
      return return_size;
    }


    /*!
      \brief データの書き戻し

      \param[in] ch 書き戻すデータ
    */
    void ungetc(const char ch)
    {
      if (free_size() <= 0) {
        // 領域が不足していたら、バッファサイズを拡張
        extendBuffer(1);
      }

      // 文字の追加
      first_ = (first_ == 0) ? (buffer_size_ - 1) : (first_ - 1);
      ring_buffer_[first_] = ch;
    }


    /*!
      \brief 格納データのクリア
    */
    void clear(void)
    {
      first_ = 0;
      last_ = 0;
    }
  };
};

#endif /* !RING_BUFFER_H */
