/*!
  \file
  \brief モニタ対応の URG 制御

  \author Satofumi KAMIMURA

  $Id: mUrgCtrl.cpp 290 2008-10-22 15:35:54Z satofumi $
*/

#include "mUrgCtrl.h"
#include "RangeSensorParameter.h"
#include "UrgServer.h"
#include "SerialCtrl.h"
#include "TcpipCtrl.h"
#include "MonitorModeManager.h"
#include "MonitorEventScheduler.h"
#include "DeviceIpManager.h"
#include "mConnection.h"


using namespace qrk;


struct mUrgCtrl::pImpl
{
  MonitorModeManager::MonitorMode mode_;
  UrgCtrl urg_;
  Connection* monitor_con_;
  Connection* device_con_;


  pImpl(int argc, char *argv[])
    : monitor_con_(NULL), device_con_(NULL)
  {
    MonitorModeManager::singleton()->setMode(argc, argv);
    mode_ = MonitorModeManager::singleton()->mode();

    if (mode_ != MonitorModeManager::Simulation) {
      device_con_ = new SerialCtrl;
    } else {
      device_con_ = new TcpipCtrl;
    }
    monitor_con_ = new mConnection(device_con_);
    urg_.setConnection(monitor_con_);
  }


  ~pImpl(void)
  {
    urg_.stop();

    if (monitor_con_) {
      delete monitor_con_;
    }
    if (device_con_) {
      delete device_con_;
    }
  }
};


mUrgCtrl::mUrgCtrl(int argc, char *argv[]) : pimpl(new pImpl(argc, argv))
{
}


mUrgCtrl::~mUrgCtrl(void)
{
}


const char* mUrgCtrl::what(void) const
{
  return pimpl->urg_.what();
}


bool mUrgCtrl::connect(const char* device, long baudrate)
{
  MonitorModeManager::MonitorMode Simulation = MonitorModeManager::Simulation;
  if (MonitorModeManager::singleton()->mode() == Simulation) {
    // device 名で IP ポート取得し、サーバを起動する
    long port = DeviceIpManager::singleton()->createIpPort(device);
    if (port > 0) {
      // IP ポートが取得済みでない場合のみ、サーバを起動する
      // !!! URG の型、シリアル ID の情報を、引数で反映させるべき
      UrgServer* urg_server = new UrgServer;
      if (! urg_server->activate(port)) {
        // サーバが起動できなければ、接続を中断する
        return false;
      }
      MonitorEventScheduler::singleton()->registerDeviceServer(urg_server);
    }
  }

  return pimpl->urg_.connect(device, baudrate);
}


void mUrgCtrl::setConnection(Connection* con)
{
  pimpl->urg_.setConnection(con);
}


Connection* mUrgCtrl::connection(void)
{
  return pimpl->urg_.connection();
}


void mUrgCtrl::disconnect(void)
{
  pimpl->urg_.disconnect();
}


bool mUrgCtrl::isConnected(void)
{
  return pimpl->urg_.isConnected();
}


long mUrgCtrl::minDistance(void) const
{
  return pimpl->urg_.minDistance();
}


long mUrgCtrl::maxDistance(void) const
{
  return pimpl->urg_.maxDistance();
}


int mUrgCtrl::maxScanLines(void) const
{
  return pimpl->urg_.maxScanLines();
}


int mUrgCtrl::scanMsec(void) const
{
  return pimpl->urg_.scanMsec();
}


void mUrgCtrl::setCaptureMode(RangeCaptureMode mode)
{
  pimpl->urg_.setCaptureMode(mode);
}


RangeCaptureMode mUrgCtrl::captureMode(void)
{
  return pimpl->urg_.captureMode();
}


void mUrgCtrl::setCaptureRange(int begin_index, int end_index)
{
  pimpl->urg_.setCaptureRange(begin_index, end_index);
}


void mUrgCtrl::setCaptureFrameInterval(size_t interval)
{
  pimpl->urg_.setCaptureFrameInterval(interval);
}


void mUrgCtrl::setCaptureTimes(size_t times)
{
  pimpl->urg_.setCaptureTimes(times);
}



void mUrgCtrl::setCaptureSkipLines(size_t skip_lines)
{
  pimpl->urg_.setCaptureSkipLines(skip_lines);
}


int mUrgCtrl::capture(std::vector<long>& data, int* timestamp)
{
  return pimpl->urg_.capture(data, timestamp);
}


int mUrgCtrl::captureIntensity(std::vector<long>& data, int* timestamp)
{
  return pimpl->urg_.captureIntensity(data, timestamp);
}


void mUrgCtrl::stop(void)
{
  pimpl->urg_.stop();
}


bool mUrgCtrl::setTimestamp(int ticks)
{
  return pimpl->urg_.setTimestamp(ticks);
}


bool mUrgCtrl::setLaserOutput(bool on)
{
  return pimpl->urg_.setLaserOutput(on);
}


double mUrgCtrl::index2rad(const int index) const
{
  return pimpl->urg_.index2rad(index);
}


int mUrgCtrl::rad2index(const double radian) const
{
  return pimpl->urg_.rad2index(radian);
}


void mUrgCtrl::setParameter(const RangeSensorParameter& parameter)
{
  pimpl->urg_.setParameter(parameter);
}


RangeSensorParameter mUrgCtrl::parameter(void)
{
  return pimpl->urg_.parameter();
}


bool mUrgCtrl::versionLines(std::vector<std::string>& lines)
{
  return pimpl->urg_.versionLines(lines);
}
