/*!
  \file
  \brief モニタイベント管理

  \author Satofumi KAMIMURA

  $Id: MonitorEventScheduler.cpp 296 2008-10-25 12:40:50Z satofumi $
*/

#include "MonitorEventScheduler.h"
#include "Thread.h"
#include "Lock.h"
#include "getTicks.h"
#include "delay.h"
#include <list>

using namespace qrk;


struct MonitorEventScheduler::pImpl
{
  enum {
    ThreadMaxDelay = 200,       // [msec]
  };
  typedef struct
  {
    int ticks;
    ConditionVariable* condition;
  } event_t;
  // !!! 比較演算子を定義

  Lock mutex_;
  Thread thread_;
  std::list<event_t> event_list_;

  Lock wait_mutex_;
  ConditionVariable wait_condition_;


  pImpl(void) : thread_(schedule_thread, this)
  {
    thread_.run();
  }


  ~pImpl(void)
  {
    terminate();
  }


  static void eventDelay(int delay_msec)
  {
    if (delay_msec > ThreadMaxDelay) {
      // Thread の実装によっては、スレッドは定期的に終了することが望ましい
      // そのための待機状態の最大値
      delay_msec = ThreadMaxDelay;
    }
    delay((delay_msec > 0) ? delay_msec : 1);
  }


  static int schedule_thread(void* args)
  {
    pImpl* obj = static_cast<pImpl*>(args);
    obj->mutex_.lock();

    // 現在の ticks を取得し、より小さい ticks のイベントを順番に起こす
    // !!! 一時停止を考慮すべき
    int current_ticks = getTicks();
    std::list<event_t>::iterator end_it = obj->event_list_.end();
    for (std::list<event_t>::iterator it = obj->event_list_.begin();
         it != end_it;) {
      if (current_ticks < it->ticks) {
        break;
      }

      if (! it->condition->isWaiting()) {
        // 待ち状態でなければ、次の周期で処理する
        // 以下の delay() 以降に、呼び出し元で待ち状態に遷移するはず
        int delay_msec = it->ticks - current_ticks;
        obj->mutex_.unlock();
        eventDelay(delay_msec);
        return 0;
      }

      ConditionVariable* condition = it->condition;
      it = obj->event_list_.erase(it);
      condition->wakeup();
    }

    if (obj->event_list_.empty()) {
      // 待ち状態に入る
      obj->wait_mutex_.lock();
      obj->mutex_.unlock();
      obj->wait_condition_.wait(&obj->wait_mutex_);
      obj->wait_mutex_.unlock();

    } else {
      // 次のイベントまで待機させる
      int delay_msec = obj->event_list_.front().ticks - current_ticks;
      obj->mutex_.unlock();
      eventDelay(delay_msec);
    }

    return 0;
  }


  void terminate(void)
  {
    thread_.stop();
    if (wait_condition_.isWaiting()) {
      wait_condition_.wakeup();
    }
    thread_.wait();
  }
};


MonitorEventScheduler::MonitorEventScheduler(void) : pimpl(new pImpl)
{
}


MonitorEventScheduler::~MonitorEventScheduler(void)
{
}


MonitorEventScheduler* MonitorEventScheduler::singleton(void)
{
  static MonitorEventScheduler singleton_object;
  return &singleton_object;
}


void MonitorEventScheduler::registerWakeupTicks(ConditionVariable* condition,
                                                int ticks)
{
  pimpl->mutex_.lock();

  // イベントリストに追加
  pImpl::event_t event;
  event.ticks = ticks;
  event.condition = condition;
  pimpl->event_list_.push_back(event);

  // イベントが空の時の追加で、スケジューラのスレッドを起こす
  if (pimpl->wait_condition_.isWaiting()) {
    pimpl->mutex_.unlock();
    pimpl->wait_condition_.wakeup();
  }
  pimpl->mutex_.unlock();
}


void MonitorEventScheduler::registerDeviceServer(DeviceServer* device)
{
  // !!!

  // !!! 登録後、接続を行わせるために、スレッドを動作させる
  // !!!
}
