#ifndef SCREEN_CTRL_H
#define SCREEN_CTRL_H

/*!
  \file
  \brief 画面管理

  \author Satofumi KAMIMURA

  $Id: ScreenCtrl.h 166 2008-08-24 00:37:24Z satofumi $

  \attention show() は単一スレッド以外から呼び出されてはならない (SDL-1.2 の仕様)
*/

#include <SDL.h>


namespace qrk
{
  /*!
    \brief 画面制御
  */
  class ScreenCtrl
  {
    ScreenCtrl(const ScreenCtrl& rhs);
    ScreenCtrl& operator = (const ScreenCtrl& rhs);

    struct pImpl;
    pImpl* pimpl;

  public:
    ScreenCtrl(void);
    ~ScreenCtrl(void);


    /*!
      \brief エラー状態を返す

      \return エラー状態を示す文字列
    */
    const char* what(void);


    /*!
      \brief 画面の表示

      \param flags SDL_SetVideoMode に渡すモード引数

      \return SDL の返すスクリーン資源
    */
    SDL_Surface* show(Uint32 flags = 0);


    /*!
      \brief 画面の非表示
    */
    void hide(void);


    /*!
      \brief 画面サイズの変更

      \param[in] width 幅
      \param[in] height 高さ

      \attention 変更は、次の show() から有効になる
    */
    void setWindowSize(size_t width, size_t height);


    /*!
      \brief 現在の画面幅

      \return 画面幅
    */
    size_t width(void);


    /*!
      \brief 現在の画面高さ

      \return 画面高さ
    */
    size_t height(void);


    /*!
      \brief 画面解像度の設定

      \param[in] bpp 画面の解像度 [bit]
    */
    void setWindowBpp(size_t bpp);


    /*!
      \brief 画面モード

      \param[in] on フルスクリーン指定のとき true / 通常スクリーンのとき false

      \attention 変更は、次の show() から有効になる
    */
    void setFullscreen(bool on = true);


    /*!
      \brief フルスクリーンかを返す

      \retval true フルスクリーン
      \retval false 通常スクリーン
    */
    bool isFullscreen(void);


    /*!
      \brief タイトル文字列の設定

      \param[in] caption タイトル文字列
      \param[in] icon_caption アイコン時のタイトル文字列

      \attention 最初に show() が呼ばれる前に呼び出されなければならない
      \attention 詳しくは、SDL_WM_SetCaption() を参照のこと
    */
    void setCaption(const char* caption, const char* icon_caption);


    /*!
      \brief タイトルアイコンの設定

      \param[in] icon タイトルアイコン
      \param[in] mask マスク領域

      \attention 最初に show() が呼ばれる前に呼び出されなければならない
      \attention 詳しくは、SDL_WM_SetIcon() を参照のこと
    */
    void setIcon(const SDL_Surface* icon, const Uint8* mask = NULL);
  };
};

#endif /* !SCREEN_CTRL_H */
