/*!
  \file
  \brief 画面管理

  \author Satofumi KAMIMURA

  $Id: ScreenCtrl.cpp 166 2008-08-24 00:37:24Z satofumi $
*/

#include <string>
#include "ScreenCtrl.h"
#include "SdlVideoInit.h"

using namespace qrk;


struct ScreenCtrl::pImpl
{
  SdlVideoInit sdl_video_;
  std::string error_message_;
  SDL_Surface* scr_;
  Uint32 pre_flags_;
  int next_width_;
  int next_height_;
  int next_bpp_;
  bool is_fullscreen_;
  bool toggle_fullscreen_;

  std::string caption_;
  std::string icon_caption_;
  const SDL_Surface* icon_surface_;
  const Uint8* icon_mask_;


  pImpl(void)
    : error_message_(SDL_GetError()), scr_(NULL), pre_flags_(0),
      next_width_(640), next_height_(480), next_bpp_(0),
      is_fullscreen_(false), toggle_fullscreen_(false),
      icon_surface_(NULL), icon_mask_(NULL) {
  }


  static pImpl* getObject(void)
  {
    static pImpl singleton;
    return &singleton;
  }


  void show(Uint32 flags)
  {
    if ((! scr_) && icon_surface_) {
      // アイコンの初期化は、SDL_SetVideoMode() を呼び出す前のみ有効
      SDL_WM_SetIcon(const_cast<SDL_Surface*>(icon_surface_),
                     const_cast<Uint8*>(icon_mask_));
    }

    sdl_video_.init();
    scr_ = SDL_SetVideoMode(next_width_, next_height_, next_bpp_, flags);
    if (scr_) {
      pre_flags_ = flags;

      // キャプションは、画面の生成後に設定
      SDL_WM_SetCaption(caption_.c_str(), icon_caption_.c_str());

      if (flags & SDL_FULLSCREEN) {
        // 実際にフルスクリーンかどうか、の厳密な保証はできない
        is_fullscreen_ = true;
      }
    }
    toggle_fullscreen_ = false;
    next_bpp_ = 0;
    error_message_ = SDL_GetError();
  }


  void hide(void)
  {
    sdl_video_.quit();
  }
};


ScreenCtrl::ScreenCtrl(void) : pimpl(pImpl::getObject()) {
}


ScreenCtrl::~ScreenCtrl(void)
{
}


const char* ScreenCtrl::what(void)
{
  return pimpl->error_message_.c_str();
}


SDL_Surface* ScreenCtrl::show(Uint32 flags)
{
  // 画面設定が変わっていたら、show() のときに画面を作り直す
  if ((! pimpl->scr_) ||
      pimpl->toggle_fullscreen_ ||
      (pimpl->next_bpp_ != 0) ||
      (pimpl->next_width_ != pimpl->scr_->w) ||
      (pimpl->next_height_ != pimpl->scr_->h)) {
    hide();
    pimpl->show(flags);
  }
  return pimpl->scr_;
}


void ScreenCtrl::hide(void)
{
  pimpl->hide();
}


void ScreenCtrl::setWindowSize(size_t width, size_t height)
{
  pimpl->next_width_ = width;
  pimpl->next_height_ = height;
}


size_t ScreenCtrl::width(void)
{
  if (! pimpl->scr_) {
    // ウィンドウ非表示
    return 0;
  }
  return pimpl->scr_->w;
}


size_t ScreenCtrl::height(void)
{
  if (! pimpl->scr_) {
    // ウィンドウ非表示
    return 0;
  }
  return pimpl->scr_->h;
}


void ScreenCtrl::setWindowBpp(size_t bpp)
{
  pimpl->next_bpp_ = bpp;
}


void ScreenCtrl::setFullscreen(bool on)
{
  if (on != pimpl->is_fullscreen_) {
    Uint32 next_flags = pimpl->pre_flags_;
    if (on) {
      next_flags |= SDL_FULLSCREEN;
    } else {
      next_flags &= ~SDL_FULLSCREEN;
    }

    // 画面の再生成を強制する
    pimpl->toggle_fullscreen_ = true;
    show(pimpl->pre_flags_);
  }
}


bool ScreenCtrl::isFullscreen(void)
{
  return pimpl->is_fullscreen_;
}


void ScreenCtrl::setCaption(const char* caption, const char* icon_caption)
{
  pimpl->caption_ = caption;
  pimpl->icon_caption_ = icon_caption;
}


void ScreenCtrl::setIcon(const SDL_Surface* icon, const Uint8* mask)
{
  pimpl->icon_surface_ = icon;
  pimpl->icon_mask_ = mask;
}
