#ifndef LINE_UTILS_H
#define LINE_UTILS_H

/*!
  \file
  \brief 直線関連の関数

  「ゲームプログラミングのためのリアルタイム衝突判定」(ELSEVIER) より

  \author Satofumi KAMIMURA

  $Id: lineUtils.h 222 2008-09-17 12:52:15Z satofumi $
*/

#include "GridTypes.h"


namespace
{
  template <class T>
  T signed2DTriangleArea(const qrk::Grid<T>& a,
                         const qrk::Grid<T>& b,
                         const qrk::Grid<T>& c)
  {
    return ((a.x - c.x) * (b.y - c.y)) - ((a.y - c.y) * (b.x - c.x));
  }
};


namespace qrk
{
  class Angle;

  /*!
    \brief 線分の交差判定

    \param[in] a 線分１の端点
    \param[in] b 線分１の端点
    \param[in] c 線分２の端点
    \param[in] d 線分２の端点
    \param[in] p 線分の交差点
  */
  template <class T>
  bool isCrossLines(const Grid<T>& a, const Grid<T>& b,
                    const Grid<T>& c, const Grid<T>& d, Grid<T>& p)
  {
    T a1 = signed2DTriangleArea(a, b, d);
    T a2 = signed2DTriangleArea(a, b, c);

    if ((a1 * a2) < 0.0) {
      T a3 = signed2DTriangleArea(c, d, a);
      T a4 = a3 + a2 - a1;

      if ((a3 * a4) < 0.0) {
        T t = a3 / (a3 - a4);
        p = (t * (b - a)) + a;
        return true;
      }
    }
    return false;
  }
};

#endif /* !LINE_UTILS_H */
