/*!
  \file
  \brief Coordinate のテスト

  \author Satofumi KAMIMURA

  $Id: TestCoordinate.cpp 186 2008-08-30 01:45:57Z satofumi $
*/

#include "TestCoordinate.h"
#include "Coordinate.h"
#include "GridTypes.h"
#include "MathUtils.h"

using namespace qrk;


CPPUNIT_TEST_SUITE_REGISTRATION(TestCoordinate);
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(TestCoordinate, "TestCoordinate");


namespace {
  class DummyRunCtrl : public Coordinate {
    qrk::Position<int> position_;

  public:

    void setPosition(const qrk::Position<int>& position)
    {
      position_ = position;
    }


    qrk::Position<int> position(Coordinate* coordinate = NULL)
    {
      return pointPosition(coordinate, position_);
    }
  };
};


void TestCoordinate::initializeTest(void)
{
  // 初期座標系が (0, 0, deg(0)) になっているかテスト
  Coordinate crd;

  Position<int> position = crd.position();
  CPPUNIT_ASSERT_EQUAL(0, position.x);
  CPPUNIT_ASSERT_EQUAL(0, position.y);
  CPPUNIT_ASSERT_EQUAL(0, position.to_deg());

  Position<int> origin_position = crd.originPosition();
  CPPUNIT_ASSERT_EQUAL(0, origin_position.x);
  CPPUNIT_ASSERT_EQUAL(0, origin_position.y);
  CPPUNIT_ASSERT_EQUAL(0, origin_position.to_deg());
}


void TestCoordinate::parentTest(void)
{
  // 親座標系が取得できるかテスト
  Coordinate parent;
  Coordinate child_1(&parent, Position<int>(0, 0, deg(0)));
  Coordinate child_2;
  parent.addChild(&child_2);

  CPPUNIT_ASSERT_EQUAL(static_cast<Coordinate*>(NULL), parent.parent());
  CPPUNIT_ASSERT_EQUAL(&parent, child_1.parent());
  CPPUNIT_ASSERT_EQUAL(&parent, child_2.parent());

  // 子座標系が取得できるかのテスト
  std::vector<Coordinate*> children = parent.children();
  CPPUNIT_ASSERT_EQUAL(2, static_cast<int>(children.size()));
  CPPUNIT_ASSERT(((children[0] == &child_1) && (children[1] == &child_2)) ||
                 ((children[1] == &child_1) && (children[0] == &child_2)));
}


void TestCoordinate::relativeTest(void)
{
  // 相対位置が適切に取得できるかのテスト
  // root と parent における位置は、同じ値になるはず
  Coordinate parent;
  Coordinate child(&parent, Position<int>(100, 100, deg(-90)));
  Coordinate grandchild;
  child.addChild(&grandchild, Position<int>(100, 100, deg(-90)));


  // root における子の位置、parent における子の位置
#if 0
  fprintf(stderr, "\n");
  fprintf(stderr, "parent:      %p\n", &parent);
  fprintf(stderr, "child:       %p\n", &child);
  fprintf(stderr, "grandchild:  %p\n", &grandchild);
#endif

  Position<int> parent_on_child_1 = child.originPosition();
  CPPUNIT_ASSERT_EQUAL(100, parent_on_child_1.x);
  CPPUNIT_ASSERT_EQUAL(100, parent_on_child_1.y);
  CPPUNIT_ASSERT_EQUAL(-90, parent_on_child_1.to_deg());

  Position<int> parent_on_child_2 = child.originPosition(&parent);
  CPPUNIT_ASSERT_EQUAL(100, parent_on_child_2.x);
  CPPUNIT_ASSERT_EQUAL(100, parent_on_child_2.y);
  CPPUNIT_ASSERT_EQUAL(-90, parent_on_child_2.to_deg());

  // root における孫の位置, parent における孫の位置, 子における孫の位置
  Position<int> parent_on_grandchild_1 = grandchild.originPosition();
  CPPUNIT_ASSERT_EQUAL(200, parent_on_grandchild_1.x);
  CPPUNIT_ASSERT_EQUAL(0, parent_on_grandchild_1.y);
  CPPUNIT_ASSERT_EQUAL(-180, parent_on_grandchild_1.to_deg());

  Position<int> parent_on_grandchild_2 = grandchild.originPosition(&parent);
  CPPUNIT_ASSERT_EQUAL(200, parent_on_grandchild_2.x);
  CPPUNIT_ASSERT_EQUAL(0, parent_on_grandchild_2.y);
  CPPUNIT_ASSERT_EQUAL(-180, parent_on_grandchild_2.to_deg());

  Position<int> child_on_grandchild_3 = grandchild.originPosition(&child);
  CPPUNIT_ASSERT_EQUAL(100, child_on_grandchild_3.x);
  CPPUNIT_ASSERT_EQUAL(100, child_on_grandchild_3.y);
  CPPUNIT_ASSERT_EQUAL(-90, child_on_grandchild_3.to_deg());

  // 子における親の位置
  Position<int> parent_on_child = parent.originPosition(&child);
  CPPUNIT_ASSERT_EQUAL(100, parent_on_child.x);
  CPPUNIT_ASSERT_EQUAL(-100, parent_on_child.y);
  CPPUNIT_ASSERT_EQUAL(90, parent_on_child.to_deg());
}


void TestCoordinate::pointPositionTest(void)
{
  Coordinate coordinate;
  Position<int> position =
    coordinate.pointPosition(NULL, Position<int>(0, 0, deg(0)));
  CPPUNIT_ASSERT_EQUAL(0, position.x);
  CPPUNIT_ASSERT_EQUAL(0, position.y);
  CPPUNIT_ASSERT_EQUAL(0, position.to_deg());

  position = coordinate.pointPosition(NULL, Position<int>(100, 20, deg(30)));
  CPPUNIT_ASSERT_EQUAL(100, position.x);
  CPPUNIT_ASSERT_EQUAL(20, position.y);
  CPPUNIT_ASSERT_DOUBLES_EQUAL(30, position.to_deg(), 1.0);
}


void TestCoordinate::toModuleTest(void)
{
  // モジュールに対する位置設定が適切に動作するかテスト
  DummyRunCtrl run;
  Coordinate module;
#if 0
  fprintf(stderr, "\n");
  fprintf(stderr, "run     %p\n", &run);
  fprintf(stderr, "module: %p\n", &module);
#endif

  run.addChildToModule(&module);
  Position<int> module_position;

  Position<int> run_position = run.position();
  CPPUNIT_ASSERT_EQUAL(0, run_position.x);
  CPPUNIT_ASSERT_EQUAL(0, run_position.y);
  CPPUNIT_ASSERT_EQUAL(0, run_position.to_deg());

  module_position = module.position();
  CPPUNIT_ASSERT_EQUAL(0, module_position.x);
  CPPUNIT_ASSERT_EQUAL(0, module_position.y);
  CPPUNIT_ASSERT_EQUAL(0, module_position.to_deg());


  run.setPosition(Position<int>(100, 20, deg(0)));
  run_position = run.position();
  CPPUNIT_ASSERT_EQUAL(100, run_position.x);
  CPPUNIT_ASSERT_EQUAL(20, run_position.y);
  CPPUNIT_ASSERT_EQUAL(0, run_position.to_deg());

  run.setPosition(Position<int>(100, 20, deg(30)));
  module_position = module.position();
  CPPUNIT_ASSERT_EQUAL(100, module_position.x);
  CPPUNIT_ASSERT_EQUAL(20, module_position.y);
  CPPUNIT_ASSERT_DOUBLES_EQUAL(30, module_position.to_deg(), 1);


  // 同じオブジェクトを登録したときに、子ノード数が変わらないことのテスト
  CPPUNIT_ASSERT_EQUAL(1, static_cast<int>(run.children().size()));
  run.addChildToModule(&module, Position<int>(50, 10, deg(90)));
  CPPUNIT_ASSERT_EQUAL(1, static_cast<int>(run.children().size()));

  run.setPosition(Position<int>(0, 0, deg(0)));
  module_position = module.position();
  CPPUNIT_ASSERT_EQUAL(50, module_position.x);
  CPPUNIT_ASSERT_EQUAL(10, module_position.y);
  CPPUNIT_ASSERT_EQUAL(90, module_position.to_deg());

  run.setPosition(Position<int>(100, 20, deg(30)));
  module_position = module.position();
  double radian = 30.0 * M_PI / 180.0;
  double expected_x = ((50 * cos(radian)) - (10 * sin(radian))) + 100;
  double expected_y = ((50 * sin(radian)) + (10 * cos(radian))) + 20;
  CPPUNIT_ASSERT_DOUBLES_EQUAL(expected_x, module_position.x, 1);
  CPPUNIT_ASSERT_DOUBLES_EQUAL(expected_y, module_position.y, 1);
  CPPUNIT_ASSERT_DOUBLES_EQUAL(120, module_position.to_deg(), 1);
}
