#ifndef WII_JOYSTICK_H
#define WII_JOYSTICK_H

/*!
  \file
  \brief Wii リモコン

  \attention 接続後、最初のデータが返されるまでの各メソッドの戻り値は不定となる

  \author Satofumi KAMIMURA

  $Id: WiiJoystick.h 238 2008-09-26 21:34:56Z satofumi $

  \todo タイムスタンプのインターフェースを定義して継承すべき
*/

#include "GridTypes.h"
#include <vector>
#include <deque>
#include <memory>
#include <time.h>


namespace qrk
{
  /*!
    \brief Wii リモコンの制御クラス
  */
  class WiiJoystick
  {
    WiiJoystick(const WiiJoystick& rhs);
    WiiJoystick& operator = (const WiiJoystick& rhs);

    struct pImpl;
    const std::auto_ptr<pImpl> pimpl;

  public:
    enum {
      DefaultTimeout = 1000,    //!< タイムアウト時間

      AxisX = 0,                //!< 十字ボタンの X 軸
      AxisY = 1,                //!< 十字ボタンの Y 軸

      BUTTON_A = 0,             //!< A ボタン
      BUTTON_B,                 //!< B ボタン
      BUTTON_MINUS,             //!< &ndash; ボタン
      BUTTON_PLUS,              //!< + ボタン
      BUTTON_HOME,              //!< HOME ボタン
      BUTTON_1,                 //!< 1 ボタン
      BUTTON_2,                 //!< 2 ボタン
    };

    /*!
      \brief IR センサの情報

      位置は [0.0, 1.0] の範囲で返される
    */
    typedef struct {
      qrk::Grid<double> position; //!< 位置
      size_t size;                //!< 強度
    } ir_position;


#if 0
    typedef struct {
      struct timespec timestamp;
      double x;
      double y;
      double z;
    } Acceleration;
#endif


    WiiJoystick(void);
    ~WiiJoystick(void);


    /*!
      \brief Wii リモコン数の取得

      \retrun Wii リモコン数

      \attention 未実装
    */
    bool findController(int timeout = DefaultTimeout);


    /*!
      \brief 状態の取得

      \return 状態を示す文字列
    */
    const char* what(void);


    /*!
      \brief 接続

      Wii への接続を行う。

      // param [i] id 接続する Wii リモコンの番号

      \todo 複数の Wii リモコンに接続できるようにする
    */
    bool connect(void);


    /*!
      \brief 切断
    */
    void disconnect(void);


    /*!
      \brief 接続中か
    */
    bool isConnected(void);


    /*!
      \brief 軸の数を取得

      \return 軸の数
    */
    size_t getAxisNum(void);


    /*!
      \brief 軸の入力値を取得

      \param[in] index 軸の番号

      \return 軸の入力値
    */
    int getAxisValue(size_t index);


    /*!
      \brief ボタンの数を取得

      \return ボタンの数
    */
    size_t getButtonsNum(void);


    /*!
      \brief ボタン押下情報の取得

      \param[in] index ボタンの番号

      \retval true 押されている
      \retval false 離されている
    */
    bool isButtonPressed(size_t index);


    /*!
      \brief ボタンが押下回数の取得

      最後にこのメソッドが呼ばれてから、ボタンが何回押されたかを返す

      \param[in] index ボタンの番号

      \return ボタンの押下回数

      \attention 未実装
    */
    int getButtonPressedTimes(size_t index);


    /*!
      \brief 加速度情報の取得

      \param[out] acceleration 加速度情報 [g]
      \param[out] timestamp データ取得時のタイムスタンプ

      \attention timestamp は未実装
    */
    void getAcceleration(qrk::Grid3D<double>& acceleration,
                         size_t* timestamp = NULL);

    //void getAccelerations(std::deque<Acceleration>& accelerations);

    /*!
      \brief バッテリー充電率の取得

      \return バッテリーの充電率
    */
    size_t getBatteryPercent(void);


    /*!
      \brief IR 情報の取得

      \param[in] positions IR 検出位置、強度

    */
    bool getIrPosition(std::vector<ir_position>& positions);


    /*!
      \brief LED の点灯

      \param[in] led_value 下位 4 bit の 1 の箇所を点灯させる
    */
    bool setLed(unsigned char led_value);


    /*!
      \brief 振動の制御

      \param[in] rumble 振動させるとき true
    */
    bool setRumble(bool rumble);

    // !!! 音


    /*!
      \brief 加速度に対する移動平均の個数を設定

    */
    void setAccelerationAverageSize(size_t size);
  };
};

#endif /*! WII_JOYSTICK_H */
