/*!
  \file
  \brief シリアル通信 (Windows 実装)

  Serial Communication Interface 制御


  \author Satofumi KAMIMURA

  $Id: serial_ctrl_win.c 280 2008-10-17 04:43:23Z satofumi $
*/

#include <stdio.h>
#include "serial_ctrl.h"

#include <ctype.h>


/* 接続 */
int serial_connect(serial_t *serial, const char *device, long baudrate) {

  char adjusted_device[16];

  /* COM ポートを開く */
  _snprintf(adjusted_device, 16, "\\\\.\\%s", device);
  serial->hCom_ = CreateFileA(adjusted_device, GENERIC_READ | GENERIC_WRITE, 0,
			      NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (serial->hCom_ == INVALID_HANDLE_VALUE) {
#if 0
    LPVOID lpText;
    FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
                  FORMAT_MESSAGE_FROM_SYSTEM |
                  FORMAT_MESSAGE_IGNORE_INSERTS,
                  NULL, GetLastError(),
                  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                  (LPTSTR)&lpText, 0, NULL);
    MessageBoxW(NULL, (LPCWSTR)lpText,
                (LPCWSTR)"Error message", MB_ICONHAND|MB_OK);
    LocalFree(lpText);
#endif
    fprintf(stderr, "open failed: %s\n", device);
    return -1;
  }

  /* 通信サイズの更新 */
  SetupComm(serial->hCom_, 4096 * 4, 4096);

  /* ボーレートの変更 */
  serial_setBaudrate(serial, baudrate);

  /* シリアル制御構造体の初期化 */
  serial->last_ch_ = '\0';

  return 0;
}


/* 切断 */
void serial_disconnect(serial_t *serial) {

  if (serial->hCom_) {
    CloseHandle(serial->hCom_);
    serial->hCom_ = INVALID_HANDLE_VALUE;
  }
}


int serial_isConnected(serial_t *serial) {

  return (serial->hCom_ == INVALID_HANDLE_VALUE) ? 0 : 1;
}


/* ボーレートの変更 */
int serial_setBaudrate(serial_t *serial, long baudrate) {

  DCB dcb;

  /* HandleTerminals から fd を探し、ボーレートを変更する */

  GetCommState(serial->hCom_, &dcb);
  dcb.BaudRate = baudrate;
  dcb.ByteSize = 8;
  dcb.Parity = NOPARITY;
  dcb.fParity = FALSE;
  dcb.StopBits = ONESTOPBIT;
  SetCommState(serial->hCom_, &dcb);

  return 0;
}


/* 送信 */
int serial_send(serial_t *serial, const char *data, int data_size) {

  DWORD n;

  WriteFile(serial->hCom_, data, (DWORD)data_size, &n, NULL);

  return n;
}


/* 受信 */
int serial_recv(serial_t *serial, char* data, int data_size_max, int timeout) {

  DWORD n;
  size_t readable_size;
  DWORD dwErrors;
  COMSTAT ComStat;
  int filled = 0;

  if (data_size_max <= 0) {
    return 0;
  }

  /* 書き戻した１文字があれば、書き出す */
  filled = 0;
  if (serial->last_ch_ != '\0') {
    data[0] = serial->last_ch_;
    serial->last_ch_ = '\0';
    ++filled;
  }

  if (data_size_max <= filled) {
    return filled;
  }

  /* 要求サイズ分のデータ受信可能ならば、読み出して返す */
  ClearCommError(serial->hCom_, &dwErrors, &ComStat);
  readable_size = (size_t)ComStat.cbInQue;

  if ((data_size_max - filled) > readable_size) {
    COMMTIMEOUTS pcto;
    int each_timeout = 2;

    if (timeout == 0) {
      /* 存在するだけを読み出させて返す */
      data_size_max = readable_size - filled;
    } else if (timeout < 0) {
      /* タイムアウトの値をゼロにすると、タイマーは動作せず受信を待ち続ける */
      timeout = 0;
      each_timeout = 0;
    }

    /* タイムアウトを設定 */
    GetCommTimeouts(serial->hCom_, &pcto);
    pcto.ReadIntervalTimeout = timeout;
    pcto.ReadTotalTimeoutMultiplier = each_timeout;
    pcto.ReadTotalTimeoutConstant = timeout;
    SetCommTimeouts(serial->hCom_, &pcto);

    /* スレッド切替えを促す */
    delay(1);
  }

  ReadFile(serial->hCom_, &data[filled], (DWORD)data_size_max, &n, NULL);

  return (n + filled);
}


/* １文字書き戻す */
void serial_ungetc(serial_t *serial, char ch) {

  serial->last_ch_ = ch;
}

