/*!
  \file
  \brief シリアル通信 (WIN32 API 実装)

  \author Satofumi KAMIMURA

  $Id: SerialCtrl_win.cpp 302 2008-10-26 23:12:43Z satofumi $

  \todo ボーレートの変更を今風の書き方で書き直す
*/

#include "SerialCtrl.h"
#include "RingBuffer.h"
#include "delay.h"
#include <windows.h>
#include <string>

using namespace qrk;


struct SerialCtrl::pImpl
{
  std::string error_message_;   //!< エラー状態
  HANDLE hCom_;                 //!< シリアル資源
  RingBuffer<char> ring_buffer_; //!< 受信バッファ
  long baudrate_;


  pImpl(void) : error_message_("no error."), hCom_(INVALID_HANDLE_VALUE),
                baudrate_(0)
  {
  }


  ~pImpl(void)
  {
    disconnect();
  }


  bool connect(const char* device, long baudrate)
  {
    /* COM ポートを開く */
    char adjusted_device[16];
    sprintf(adjusted_device, "\\\\.\\%s", device);
    hCom_ = CreateFileA(adjusted_device, GENERIC_READ | GENERIC_WRITE, 0,
                        NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
    if (hCom_ == INVALID_HANDLE_VALUE) {
#if 1
      LPVOID lpText;
      FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
                    FORMAT_MESSAGE_FROM_SYSTEM |
                    FORMAT_MESSAGE_IGNORE_INSERTS,
                    NULL, GetLastError(),
                    MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                    (LPTSTR)&lpText, 0, NULL);
      MessageBoxW(NULL, (LPCWSTR)lpText, NULL, MB_ICONHAND | MB_OK);
      LocalFree(lpText);
#endif
      fprintf(stderr, "open failed: %s\n", device);
      return false;
    }

    // 通信サイズの更新
    SetupComm(hCom_, 4096 * 4, 4096);

    // ボーレートの変更
    bool ret = setBaudrate(baudrate);
    if (! ret) {
      // !!! エラー状態を格納すべき
      // !!!
      fprintf(stderr, "fail SerialCtrl::setBaudrate()\n");
    }
    return true;
  }


  void disconnect(void)
  {
    if (hCom_ != INVALID_HANDLE_VALUE) {
      CloseHandle(hCom_);
      hCom_ = INVALID_HANDLE_VALUE;
    }
  }


  bool setBaudrate(long baudrate)
  {
    DCB dcb;
    GetCommState(hCom_, &dcb);
    dcb.BaudRate = baudrate;
    dcb.ByteSize = 8;
    dcb.Parity = NOPARITY;
    dcb.fParity = FALSE;
    dcb.StopBits = ONESTOPBIT;
    SetCommState(hCom_, &dcb);
    baudrate_ = baudrate;

    return true;
  }


  int recv(char* data, size_t count, int timeout)
  {
    if (hCom_ == INVALID_HANDLE_VALUE) {
      // 未接続ならば、戻る
      return 0;
    }

    /* 要求サイズ分のデータ受信可能ならば、読み出して返す */
    DWORD dwErrors;
    COMSTAT ComStat;
    ClearCommError(hCom_, &dwErrors, &ComStat);
    size_t readable_size = (size_t)ComStat.cbInQue;

    if (count > readable_size) {
      COMMTIMEOUTS pcto;
      int each_timeout = 2;

      if (timeout == 0) {
        /* 存在するだけを読み出させて返す */
        count = readable_size;

      } else if (timeout < 0) {
        /* タイムアウトの値をゼロにするとタイマーは動作せず受信を待ち続ける */
        timeout = 0;
        each_timeout = 0;
      }

      /* タイムアウトを設定 */
      GetCommTimeouts(hCom_, &pcto);
      pcto.ReadIntervalTimeout = timeout;
      pcto.ReadTotalTimeoutMultiplier = each_timeout;
      pcto.ReadTotalTimeoutConstant = timeout;
      SetCommTimeouts(hCom_, &pcto);

      /* スレッド切替えを促す */
      delay(1);
    }
    DWORD n;
    ReadFile(hCom_, data, (DWORD)count, &n, NULL);

    return n;
  }


  void flush(void)
 {
    // !!! 未実装
  }


  void updateRecvBuffer(size_t require_n, int timeout)
 {
    enum { BufferSize = 4096 };
    char buffer[BufferSize];

    if ((require_n == 0) && (timeout == 0)) {
      // 受信サイズのチェックとみなし、バッファ更新後に戻る
      int n = recv(buffer, BufferSize, 0);
      if (n > 0) {
        ring_buffer_.put(buffer, n);
      }
      return;
    }

    size_t readable_n = ring_buffer_.size();
    if (readable_n >= require_n) {
      // 必要なだけバッファにデータがあるならば、戻る
      return;
    }

    // 要求サイズ分、受信バッファのタイムアウト付き読み出しを行う
    while (1) {
      int require_n_actual = static_cast<int>(require_n - ring_buffer_.size());
      if (require_n_actual == 0) {
        // 受信完了
        break;
      }

      // １回の読み出しは、バッファサイズで制限する
      require_n_actual =
        std::min(require_n_actual, static_cast<int>(BufferSize));

      int n = recv(buffer, require_n_actual, timeout);
      if (n > 0) {
        ring_buffer_.put(buffer, n);
      }
      if (n != require_n_actual) {
        // タイムアウト
        break;
      }
    }
  }
};


SerialCtrl::SerialCtrl(void) : pimpl(new pImpl)
{
}


SerialCtrl::~SerialCtrl(void)
{
}


const char* SerialCtrl::what(void)
{
  return pimpl->error_message_.c_str();
}


bool SerialCtrl::connect(const char* device, long baudrate)
{
  if (isConnected()) {
    // 接続されていれば、切断する
    disconnect();
  }
  return pimpl->connect(device, baudrate);
}


void SerialCtrl::disconnect(void)
{
  if (! isConnected()) {
    // !!! disconnect() が呼ばれたが未接続、というエラーメッセージを格納する
    return;
  }
  return pimpl->disconnect();
}


bool SerialCtrl::setBaudrate(long baudrate)
{
  if (! isConnected()) {
    // !!!
    //fprintf(stderr, "setBaudrate: isConnected()\n");
    return false;
  }
  return pimpl->setBaudrate(baudrate);
}


long SerialCtrl::baudrate(void)
{
  return pimpl->baudrate_;
}


bool SerialCtrl::isConnected(void)
{
  return (pimpl->hCom_ != INVALID_HANDLE_VALUE) ? true : false;
}


int SerialCtrl::send(const char* data, size_t count)
{
  // !!! 未接続時の処理

  DWORD n;
  WriteFile(pimpl->hCom_, data, (DWORD)count, &n, NULL);

  return n;
}


int SerialCtrl::recv(char* data, size_t count, int timeout)
{
  // !!! 未接続時の処理
  // !!! ただし、バッファにデータがあれば、読み出せるようにすべき

  // 受信バッファの更新
  pimpl->updateRecvBuffer(count, timeout);

  // 受信バッファの内容を取得
  size_t readable_n = pimpl->ring_buffer_.size();
  size_t require_n = std::min(count, readable_n);

  return static_cast<int>(pimpl->ring_buffer_.get(data, require_n));
}


size_t SerialCtrl::size(void)
{
  // リングバッファのサイズを更新
  pimpl->updateRecvBuffer(0, 0);
  return pimpl->ring_buffer_.size();
}


void SerialCtrl::flush(void)
{
  pimpl->flush();
}


void SerialCtrl::clear(void)
{
  flush();
  pimpl->updateRecvBuffer(0, 0);
  pimpl->ring_buffer_.clear();
}


void SerialCtrl::ungetc(const char ch)
{
  pimpl->ring_buffer_.ungetc(ch);
}
