#ifndef CONNECTION_H
#define CONNECTION_H

/*!
  \file
  \brief 通信インターフェース

  \author Satofumi KAMIMURA

  $Id: Connection.h 287 2008-10-21 21:32:54Z satofumi $
*/

#include <stddef.h>


namespace qrk
{
  enum {
    RecvTimeout = -1,
    ErrorLastIndex = -2,
  };

  /*!
    \brief 通信インターフェース
  */
  class Connection
  {
  public:
    virtual ~Connection(void) {}


    /*!
      \brief 内部状態を返す

      \return 内部状態を示す文字列
    */
    virtual const char* what(void) = 0;


    /*!
      \brief デバイスへの接続

      \param[i] device 接続デバイス名
      \param[i] baudrate 接続ボーレート

      \retval true 成功
      \retval false 失敗
    */
    virtual bool connect(const char* device, long baudrate) = 0;


    /*!
      \brief 切断
    */
    virtual void disconnect(void) = 0;


    /*!
      \brief ボーレートの変更

      \param[in] baudrate ボーレート

      \retval 0 正常
      \retval < 0 エラー
    */
    virtual bool setBaudrate(long baudrate) = 0;
    virtual long baudrate(void) = 0;


    /*!
      \brief 接続状態を返す

      \retval true 接続中
      \retval false 切断中
    */
    virtual bool isConnected(void) = 0;


    /*!
      \brief 送信

      \param data [i] 送信データ
      \param size [i] 送信バイト数

      \return 送信したバイト数
    */
    virtual int send(const char* data, size_t count) = 0;


    /*!
      \brief 受信

      \param data [o] 受信用バッファ
      \param size [i] 受信バッファの最大サイズ
      \param timeout [i] タイムアウト時間 [msec]
    */
    virtual int recv(char* data, size_t count, int timeout) = 0;


    /*!
      \brief 受信済みデータのバイト数を返す

      \param timeout [i] タイムアウト時間 [msec]

      \return 受信済みデータのバイト数
    */
    virtual size_t size(void) = 0;


    /*!
      \brief
    */
    virtual void flush(void) = 0;


    /*!
      \brief 送受信データのクリア

      送信バッファ、受信済みデータをクリアする
    */
    virtual void clear(void) = 0;


    /*!
      \brief
    */
    virtual void ungetc(const char ch) = 0;
  };
}

#endif /* !CONNECTION_H */
