#ifndef LEAST_SQUARE_H
#define LEAST_SQUARE_H

/*!
  \file
  \brief 最小二乗の計算

  \author Satofumi KAMIMURA

  $Id: LeastSquare.h 222 2008-09-17 12:52:15Z satofumi $

  http://szksrv.isc.chubu.ac.jp/lms/lms1.html より
*/

#include "AngleTypes.h"
#include "MathUtils.h"


namespace qrk
{
  template <class T>
  class LeastSquare
  {
    LeastSquare(const LeastSquare& rhs);
    LeastSquare& operator = (const LeastSquare& rhs);

    size_t n_;
    T b_;
    T c_;
    T d_;
    T e_;


  public:
    LeastSquare(void) : n_(0), b_(0), c_(0), d_(0), e_(0) {
    }


    ~LeastSquare(void) {
    }


    void clear(void)
    {
      n_ = 0;
      b_ = 0;
      c_ = 0;
      d_ = 0;
      e_ = 0;
    }


    void push_back(T x, T y)
    {
      b_ += x * x;
      c_ += y;
      d_ += x * y;
      e_ += x;

      ++n_;
    }


    Angle gradients(void)
    {
      if (n_ <= 0) {
        return rad(0);
      }

      T a = ((n_ * d_) - (c_ * e_)) / ((n_ * b_) - (e_ * e_));
      return rad(atan2(a, 1));
    }


    T intercepts(void)
    {
      if (n_ <= 0) {
        return 0;
      }

      return ((b_ * c_) - (d_ * e_)) / ((n_ * b_) - (e_ * e_));
    }
  };
};

#endif /* !LEAST_SQUARE_H */
