/*!
  \file
  \brief 距離データの閲覧ウィジット

  \author Satofumi KAMIMURA

  $Id: LengthViewWidget.cpp 1817 2010-05-09 09:04:32Z satofumi $

  \todo index がゼロから始まるようにする
  \todo index の数値も右寄せにする
  \todo 編集できないようにする
  \todo 選択された列を、強調表示する。データが不連続でも表示させる
*/

#include "LengthViewWidget.h"
#include <QHeaderView>
#include <QFileDialog>
#include <QTextStream>
#include <QShortcut>

using namespace std;


struct LengthViewWidget::pImpl
{
    LengthViewWidget* widget_;
    size_t min_distance_;

    pImpl(LengthViewWidget* widget) : widget_(widget), min_distance_(0)
    {
    }


    void initializeForm(void)
    {
        widget_->length_table_->
            setEditTriggers(QAbstractItemView::NoEditTriggers);

        // コンテナ
        for (int i = 0; i < 2; ++i) {
            widget_->length_table_->
                horizontalHeader()->setResizeMode(i, QHeaderView::Stretch);
        }

        // イベント
        connect(widget_->update_button_, SIGNAL(clicked()),
                widget_, SLOT(updatePressed()));
        connect(widget_->save_button_, SIGNAL(clicked()),
                widget_, SLOT(savePressed()));
        connect(widget_->close_button_, SIGNAL(clicked()),
                widget_, SLOT(close()));
        connect(widget_->length_table_, SIGNAL(itemSelectionChanged()),
                widget_, SLOT(itemSelectionChanged()));

        // キー割り付け
        (void) new QShortcut(Qt::CTRL + Qt::Key_Q, widget_,
                             SLOT(quitPressed()));
        (void) new QShortcut(Qt::CTRL + Qt::Key_W, widget_, SLOT(close()));
    }


    void setData(int index, const vector<long>& data)
    {
        size_t row_count = widget_->length_table_->rowCount();
        size_t data_size = data.size();
        if (data_size == 0) {
            return;
        }
        widget_->save_button_->setEnabled(true);

        if (data_size > row_count) {
            widget_->length_table_->setRowCount(data_size);

            QStringList labels;
            for (size_t i = 0; i < data_size; ++i) {
                labels.push_back(tr("%1").arg(i));
            }
            widget_->length_table_->setVerticalHeaderLabels(labels);
        }

        const QBrush error_brush(QColor(255, 0, 0));
        for (size_t i = 0; i < data_size; ++i) {
            long length = data[i];
            QTableWidgetItem *new_item =
                new QTableWidgetItem(tr("%1").arg(length));
            new_item->setTextAlignment(Qt::AlignRight | Qt::AlignVCenter);

            if ((index == 0) && (length < static_cast<long>(min_distance_))) {
                // エラーデータであることを強調するために、色を変える
                new_item->setForeground(error_brush);
            }
            widget_->length_table_->setItem(i, index, new_item);
        }
        widget_->length_table_->setRowCount(data_size);
    }
};


LengthViewWidget::LengthViewWidget(QWidget* parent)
    : QWidget(parent), pimpl(new pImpl(this))
{
    setupUi(this);
    pimpl->initializeForm();
}


LengthViewWidget::~LengthViewWidget(void)
{
}


void LengthViewWidget::clear(void)
{
    length_table_->clearContents();
}


void LengthViewWidget::setMinDistance(size_t min_distance)
{
    pimpl->min_distance_ = min_distance;
}


void LengthViewWidget::setLength(const vector<long>& length_data)
{
    pimpl->setData(0, length_data);
}


void LengthViewWidget::setIntensity(const vector<long>& intensity_data)
{
    pimpl->setData(1, intensity_data);
}


void LengthViewWidget::updatePressed(void)
{
    emit lengthViewRequest();
}


void LengthViewWidget::savePressed(void)
{
    // 保存処理
    QString default_file_name = "length_data.csv";
    QString file_name =
        QFileDialog::getSaveFileName(this, tr("Save length data."),
                                     default_file_name, tr("CSV (*.csv)"));
    if (file_name.isEmpty()) {
        return;
    }

    QFile save_file(file_name);
    if (! save_file.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return;
    }
    QTextStream fout(&save_file);

    size_t row_count = length_table_->rowCount();
    for (size_t y = 0; y < row_count; ++y) {

        fout << y << ',';

        for (int x = 0; x < 2; ++x) {
            QTableWidgetItem* item = length_table_->item(y, x);
            if (! item) {
                continue;
            }
            long value = item->text().toLong();

            fout << value << ',';
        }
        fout << '\n';
    }
}


void LengthViewWidget::closeEvent(QCloseEvent* event)
{
    static_cast<void>(event);

    emit widgetClose("LengthViewWidget");
    close();
}


void LengthViewWidget::quitPressed(void)
{
    emit quit();
}


void LengthViewWidget::itemSelectionChanged(void)
{
    vector<int> indexes;

    QList<QTableWidgetItem*> selected_list = length_table_->selectedItems();
    for (QList<QTableWidgetItem*>::iterator it = selected_list.begin();
         it != selected_list.end(); ++it) {
        if (*it) {
            indexes.push_back((*it)->row());
        }
    }
    emit drawLineRequest(indexes);
}
