/*!
  \file
  \brief データ描画ウィジット

  \author Satofumi KAMIMURA

  $Id: DrawWidget.cpp 976 2009-06-06 23:21:42Z satofumi $
*/

#include "DrawWidget.h"
#include "UrgDrawWidget.h"
#include <QHBoxLayout>
#include <QSettings>
#include <QShortcut>


namespace
{
  const char* Organization = "Hokuyo LTD.";
  const char* Application = "Multi URG Viewer";
}


struct DrawWidget::pImpl
{
  DrawWidget* widget_;
  QHBoxLayout layout_;
  UrgDrawWidget urg_draw_widget_;
  UrgStateWidget* urg_state_widget_;


  pImpl(DrawWidget* widget)
    : widget_(widget), layout_(widget_), urg_state_widget_(NULL)
  {
    layout_.addWidget(&urg_draw_widget_);
  }


  void initializeForm(void)
  {
    // 位置と大きさを読み出す
    loadSettings();

    enum {
      MinimumWidth = 200,
      MinimumHeight = 200,
    };
    widget_->setMinimumSize(MinimumWidth, MinimumHeight);

    // キー操作によるウィンドウ操作
    (void) new QShortcut(Qt::Key_Return, widget_, SLOT(initializeView()));
    (void) new QShortcut(Qt::Key_Less, widget_, SLOT(zoomSmaller()));
    (void) new QShortcut(Qt::Key_Comma, widget_, SLOT(zoomSmaller()));
    (void) new QShortcut(Qt::Key_Greater, widget_, SLOT(zoomLarger()));
    (void) new QShortcut(Qt::Key_Period, widget_, SLOT(zoomLarger()));

    // Ctrl-w でウィンドウを閉じ、Ctrl-q で終了
    (void) new QShortcut(Qt::CTRL + Qt::Key_Q, widget_, SLOT(quitPressed()));
    (void) new QShortcut(Qt::ALT + Qt::Key_F4, widget_, SLOT(quitPressed()));
    (void) new QShortcut(Qt::CTRL + Qt::Key_W, widget_, SLOT(close()));
  }


  void loadSettings(void)
  {
    QSettings settings(Organization, Application);
    widget_->restoreGeometry(settings.value("geometry").toByteArray());
  }


  void saveSettings(void)
  {
    QSettings settings(Organization, Application);
    settings.setValue("geometry", widget_->saveGeometry());
  }


  void setUrgData(std::vector<long>& data,
                  const qrk::RangeSensor* sensor, int timestamp)
  {
    urg_draw_widget_.setUrgData(data, sensor, timestamp);
    urg_draw_widget_.redraw();
  }
};


DrawWidget::DrawWidget(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this))
{
  pimpl->initializeForm();
}


DrawWidget::~DrawWidget(void)
{
}


void DrawWidget::closeEvent(QCloseEvent* event)
{
  static_cast<void>(event);

  pimpl->saveSettings();
}


void DrawWidget::quitPressed(void)
{
  close();
  emit quit();
}


void DrawWidget::initializeView(void)
{
  pimpl->urg_draw_widget_.initializeView();
}


void DrawWidget::zoomSmaller(void)
{
  pimpl->urg_draw_widget_.updateZoomRatio(+1);
}


void DrawWidget::zoomLarger(void)
{
  pimpl->urg_draw_widget_.updateZoomRatio(-1);
}


void DrawWidget::setTitle(const QString& title,
                          UrgStateWidget* urg_state_widget)
{
  pimpl->widget_->setWindowTitle(title);
  pimpl->urg_state_widget_ = urg_state_widget;
  pimpl->urg_draw_widget_.clear();
}


void DrawWidget::setUrgData(std::vector<long>& data,
                            const qrk::RangeSensor* sensor, int timestamp,
                            UrgStateWidget* urg_state_widget)
{
  if (urg_state_widget != pimpl->urg_state_widget_) {
    return;
  }

  pimpl->setUrgData(data, sensor, timestamp);
}


void DrawWidget::disconnected(UrgStateWidget* urg_state_widget)
{
  if (urg_state_widget == pimpl->urg_state_widget_) {
    hide();
    pimpl->urg_state_widget_ = NULL;
  }
}
