/*!
  \file
  \brief 時間再生の操作

  \author Satofumi KAMIMURA

  $Id: TicksControlWidget.cpp 1053 2009-06-20 22:52:51Z satofumi $
*/

#include "TicksControlWidget.h"
#include "ManagedTicks.h"
#include <QTimer>
#include <QShortcut>

using namespace qrk;


namespace
{
  enum {
    RedrawInterval = 100,       // [msec]
  };
}


struct TicksControlWidget::pImpl
{
  TicksControlWidget* widget_;
  ManagedTicks managed_ticks_;

  QTimer redraw_timer_;


  pImpl(TicksControlWidget* widget) : widget_(widget)
  {
    redraw_timer_.setInterval(RedrawInterval);
  }


  void initializeForm(void)
  {
    widget_->ticks_label_->hide();

    connect(&redraw_timer_, SIGNAL(timeout()), widget_, SLOT(updateTicks()));
    connect(widget_->play_button_, SIGNAL(clicked()),
            widget_, SLOT(playPressed()));
    connect(widget_->pause_button_, SIGNAL(clicked()),
            widget_, SLOT(pausePressed()));
    connect(widget_->speed_up_button_, SIGNAL(clicked()),
            widget_, SLOT(speedUpPressed()));
    connect(widget_->speed_down_button_, SIGNAL(clicked()),
            widget_, SLOT(speedDownPressed()));
  }
};


TicksControlWidget::TicksControlWidget(QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this))
{
  setupUi(this);
  pimpl->initializeForm();

  (void) new QShortcut(Qt::CTRL + Qt::Key_W, this, SLOT(close()));
}


TicksControlWidget::~TicksControlWidget(void)
{
}


void TicksControlWidget::closeEvent(QCloseEvent* event)
{
  static_cast<void>(event);
  // !!!
}


void TicksControlWidget::updateTicks(void)
{
  int ticks_sec = ticks() / 1000;

  int hour = ticks_sec / (60 * 60);
  ticks_sec %= (60 * 60);

  int minute = ticks_sec / 60;
  ticks_sec %= 60;

  QString ticks_text;
  ticks_text.sprintf("%02d : %02d : %02d", hour, minute, ticks_sec);
  ticks_label_->setText(ticks_text);
}


void TicksControlWidget::playPressed(void)
{
  pimpl->managed_ticks_.play();
}


void TicksControlWidget::pausePressed(void)
{
  pimpl->managed_ticks_.pause();
}


void TicksControlWidget::speedUpPressed(void)
{
  pimpl->managed_ticks_.moreFaster();
}


void TicksControlWidget::speedDownPressed(void)
{
  pimpl->managed_ticks_.moreSlower();
}


long TicksControlWidget::ticks(void) const
{
  return pimpl->managed_ticks_.ticks();
}


void TicksControlWidget::showTicks(void)
{
  pimpl->redraw_timer_.start();
  ticks_label_->show();
}


void TicksControlWidget::hideTicks(void)
{
  ticks_label_->hide();
  pimpl->redraw_timer_.stop();
}
