#!/usr/bin/ruby
# doxygen フォーマットの NEWS から、RSS フィードを生成するスクリプト


# １つの記事は、
#
#   - 年月日
#   - 概要
#   - \n \n (１つの記事の最後)
#
# から構成されるとする。
#
#  例)
#  - 2009-08-21
#    - シミュレータが動作するバージョンを、アルファ版としてリリースしました。\n
#      Windows 用の実行バイナリも用意してあり、再コンパイル不要なプログラム開発を体験できます。
#      \n \n

require "rss"

$KCODE = 'UTF8'


if ARGV.length < 2 then
  print "usage:\n\truby " + __FILE__ + " news.dox forge_conf.txt\n\n"
  exit
end

news_file = ARGV[0]
config_file = ARGV[1]


# 設定ファイルの読み出し
class ConfigInformation
  attr_reader :rss_10_file, :rss_20_file, :rss_base, :rss_title, :rss_description

  def loadReplaceWords(io)
    while line = io.gets
      case line.chomp
      when /^\s*$/
        # 空行までを読み出す
        return

      when /^(.+?)\s(.+)$/
        @replace_words[$1] = $2
      end
    end
  end

  def initialize(file)
    @rss_10_file = nil
    @rss_20_file = nil
    @rss_base = 'rss_base_address'
    @rss_title = 'rss title'
    @rss_description = 'rss description'

    File.open(file) { |io|
      while line = io.gets
        case line.chomp
        when /^#/
          next

        when /OutputRss10\s(.+)/
          @rss_10_file = $1

        when /OutputRss20\s(.+)/
          @rss_20_file = $1

        when /RssBaseUrl\s(.+)/
          @rss_base = $1

        when /RssTitle\s(.+)/
          @rss_title = $1

        when /RssDescription\s(.+)/
          @rss_description = $1

        when /InsertRss\s[Yy][Ee][Ss]/
          @rss_insert = true
        end
      end
    }
  end
end
configs = ConfigInformation.new(config_file)


articles = []
page = nil
File.open(news_file) { |io|

  tag = nil
  title = nil
  date = nil
  article = nil
  while line = io.gets
    line.chomp!

    # ページ名、およびタイトルと年月日を読み出す
    case line
    when /\s+\\page\s+([a-zA-Z0-9_]+)/
      page = $1 + '.html'

    when /\s+\\section\s+([a-zA-Z0-9_]+)\s+(.+)/
      tag = $1
      title = $2

    when /  - (\d\d\d\d)-(\d\d)-(\d\d)/
      date = Time.local($1, $2, $3)
    end

    # 年月日が読み出せたら、それ以降の "\n \n" までを記事として登録する
    if date then
      article = ''
      while line = io.gets
        line.chomp!

        if line =~ /\\n\s*\\n/
          break
        end
        article += line.strip.sub(/\\n/, '')
      end

      article.strip!
      article = article[1, article.length - 1]

      # RSS に登録
      articles << {
        'link' => page + '#' + tag,
        'title' => title,
        'date' => date,
        'description' => article
      }

      tag = nil
      title = nil
      date = nil
      article = nil
    end

  end
}


def createRss(type, fname, rss_settings, articles)
  rss = RSS::Maker.make(type) do |maker|
    maker.channel.about = rss_settings['base'] + File.basename(fname)
    maker.channel.title = rss_settings['title']
    maker.channel.description = rss_settings['description']
    maker.channel.link = rss_settings['base']
    maker.items.do_sort = true

    items_num = 0;
    articles.each { |article|
      maker.items.new_item do |item|
        item.link = rss_settings['base'] + article['link']
        item.title = article['title']
        item.date = article['date']
        item.description = article['description']
      end

      items_num += 1
      if items_num >= 15
        break
      end
    }
  end

  File.open(fname, 'w') { |io|
    io << rss.to_s
  }
end


# RSS を生成
rss_settings = {
  'base' => configs.rss_base,
  'title' => configs.rss_title,
  'description' => configs.rss_description
}
createRss('1.0', configs.rss_10_file, rss_settings, articles)
createRss('2.0', configs.rss_20_file, rss_settings, articles)
