/**
 * Copyright (C) 2010,2011 Yasuhiro ABE <yasu@yasundial.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/

#include "stdafx.h"

std::string genuuid() {
	GUID guid;
	HRESULT hr = CoCreateGuid(&guid);
	
	std::ostringstream oss;
	oss << toHex<int>(guid.Data1,8) << "-" << toHex(guid.Data2,4) << "-" <<
		toHex(guid.Data3,4) << "-" << toHex((int)guid.Data4[0],2) << toHex((int)guid.Data4[1],2) <<
		"-" << std::setw(2) << std::setfill('0') << std::hex << (int)guid.Data4[2] << (int)guid.Data4[3] << 
		(int)guid.Data4[4] << (int)guid.Data4[5] << (int)guid.Data4[6] << (int)guid.Data4[7];
	
	return oss.str();
}

static void help(TCHAR* cmdpath) {
	using namespace std;
	_tprintf(_T("Usage: %s <dbfilepath> <action> <filepath>\n"), cmdpath);
	_tprintf(_T("\t<action>: generate|check|delete"));
}

int _tmain(int argc, _TCHAR* argv[])
{  
	if(argc != 4) {
		help(argv[0]);
		return 1;
	}
    const tstring dbname(argv[1]);
	const tstring action(argv[2]);
	const tstring pathname(argv[3]);
	boost::filesystem::path p(argv[3]);
	if(! boost::filesystem::exists(p)) {
		help(argv[0]);
		return 1;
	}
	boost::filesystem::path dbpath(dbname);

	TCHAR* dbfilename = argv[1];
	TCHAR* filename = argv[3];

	yamain* main = new yamain(filename, dbfilename);

	std::string dbuuid = main->getUUIDFromDB();
	std::string diskuuid = main->getUUIDFromDisk();
#ifdef _DEBUG
	std::cout << "dbuuid: " << dbuuid << std::endl;
	std::cout << "diskuuid: " << diskuuid << std::endl;
#endif

	int rc;
	std::string uuid;
	uuidstatus status = bad;
	if       (dbuuid.empty() && diskuuid.empty()) {
	  // initialize db and disk
	  uuid = genuuid();
	  rc = main->setUUIDToDB(uuid);
	  if(rc) {
	    rc = main->setUUIDToDisk(uuid);
	  } else {
	    status = bad;
	  }
        } else if(  dbuuid.empty() && ! diskuuid.empty()) {
	  // recheck uuid from db
	  uuid = diskuuid;
	  if(main->checkUUIDonDB(diskuuid)) {
	    // exists on db: file was moved
	    rc = main->updateFilepathonDBByUUID(filename, diskuuid);
	    if(rc) {
	      status = moved;
	    } else {
	      status = bad;
	    }
	  } else {
	    // maybe, something wrong
	    rc = main->setUUIDToDB(diskuuid);
	    if(rc) {
          // recreated 
	      status = not_found_on_db;
	    } else {
          // maybe, a db file was initialized.
	      status = bad;
	    }
	  }
	} else if(! dbuuid.empty() &&   diskuuid.empty()) {
	  // file was recreated
	  rc = main->setUUIDToDisk(dbuuid);
	  if(rc) {
	    uuid = dbuuid;
	    status = recreated;
	  } else {
	    status = bad;
	  }
	} else if(! dbuuid.empty() && ! diskuuid.empty()) {
	  // normal state. just update check.
	  if(dbuuid.compare(diskuuid) == 0) {
	    
	    uuid = diskuuid;
	    status = normal;
	  } else {
	    main->setUUIDToDB(diskuuid);
	    uuid = diskuuid;
	    status = moved;
	  }
	}
	switch(status) {
	case normal:
#ifdef _DEBUG
	  std::cout << "[status] normal." << std::endl;
#endif
	  break;
	case recreated:
	  std::cout << "[status] file was recreated." << std::endl;
	  break;
	case moved:
	  std::cout << "[status] file was moved." << std::endl;
	  break;
	case not_found_on_db:
	  std::cout << "[status] data mismatched (, local db might be initialized.)" << std::endl;
	  break;
	case bad:
	  std::cout << "[status] error" << std::endl;
	  break;
	}
#ifdef _DEBUG
	std::cout << std::endl << "fileid: " << uuid << std::endl;
#else
	std::cout << std::endl;
#endif
	if(action.compare(_T("check")) == 0) {
	  rc = main->checkHash(true);
	} else if(action.compare(_T("generate")) == 0) { 
	  rc = main->createHash();
	} else if(action.compare(_T("delete")) == 0) { 
	  rc = main->deleteData();
	}
#ifdef _DEBUG
	_tprintf(_T("action: %s\n"), action.c_str());
#endif
	
	delete main;
	return 0;
}

