//---------------------------------------------------------------------------
//  Base Class of CalcUnit
//---------------------------------------------------------------------------
#include "QCalcUnit.h"
//---------------------------------------------------------------------------
#ifdef __USE__MPI
int QCalcUnit::gMdata;
int QCalcUnit::gNqubits;
int QCalcUnit::gProcMask;
int QCalcUnit::gProcBit;
int QCalcUnit::gMyID;
int QCalcUnit::gNumprocs;
#endif

//---------------------------------------------------------------------------
/**
 *
**/
unsigned int
QCalcUnit::insert0(unsigned int i0, unsigned int BitNum) {
    unsigned int msk = (1<<BitNum) -1;
    return ((~msk & i0) << 1) | (msk & i0);
}
//---------------------------------------------------------------------------
/**
 *
**/
unsigned int
QCalcUnit::insert1(unsigned int i0, unsigned int BitNum) {
    unsigned int msk = (1<<BitNum) -1;
    return ((~msk & i0) << 1) | (1 << BitNum) | (msk & i0);
}

//---------------------------------------------------------------------------
// for MPI calculation
//---------------------------------------------------------------------------
#ifdef __USE__MPI
//---------------------------------------------------------------------------
/**
 * N - number of qbits
 */
int
QCalcUnit::initmpi(int N, int myid, int numprocs) {
    // numproc must be a power of 2.
    int p = -1;
    int n = numprocs;
    while (n){
        n = n >> 1;
        p++;
    }
    gProcBit = p;
    gNqubits = N;
    gMdata          = 1 << (N - gProcBit);
    gNumprocs       = numprocs;
    gProcMask       = (gNumprocs - 1) << (N-gProcBit);
    gMyID           = myid;
    return gMdata;
}
//---------------------------------------------------------------------------
inline int
QCalcUnit::getOriginalID(int id){
    return  gMyID << ((gNqubits-gProcBit))|id;
}
//----------------------------------------------------------------------------
inline int
QCalcUnit::getProcFromID(int id){
    return (id >> (gNqubits - gProcBit));
}
//----------------------------------------------------------------------------
void
QCalcUnit::initdata(double R[], double I[]) {
    for (int i = 0; i < gMdata; i++) {
        R[i] = 0;
        I[i] = 0;
    }
    if (0 == gMyID) {
        R[0] = 1;
    }
}
//----------------------------------------------------------------------------
/**
 * Show Data
 */
void
QCalcUnit::showdata(double R[], double I[]) {
    for (int i = 0; i < gMdata; i++) {
        printf("proc=%3d: idx = %03d:%f,%f\n",
               gMyID, getOriginalID(i), R[i], I[i]);
    }
}
//----------------------------------------------------------------------------
/**
 *  Common procedure to prepare data
 *  Return Value:
 *  true:   need to store data
 *  false:  don't need to store
 */
bool
QCalcUnit::setup(const double R[], const double I[],
                 const unsigned int &ix0, const unsigned int &ix1,
                 double &r0, double &i0,
                 double &r1, double &i1) {
    MPI_Status  status;
    MPI_Request req;
    const int   TAG = 0;

    unsigned int idx0 = ix0 & ((1<<(gNqubits-gProcBit))-1);
    unsigned int idx1 = ix1 & ((1<<(gNqubits-gProcBit))-1);

    int p0 = getProcFromID(ix0);
    int p1 = getProcFromID(ix1);

    //----
    // Communication with MPI
    //----
    if (p0 != gMyID && p1 != gMyID) {
        return false;
    }

    if (p0 == gMyID && p1 == gMyID) {
        r0 = R[idx0];
        r1 = R[idx1];
        i0 = I[idx0];
        i1 = I[idx1];
    }

    if (p0 == gMyID && p1 != gMyID) {
        r0 = R[idx0];
        MPI_Isend(&r0, 1, MPI_DOUBLE, p1, TAG,MPI_COMM_WORLD,&req);
        MPI_Irecv(&r1, 1, MPI_DOUBLE, p1, TAG, MPI_COMM_WORLD, &req);
        MPI_Wait(&req, &status);

        i0 = I[idx0];
        MPI_Isend(&i0, 1, MPI_DOUBLE, p1, TAG, MPI_COMM_WORLD, &req);
        MPI_Irecv(&i1, 1, MPI_DOUBLE, p1, TAG, MPI_COMM_WORLD, &req);
        MPI_Wait(&req, &status);
    }
    if (p0 != gMyID && p1 == gMyID) {
        r1 = R[idx1];
        MPI_Isend(&r1, 1, MPI_DOUBLE, p0, TAG, MPI_COMM_WORLD, &req);
        MPI_Irecv(&r0, 1, MPI_DOUBLE, p0, TAG, MPI_COMM_WORLD, &req);
        MPI_Wait(&req, &status);

        i1 = I[idx1];
        MPI_Isend(&i1, 1, MPI_DOUBLE, p0, TAG, MPI_COMM_WORLD, &req);
        MPI_Irecv(&i0, 1, MPI_DOUBLE, p0, TAG, MPI_COMM_WORLD, &req);
        MPI_Wait(&req, &status);
    }
    return true;
}
//----------------------------------------------------------------------------
/**
 *  Common procedure to store data
 */
void QCalcUnit::store(double R[], double I[],
                      const unsigned int &ix0, const unsigned int &ix1,
                      const double &r0, const double &i0,
                      const double &r1, const double &i1) {
    int p0 = getProcFromID(ix0);
    int p1 = getProcFromID(ix1);
    unsigned int idx0 = ix0 & ((1<<(gNqubits-gProcBit))-1);
    unsigned int idx1 = ix1 & ((1<<(gNqubits-gProcBit))-1);

    if (p0 == gMyID) {
        R[idx0] = r0;
        I[idx0] = i0;
    }
    if (p1 == gMyID) {
        R[idx1] = r1;
        I[idx1] = i1;
    }
}
//----------------------------------------------------------------------------
#endif //__USE__MPI
