/*
 *  Tuner.cpp
 */

#define __ENABLE_B25__
//#define __DUMMY_TUNER__

#include "Tuner.h"
#include "debug.h"

#include <stdlib.h>

namespace PTx
{
namespace PTxD
{

void *Tuner_Core_run(void *arg)
{
    ((Tuner::Core *)arg)->run();
    return NULL;
}


#pragma mark -
#pragma mark #################### class Tuner ##################

int Tuner::scan(Tuner *tuners[MAX_DEV_COUNT * MAX_TUNER_COUNT])
{
    int count = 0;

    PT1 *pt1s[MAX_DEV_COUNT];
    int pt1_count = PT1::scan(pt1s);
    for (int i = 0; i < pt1_count; ++i)
    {
        Core *core = new Core();
        if (core != NULL)
        {
            if (core->init(pt1s[i]))
            {
                int cnt = count;
                for (int j = 0; j < MAX_TUNER_COUNT; ++j)
                {
                    Tuner *tuner = new Tuner();
                    if (tuner != NULL)
                    {
                        if (tuner->init(j, core))
                        {
                            tuners[count++] = tuner;
                        }
                        else
                        {
                            delete tuner;
                        }
                    }
                }
                if (cnt == count)
                {
                    delete core;
                }
            }
            else
            {
                delete core;
            }            
        }
    }

    return count;
}

Tuner::Tuner()
{
    _count = 0;
    _packetCount = 0;
    _packetOffset = 0;
    _core = NULL;
    pthread_mutex_init(&_lock, NULL);
    pthread_cond_init(&_cond, NULL);
    _analyzer = new MPEG2::TS::Analyzer(this);
    _recfd = -1;
    _stationInfo = NULL;
    _lockKey = NULL;
}

Tuner::~Tuner()
{
    if (_core != NULL)
    {
        _core->release();
    }
    delete _analyzer;
    pthread_mutex_destroy(&_lock);
    pthread_cond_destroy(&_cond);
    if (_recfd >= 0)
    {
        close(_recfd);
    }
    if (_stationInfo != NULL)
    {
        delete _stationInfo;
    }
    if (_lockKey != NULL)
    {
        free(_lockKey);
    }
}

bool Tuner::init(uint32_t tuner, Core *core)
{
    DebugLog2("Tuner::%s(%d, 0x%08x)\n", __FUNCTION__, tuner, (unsigned int)core);

    bool result = false;
    while ((tuner < MAX_TUNER_COUNT) && (core != NULL))
    {

        _tuner = tuner;
        _core = core;
        _core->retain();
        _core->setTuner(this, _tuner);

        // ok
        result = true;
        break;
    }
    return result;
}

void Tuner::addPacket(uint packet)
{
    uint packetCounter = BIT_SHIFT_MASK(packet, 26,  3);
    uint packetStart   = BIT_SHIFT_MASK(packet, 25,  1);
    uint packetData    = BIT_SHIFT_MASK(packet,  0, 24);

    // カウンタ値を確認
    uint count = BIT_SHIFT_MASK(_count, 0, 3);
    _count++;

    if (packetCounter != count)
    {
//        DebugLog3("カウンタ値が異常です。\n");
    }

    // パケット開始位置フラグを確認
    if (packetStart)
    {
        if (BIT_SHIFT_MASK(packetData, 15, 1))
        {
//            DebugLog3("リードソロモン復号で訂正しきれなかったエラーがあります。\n");
        }
        else
        {
            // 同期バイトを確認
            if (BIT_SHIFT_MASK(packetData, 16, 8) != 0x47)
            {
                DebugLog3("パケットの先頭バイトが 0x47 になっていません。\n");
            }
        }

        if (_packetOffset != 0)
        {
            //
            DebugLog3("前のパケットが完了しませんでした。\n");
        }
        _packetOffset = 0;
    }

    // データをコピー
    uint i;
    for (i = 0; i < 3; ++i)
    {
        if (_packetOffset < kPT1PacketSize)
        {
            _buffer[kPT1PacketSize * _packetCount + _packetOffset] = BIT_SHIFT_MASK(packetData, 8*(2-i), 8);
            _packetOffset++;
        }
    }
    if (kPT1PacketSize <= _packetOffset)
    {
        // ひとつのパケットが完成
        _packetOffset = 0;

        if (kPT1PageSize * kPT1PageCount / kPT1PacketSize <= ++_packetCount)
        {
            _packetCount = 0;

            uint8_t *ptr = _buffer;
            int32_t size = (kPT1PageSize * kPT1PageCount);


            //
            if (pthread_mutex_lock(&_lock) == 0)
            {
                if (_recfd >= 0)
                {
                    write(_recfd, ptr, size);
                }

                _analyzer->put(ptr, size);

                pthread_mutex_unlock(&_lock);
            }
        }
    }
}

#pragma mark #### for Tuner Control

bool Tuner::setChannel(int channel)
{
    return _core->setChannel(channel, _tuner);
}

int Tuner::channel()
{
    return _core->channel(_tuner);
}

#pragma mark ####

const char *Tuner::name()
{
    return _core->name(_tuner);
}

ISDB Tuner::isdb()
{
    return (_tuner % 2) == 0 ? ISDB_S : ISDB_T;
}

Dictionary *Tuner::stationInfo()
{
    DebugLog2("Tuner::stationInfo()\n");
    Dictionary *result = NULL;
    int count1 = 0;
#define COUNT1_MAX (6)
#define COUNT2_MAX (30)
    while (count1 < COUNT1_MAX)
    {
        if (pthread_mutex_lock(&_lock) == 0)
        {
            DebugLog2("Tuner::stationInfo() lock ok\n");
            int count2 = 0;
            _analyzer->setEnableSDT(true);
            while (_stationInfo == NULL)
            {
                DebugLog2("Tuner::stationInfo() _stationInfo == NULL\n");
                struct timespec sec = {time(NULL) + 1, 0};
                pthread_cond_timedwait(&_cond, &_lock, &sec);
                if (++count2 >= COUNT2_MAX)
                {
                    break;
                }
            }
            result = _stationInfo;
            _stationInfo = NULL;
            _analyzer->setEnableSDT(false);
            pthread_mutex_unlock(&_lock);
        }
        if (result != NULL)
        {
            Dictionary *self = result->dictionaryForKey(MPEG2_TS_SDT_SELF);
            if (self != NULL)
            {
                delete self;
                break;
            }
        }
        ++count1;
    }
    return result;
}

bool Tuner::setRecording(int fd)
{
    bool result = false;
    if (pthread_mutex_lock(&_lock) == 0)
    {
        if (_recfd < 0)
        {
    static int flag1 = 0;
    if (flag1 == 0)
    {
        flag1 = 1;
        printf("setRec: %d\n", fd);
    }
            _recfd = fd;
            result = true;
        }
        pthread_mutex_unlock(&_lock);
    }
    return result;
}

/*
bool Tuner::isRecording()
{
    bool result = false;
    if (pthread_mutex_lock(&_lock) == 0)
    {
        result = (_recfd >= 0);
        pthread_mutex_unlock(&_lock);
    }
    return result;
}
*/

bool Tuner::lock(char *key)
{
    bool result = false;
    if (_lockKey == NULL)
    {
        _lockKey = strdup(key);
        if (_lockKey != NULL)
        {
            result = true;
        }
    }
    return result;
}

bool Tuner::unlock(char *key)
{
    bool result = false;
    if (_lockKey != NULL)
    {
        if (strcmp(_lockKey, key) == 0)
        {
            free(_lockKey);
            _lockKey = NULL;
            result = true;
        }
    }
    return result;
}

bool Tuner::isLocked(char *key)
{
    bool result = false;
    if (_lockKey != NULL)
    {
        if (strcmp(_lockKey, key) == 0)
        {
            result = true;
        }
    }
    return result;
}

#pragma mark #### for Analyzer::Delegate

void Tuner::detect(MPEG2::TS::PAT *pat)
{
//    DebugLog2("Tuner::detect(PAT *)\n");
}

// called while locked
void Tuner::detect(MPEG2::TS::SDT *sdt)
{
    if (_stationInfo == NULL)
    {
        _stationInfo = new Dictionary();
        if (!_stationInfo->init(sdt->_dict))
        {
            delete _stationInfo;
            _stationInfo = NULL;
        }
    }
    pthread_cond_signal(&_cond);
}

void Tuner::packet(uint8_t packet_type, uint8_t *buf)
{
//    DebugLog2("Tuner::packet(uint8_t, uint8_t *)\n");
}

void Tuner::packet(uint8_t packet_type, uint16_t program_number, uint8_t *buf)
{
//    DebugLog2("Tuner::packet(uint8_t, uint16_t, uint8_t *)\n");
}

void Tuner::packet(uint8_t packet_type, uint16_t program_number, uint8_t stream_type, uint8_t *buf)
{
//    DebugLog2("Tuner::packet(uint8_t, uint16_t, uint8_t, uint8_t *)\n");
}

#pragma mark -
#pragma mark ################## class Tuner::Core ##################

Tuner::Core::Core()
{
#if 0
    pthread_mutexattr_t attr;
    pthread_mutexattr_init(&attr);
    pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init(&_lock, &attr);
    pthread_mutexattr_destroy(&attr);
#else
    pthread_mutex_init(&_lock, NULL);
#endif

    _pt1 = NULL;
    _retainCount = 1;
    _lnbPower = LNB_POWER_OFF;
    for (int i = 0; i < MAX_TUNER_COUNT; ++i)
    {
        _child[i] = NULL;
        _channel[i] = 0;
        _locked[i] = false;
        _name[i] = NULL;
    }
    _transfer = 0;
}

Tuner::Core::~Core()
{
    // wait run loop finish
    pthread_mutex_destroy(&_lock);

    for (int i = 0; i < MAX_TUNER_COUNT; ++i)
    {
        if (_name[i] != NULL)
        {
            free(_name[i]);
        }
    }
}

bool Tuner::Core::init(PT1 *pt1)
{
    bool result = false;
    while (pt1 != NULL)
    {
        if (pt1->open() != kStatusSuccess)
        {
            DebugLog3("PT1: open() ng.\n");
            break;
        }

        if (pt1->setTunerPowerReset(kTunerPowerOnResetEnable) != kStatusSuccess)
        {
            DebugLog3("PT1: setTunerPowerReset(kTunerPowerOnResetEnable) ng.\n");
            pt1->close();
            break;
        }
        usleep(20 * 1000);

        if (pt1->setTunerPowerReset(kTunerPowerOnResetDisable) != kStatusSuccess)
        {
            DebugLog3("PT1: setTunerPowerReset(kTunerPowerOnResetDisable) ng.\n");
            pt1->close();
            break;
        }
        usleep(1 * 1000);

        uint tuner;
        for (tuner = 0; tuner < 2; ++tuner)
        {
            if (pt1->initTuner(tuner) != kStatusSuccess)
            {
                DebugLog3("PT1: initTuner(%d) ng.\n", tuner);
                break;
            }
        }
        if (tuner < 2)
        {
            DebugLog3("Error: initTuner()\n");
            pt1->close();
            break;
        }

        for (tuner = 0; tuner < 2; ++tuner)
        {
            int isdb;
            for (isdb = ISDB_S; isdb < ISDB_COUNT; isdb = isdb + 1)
            {
                if (pt1->setTunerSleep(tuner, (ISDB)isdb, false) != kStatusSuccess)
                {
                    DebugLog3("PT1: setTunerSleep(%d, %d, %d)\n", tuner, isdb, false);
                    break;
                }
                if (pt1->setStreamEnable(tuner, (ISDB)isdb, true) != kStatusSuccess)
                {
                    DebugLog3("PT1: setStreamEnable(%d, %d, %d)\n", tuner, isdb, true);
                    break;
                }
            }
            if (isdb < 2)
            {
                break;
            }
        }
        if (tuner < 2)
        {
            DebugLog3("Error: setTunerSleep() or setStreamEnable()\n");
            pt1->close();
            break;
        }

        if (pthread_mutex_lock(&_lock) == 0)
        {
            _pt1 = pt1;
            pthread_t pid;
            if (pthread_create(&pid, NULL, Tuner_Core_run, this) == 0)
            {
                pthread_detach(pid);
                _transfer = 1;
            }

            pthread_mutex_unlock(&_lock);
        }
        else
        {
            DebugLog3("lock ng.\n");
            pt1->close();
            break;
        }

        // ok
        result = true;
        break;
    }
    return result;
}

void Tuner::Core::setTuner(Tuner *t, uint32_t tuner)
{
    if ((t != NULL) && (tuner < MAX_TUNER_COUNT))
    {
        _child[tuner] = t;
    }
}

void Tuner::Core::retain()
{
    _retainCount++;
}

void Tuner::Core::release()
{
    _retainCount--;
    if (_retainCount == 0)
    {
        delete this;
    }
}

bool Tuner::Core::setChannel(uint32_t channel, uint32_t tuner)
{
    bool result = false;

    if (pthread_mutex_lock(&_lock) == 0)
    {
        if (tuner >= 0 && tuner < 4)
        {
            if ((tuner % 2) == 0)
            {
                // ISDB-S
                while ((0 <= channel) && (channel <= TunerMaxChannel_ISDB_S))
                {
                    TmccS tmcc;
                    bool tmccIsValid = false;

                    // 周波数設定
                    if (_pt1->setFrequency((tuner / 2), ISDB_S, channel, 0) != kStatusSuccess)
                    {
                        DebugLog3("PT1: setFrequency NG.\n");
                        break;
                    }

                    for (int i = 0; i < 6; ++i)
                    {
                        if (_pt1->getTmccS((tuner / 2), &tmcc) == kStatusSuccess)
                        {
                            tmccIsValid = true;
                            break;
                        }
                        usleep((50)*1000);
                    }

                    result = tmccIsValid;
                    break;
                }
            }
            else
            {
                // ISDB-T
                while ((0 <= channel) && (channel <= TunerMaxChannel_ISDB_T))
                {
                    TmccT tmcc;
                    bool tmccIsValid = false;

                    if (_pt1->setFrequency((tuner / 2), ISDB_T, channel, 0) != kStatusSuccess)
                    {
                        DebugLog3("PT1: setFrequency NG.\n");
                        break;
                    }

                    for (int i = 0; i < 2; ++i)
                    {
                        if (_pt1->getTmccT((tuner / 2), &tmcc) == kStatusSuccess)
                        {
                            tmccIsValid = true;
                            break;
                        }
                    }

                    result = tmccIsValid;
                    break;
                }
            }
            _channel[tuner] = channel;
            _locked[tuner] = result;
        }

        pthread_mutex_unlock(&_lock);
    }

    return result;
}

uint32_t Tuner::Core::channel(uint32_t tuner)
{
    return (tuner < MAX_TUNER_COUNT) ? _channel[tuner] : -1;
}

const char *Tuner::Core::name(uint32_t tuner)
{
    char *result = NULL;
    if (tuner < MAX_TUNER_COUNT)
    {
        if (_name[tuner] == NULL)
        {
            if (pthread_mutex_lock(&_lock) == 0)
            {
                DeviceInfo devInfo;
                if (_pt1->getDeviceInfo(&devInfo))
                {
                    char tmp[64];
                    sprintf(tmp, "PT1@%02d%02d%02d_%d", devInfo.Bus, devInfo.Slot, devInfo.Function, tuner);
                    _name[tuner] = strdup(tmp);
                }
                pthread_mutex_unlock(&_lock);
            }
        }
        result = _name[tuner];
    }
    return result;
}

#pragma mark #### for DMA Transfer

void Tuner::Core::run()
{
    DebugLog2("Tuner::Core::run()\n");
    int result;

    while (true)
    {
        if (pthread_mutex_lock(&_lock) == 0)
        {
            // バッファが確保されていない場合は確保する
            BufferInfo bufferInfo;
            result = _pt1->getBufferInfo(&bufferInfo);
            if (result != kStatusSuccess)
            {
                DebugLog3("%s getBufferInfo ng. %d\n", __FUNCTION__, result);
                break;
            }

            if (bufferInfo.VirtualSize == 0)
            {
                bufferInfo.VirtualSize  = kPT1VirtualSize;
                bufferInfo.VirtualCount = kPT1VirtualCount;
                bufferInfo.LockSize     = kPT1LockSize;
                result = _pt1->setBufferInfo(&bufferInfo);
                if (result != kStatusSuccess)
                {
                    DebugLog3("%s setBufferInfo ng. %d\n", __FUNCTION__, result);
                    break;
                }
            }
            pthread_mutex_unlock(&_lock);
        }
        else
        {
            DebugLog3("error.\n");
            break;
        }


        while (_transfer == 1)
        {
            if (pthread_mutex_lock(&_lock) == 0)
            {
                // DMA 転送がどこまで進んだかを調べるため、各ブロックの末尾を 0 でクリアする
                for (int i = 0; i < kPT1VirtualCount; ++i)
                {
                    for (int j = 0; j < kPT1VirtualSize; ++j)
                    {
                        for (int k = 0; k < kPT1BlockCount; ++k)
                        {
                            clearBlock(i, j, k);
                        }
                    }
                }

                // 転送カウンタをリセットする
                result = _pt1->resetTransferCounter();
                if (result != kStatusSuccess)
                {
                    DebugLog3("%s resetTransferCounter ng. %d\n", __FUNCTION__, result);
                    pthread_mutex_unlock(&_lock);
                    break;
                }

                // 転送カウンタをインクリメントする
                bool flag = false;
                for (int i = 0; i < kPT1VirtualSize * kPT1VirtualCount; ++i)
                {
                    result = _pt1->incrementTransferCounter();
                    if (result != kStatusSuccess)
                    {
                        DebugLog3("%s incrementTransferCounter ng. %d\n", __FUNCTION__, result);
                        flag = true;
                        break;
                    }
                }
                if (flag)
                {
                    pthread_mutex_unlock(&_lock);
                    break;
                }

                // DMA 転送を許可する
                result = _pt1->setTransferEnable(true);
                if (result != kStatusSuccess)
                {
                    DebugLog3("%s setTransferEnable ng. %d\n", __FUNCTION__, result);
                    pthread_mutex_unlock(&_lock);
                    break;
                }
                pthread_mutex_unlock(&_lock);
            }
            else
            {
                DebugLog3("error.\n");
                break;
            }


            _virtualIndex = 0;
            _imageIndex = 0;
            _blockIndex = 0;

            DebugLog2("transfer start.\n");
            while (true)
            {
                if (!waitBlock())
                {
                    break;
                }

                if (pthread_mutex_lock(&_lock) == 0)
                {
                    copyBlock();

                    if (!dispatchBlock())
                    {
                        pthread_mutex_unlock(&_lock);
                        break;
                    }
                    pthread_mutex_unlock(&_lock);
                }
                else
                {
                    DebugLog3("error.\n");
                    break;
                }
            }
            DebugLog2("transfer stop.\n");

            result = _pt1->setTransferEnable(false);
            if (result != kStatusSuccess)
            {
                DebugLog3("%s setTransferEnable(done) ng. %d\n", __FUNCTION__, result);
                break;
            }
        }
        _transfer = 0;
        break;
    }

    _pt1->close();
}

uint Tuner::Core::offset(uint imageIndex, uint blockIndex, uint additionalOffset)
{
    blockIndex += kPT1BlockCount * imageIndex;
    uint offset = (kPT1BlockSize * blockIndex + additionalOffset) / sizeof(uint);
    return offset;
}

uint Tuner::Core::offset(uint imageIndex, uint blockIndex)
{
    return offset(imageIndex, blockIndex, (kPT1BlockSize - sizeof(uint)));
}

uint Tuner::Core::read(uint virtualIndex, uint imageIndex, uint blockIndex)
{
    void *voidPtr;
    if (pthread_mutex_lock(&_lock) == 0)
    {
        if (_pt1->getBufferPtr(virtualIndex, &voidPtr) != kStatusSuccess)
        {
            DebugLog3("PT1: getBufferPtr(read) ng.\n");
            pthread_mutex_unlock(&_lock);
            return 0;
        }
        pthread_mutex_unlock(&_lock);
    }
    else
    {
        DebugLog3("error.\n");
        return 0;
    }

    volatile const uint *ptr = (volatile const uint *)voidPtr;
    if (ptr != NULL)
    {
        return ptr[offset(imageIndex, blockIndex)];
    }
    else
    {
        DebugLog3("ptr is NULL\n");
    }
    return 0;
}

void Tuner::Core::clearBlock(uint virtualIndex, uint imageIndex, uint blockIndex)
{
    void *voidPtr = NULL;

    if (_pt1->getBufferPtr(virtualIndex, &voidPtr) != kStatusSuccess)
    {
        DebugLog3("getBufferPtr(clearBlock) ng.\n");
        return;
    }

    uint *ptr = (uint *)voidPtr;
    if (ptr != NULL)
    {
        ptr[offset(imageIndex, blockIndex)] = 0;
    }
    else
    {
        DebugLog3("ptr is NULL");
    }
}

bool Tuner::Core::waitBlock()
{
    bool result = true;
    while (true)
    {
        if (_transfer != 1)
        {
            result = false;
            break;
        }

        // ブロックの末尾が 0 でなければ、そのブロックの DMA 転送が完了したことになる
        if (read(_virtualIndex, _imageIndex, _blockIndex) != 0)
        {
            break;
        }

        usleep(10 * 1000);  // 10ms
    }

    return result;
}

void Tuner::Core::copyBlock()
{
    void *voidPtr;
    if (_pt1->getBufferPtr(_virtualIndex, &voidPtr) != kStatusSuccess)
    {
        DebugLog3("getBufferPtr(copyBlock) ng.\n");
        return;
    }

    uint *srcPtr = (uint *)voidPtr;

    srcPtr += offset(_imageIndex, _blockIndex, 0);

    memcpy(_buffer, srcPtr, kPT1BlockSize);

    // コピーし終わったので、ブロックの末尾を 0 にします。
    srcPtr[kPT1BlockSize / sizeof(*srcPtr) - 1] = 0;

    if (kPT1BlockCount <= ++_blockIndex)
    {
        _blockIndex = 0;

        // 転送カウンタは OS::Memory::PAGE_SIZE * PT::Device::BUFFER_PAGE_COUNT バイトごとにインクリメントします。
        if (_pt1->incrementTransferCounter() != kStatusSuccess)
        {
            DebugLog3("incrementTransferCounter ng.\n");
            return;
        }

        if (kPT1VirtualSize <= ++_imageIndex)
        {
            _imageIndex = 0;
            if (kPT1VirtualCount <= ++_virtualIndex)
            {
                _virtualIndex = 0;
            }
        }
    }
}

bool Tuner::Core::dispatchBlock()
{
    const uint *ptr = (uint *)_buffer;

    uint i;
    for (i = 0; i < kPT1BlockSize / sizeof(*ptr); ++i)
    {
        uint packet = ptr[i];

        uint packetId    = BIT_SHIFT_MASK(packet, 29, 3);
        uint packetError = BIT_SHIFT_MASK(packet, 24, 1);

        if (packetError)
        {
            // エラーの原因を調べる
            TransferInfo info;
            if (_pt1->getTransferInfo(&info) != kStatusSuccess)
            {
                DebugLog3("getTransferInfo ng.\n");
                return false;
            }

            if (info.TransferCounter1)
            {
                DebugLog3("★転送カウンタが 1 以下になりました。\n");
            }
            else if (info.BufferOverflow)
            {
                DebugLog3("★PCI バスを長期に渡り確保できなかったため、ボード上の FIFO が溢れました。\n");
            }
            else
            {
                DebugLog3("★転送エラーが発生しました。\n");
            }
            DebugLog3("★ファイルへの書き出しを中止します。\n");
            return false;
        }
        if ((1 <= packetId) && (packetId <= 4) && (_locked[packetId - 1]))
        {
    static int flag1 = 0;
    if (flag1 == 0)
    {
        flag1 = 1;
        printf("befor addPacket\n");
    }
            _child[packetId - 1]->addPacket(packet);
        }
    }

    return true;
}


#pragma mark -
#pragma mark ################## class Tuner::DummyCore ##################

} // PTxD
} // PTx
