/*
 *  PT1.cpp
 */
#include <stdio.h>
#include "PTx/PT1.h"

namespace PTx
{


/**
 * @brief スキャン
 */
int PT1::scan(PT1 *pt1s[MAX_DEV_COUNT])
{
    int result = -1;
    io_iterator_t   iterator;

    // PT1Deviceを取得
    if (IOServiceGetMatchingServices(kIOMasterPortDefault, IOServiceMatching(kPT1Device), &iterator) == KERN_SUCCESS)
    {
        result = 0;

        io_service_t service;
        while ((service = IOIteratorNext(iterator)) != IO_OBJECT_NULL)
        {
            // Tunerインスタンス生成
            PT1 *pt1 = new PT1();
            if (pt1 != NULL)
            {
                if (pt1->init(service))
                {
                    pt1s[result++] = pt1;
                }
                else
                {
                    delete pt1;
                }
            }
        }

        // Release the io_iterator_t now that we're done with it.
        IOObjectRelease(iterator);
    }

    return result;
}

PT1::PT1()
{
    _connect = (io_connect_t)NULL;
    _opened = false;
}

/**
 * @brief インスタンスを解放
 */
PT1::~PT1()
{
    printf("%s\n", __FUNCTION__);
    this->close();
}

bool PT1::init(io_service_t service)
{
    printf("%s\n", __FUNCTION__);

    _buffer  = (vm_address_t)NULL;
    _bufsize = 0;
    _bufferInfo.VirtualSize = 0;
    _bufferInfo.VirtualCount = 0;
    _bufferInfo.LockSize = 0;
    _opened = false;

    kern_return_t result = IOServiceOpen(service, mach_task_self(), 0, &_connect);
    return (result == KERN_SUCCESS);
}

bool PT1::getDeviceInfo(DeviceInfo *info)
{
    kern_return_t result = KERN_FAILURE;

    bool open = false;
    if (!_opened)
    {
        open = (this->open() == KERN_SUCCESS);
    }
    if (_opened)
    {
        size_t size = sizeof(DeviceInfo);
        result = IOConnectCallStructMethod(_connect,             // an io_connect_t returned from IOServiceOpen().
                                           kMethodGetDeviceInfo, // selector of the function to be called via the user client.
                                           NULL,                 // pointer to the input struct parameter.
                                           0,                    // the size of the input structure parameter.
                                           info,                 // pointer to the output struct parameter.
                                           &size);               // pointer to the size of the output structure parameter.
    }
    if (open)
    {
        this->close();
    }

    return (result == KERN_SUCCESS);
}


/**
 * @brief デバイスのオープン
 *
 * デバイスのオープンは以下の手順に沿って行われます。
 *   1. 既にデバイスがオープンされていないかを確認する。
 *   2. リビジョンID (コンフィギュレーション空間 アドレス 0x08) が 0x01 であるかを調べる。
 *   3. コンフィギュレーション空間のデバイス固有レジスタ領域を使って PCI バスでのビット化けがないかを確認する。
 *   4. この SDK で制御が可能な FPGA 回路のバージョンであるかを確認する。
 */
int PT1::open()
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;

    if (!_opened)
    {
        kr = IOConnectCallScalarMethod(_connect, kMethodOpen, NULL, 0, &scalarO_64, &outputCount);
        if (kr == KERN_SUCCESS)
        {
            _opened = true;
            result = (uint32_t)scalarO_64;
        }
    }
    return result;
}

/**
 * @brief デバイスのクローズ
 */
int PT1::close()
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64;
    uint32_t outputCount = 1; 
    kern_return_t kr;

    if (_opened)
    {
        kr = IOConnectCallScalarMethod(_connect, kMethodClose, NULL, 0, &scalarO_64, &outputCount);
        if (kr == KERN_SUCCESS)
        {
            _opened = false;
            result = (uint32_t)scalarO_64;
        }
    }
    return result;
}

/**
 * @brief PCI クロックカウンタを取得
 *
 * カウンタ長は 32 ビットです。0xffffffff の次は 0 になります。
 */
int PT1::getPciClockCounter(uint32_t *counter)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetPciClockCounter,   // selector of the function to be called via the user client.
                                   NULL,                        // array of scalar (64-bit) input values.
                                   0,                           // the number of scalar input values.
                                   scalarO_64,                  // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *counter = (uint32_t)scalarO_64[1];
    }
    return result;
}

/**
 * @brief PCI レイテンシタイマ値の設定
 */
int PT1::setPciLatencyTimer(uint8_t latencyTimer)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = latencyTimer;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetPciLatencyTimer,   // selector of the function to be called via the user client.
                                   &scalarI_64,                 // array of scalar (64-bit) input values.
                                   1,                           // the number of scalar input values.
                                   &scalarO_64,                 // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief PCI レイテンシタイマ値の取得
 *
 * 下位 3 ビットは実装されていないため、取得した値は 8 の倍数になります。
 */
int PT1::getPciLatencyTimer(uint8_t *latencyTimer)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetPciLatencyTimer,   // selector of the function to be called via the user client.
                                   NULL,                        // array of scalar (64-bit) input values.
                                   0,                           // the number of scalar input values.
                                   scalarO_64,                  // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *latencyTimer = (uint8_t)scalarO_64[1];
    }
    return result;
}

/**
 * @brief LNB 電源制御 設定
 *
 * チューナーの電源とは独立に制御可能です。デフォルト値は LNB_POWER_OFF です。
 */
int PT1::setLnbPower(LnbPower lnbPower)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = lnbPower;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,           // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetLnbPower, // selector of the function to be called via the user client.
                                   &scalarI_64,        // array of scalar (64-bit) input values.
                                   1,                  // the number of scalar input values.
                                   &scalarO_64,        // array of scalar (64-bit) output values.
                                   &outputCount);      // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief LNB 電源制御 取得
 */
int PT1::getLnbPower(LnbPower *lnbPower)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,           // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetLnbPower, // selector of the function to be called via the user client.
                                   NULL,               // array of scalar (64-bit) input values.
                                   0,                  // the number of scalar input values.
                                   scalarO_64,         // array of scalar (64-bit) output values.
                                   &outputCount);      // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *lnbPower = (LnbPower)scalarO_64[1];
    }
    return result;
}

/**
 * @brief デバイスをクローズ（異常終了にともなうクローズを含む）時の LNB 電源制御 設定
 *
 * デフォルト値は LNB_POWER_OFF です。
 */
int PT1::setLnbPowerWhenClose(LnbPower lnbPower)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = lnbPower;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetLnbPowerWhenClose, // selector of the function to be called via the user client.
                                   &scalarI_64,                 // array of scalar (64-bit) input values.
                                   1,                           // the number of scalar input values.
                                   &scalarO_64,                 // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief デバイスをクローズ（異常終了にともなうクローズを含む）時の LNB 電源制御 取得
 */
int PT1::getLnbPowerWhenClose(LnbPower *lnbPower)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetLnbPowerWhenClose, // selector of the function to be called via the user client.
                                   NULL,                        // array of scalar (64-bit) input values.
                                   0,                           // the number of scalar input values.
                                   scalarO_64,                  // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *lnbPower = (LnbPower)scalarO_64[1];
    }
    return result;
}

/**
 * @brief チューナー電源・ハードウェアリセット制御 設定
 *
 * TUNER_POWER_ON_RESET_ENABLE から TUNER_POWER_ON_RESET_DISABLE の遷移には最低 15ms の待ち時間が必要です。
 */
int PT1::setTunerPowerReset(TunerPowerReset tunerPowerReset)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = tunerPowerReset;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetTunerPowerReset,   // selector of the function to be called via the user client.
                                   &scalarI_64,                 // array of scalar (64-bit) input values.
                                   1,                           // the number of scalar input values.
                                   &scalarO_64,                 // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief チューナー電源・ハードウェアリセット制御 取得
 */
int PT1::getTunerPowerReset(TunerPowerReset *tunerPowerReset)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetTunerPowerReset,   // selector of the function to be called via the user client.
                                   NULL,                        // array of scalar (64-bit) input values.
                                   0,                           // the number of scalar input values.
                                   scalarO_64,                  // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *tunerPowerReset = (TunerPowerReset)scalarO_64[1];
    }
    return result;
}


/**
 * @brief チューナー初期化
 *
 * SetTunerPowerReset(TUNER_POWER_ON_RESET_DISABLE) から最低 1μs 経過後に 1 回だけ呼び出します。 
 */
int PT1::initTuner(uint32_t tuner)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = tuner;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,            // an io_connect_t returned from IOServiceOpen().
                                   kMethodInitTuner,    // selector of the function to be called via the user client.
                                   &scalarI_64,         // array of scalar (64-bit) input values.
                                   1,                   // the number of scalar input values.
                                   &scalarO_64,         // array of scalar (64-bit) output values.
                                   &outputCount);       // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief チューナー省電力制御 設定
 *
 * 復調IC と衛星側 PLL-IC のみが対象です。チューナーユニット内の他の回路は SetTunerPowerReset(TUNER_POWER_OFF)
 * としない限り、電力を消費し続けます。復調IC の消費電力はチューナーモジュールの 15% です。
 */
int PT1::setTunerSleep(uint32_t tuner, ISDB isdb, bool sleep)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[3];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    scalarI_64[2] = sleep;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetTunerSleep,    // selector of the function to be called via the user client.
                                   scalarI_64,              // array of scalar (64-bit) input values.
                                   3,                       // the number of scalar input values.
                                   &scalarO_64,             // array of scalar (64-bit) output values.
                                   &outputCount);           // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief チューナー省電力制御 取得
 */
int PT1::getTunerSleep(uint32_t tuner, ISDB isdb, bool *sleep)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetTunerSleep,    // selector of the function to be called via the user client.
                                   scalarI_64,              // array of scalar (64-bit) input values.
                                   2,                       // the number of scalar input values.
                                   scalarO_64,              // array of scalar (64-bit) output values.
                                   &outputCount);           // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *sleep = (bool)scalarO_64[1];
    }
    return result;
}

/**
 * @brief 局発周波数の制御 設定
 *
 * offset で周波数の調整が可能です。単位は ISDB-S の場合は 1MHz、ISDB-T の場合は 1/7MHz です。
 * 例えば、C24 を標準より 2MHz 高い周波数に設定するには SetFrequency(tuner, ISDB_T, 23, 7*2) とします。
 *
 * (ISDB-S)
 * PLL 周波数ステップが 1MHz のため、実際に設定される周波数は f' になります。
 * +----+------+---------+---------+ +----+------+---------+---------+ +----+------+---------+---------+
 * | ch | TP # | f (MHz) | f'(MHz) | | ch | TP # | f (MHz) | f'(MHz) | | ch | TP # | f (MHz) | f'(MHz) |
 * +----+------+---------+---------+ +----+------+---------+---------+ +----+------+---------+---------+
 * |  0 | BS 1 | 1049.48 | 1049.00 | | 12 | ND 2 | 1613.00 | <--     | | 24 | ND 1 | 1593.00 | <--     |
 * |  1 | BS 3 | 1087.84 | 1088.00 | | 13 | ND 4 | 1653.00 | <--     | | 25 | ND 3 | 1633.00 | <--     |
 * |  2 | BS 5 | 1126.20 | 1126.00 | | 14 | ND 6 | 1693.00 | <--     | | 26 | ND 5 | 1673.00 | <--     |
 * |  3 | BS 7 | 1164.56 | 1165.00 | | 15 | ND 8 | 1733.00 | <--     | | 27 | ND 7 | 1713.00 | <--     |
 * |  4 | BS 9 | 1202.92 | 1203.00 | | 16 | ND10 | 1773.00 | <--     | | 28 | ND 9 | 1753.00 | <--     |
 * |  5 | BS11 | 1241.28 | 1241.00 | | 17 | ND12 | 1813.00 | <--     | | 29 | ND11 | 1793.00 | <--     |
 * |  6 | BS13 | 1279.64 | 1280.00 | | 18 | ND14 | 1853.00 | <--     | | 30 | ND13 | 1833.00 | <--     |
 * |  7 | BS15 | 1318.00 | <--     | | 19 | ND16 | 1893.00 | <--     | | 31 | ND15 | 1873.00 | <--     |
 * |  8 | BS17 | 1356.36 | 1356.00 | | 20 | ND18 | 1933.00 | <--     | | 32 | ND17 | 1913.00 | <--     |
 * |  9 | BS19 | 1394.72 | 1395.00 | | 21 | ND20 | 1973.00 | <--     | | 33 | ND19 | 1953.00 | <--     |
 * | 10 | BS21 | 1433.08 | 1433.00 | | 22 | ND22 | 2013.00 | <--     | | 34 | ND21 | 1993.00 | <--     |
 * | 11 | BS23 | 1471.44 | 1471.00 | | 23 | ND24 | 2053.00 | <--     | | 35 | ND23 | 2033.00 | <--     |
 * +----+------+---------+---------+ +----+------+---------+---------+ +----+------+---------+---------+
 *
 * (ISDB-T)
 * +-----+-----+---------+ +-----+-----+---------+ +-----+-----+---------+ +-----+-----+---------+ +-----+-----+---------+
 * | ch. | Ch. | f (MHz) | | ch. | Ch. | f (MHz) | | ch. | Ch. | f (MHz) | | ch. | Ch. | f (MHz) | | ch. | Ch. | f (MHz) |
 * +-----+-----+---------+ +-----+-----+---------+ +-----+-----+---------+ +-----+-----+---------+ +-----+-----+---------+
 * |   0 |   1 |  93+1/7 | |  23 | C24 | 231+1/7 | |  46 | C47 | 369+1/7 | |  69 |  19 | 509+1/7 | |  92 |  42 | 647+1/7 |
 * |   1 |   2 |  99+1/7 | |  24 | C25 | 237+1/7 | |  47 | C48 | 375+1/7 | |  70 |  20 | 515+1/7 | |  93 |  43 | 653+1/7 |
 * |   2 |   3 | 105+1/7 | |  25 | C26 | 243+1/7 | |  48 | C49 | 381+1/7 | |  71 |  21 | 521+1/7 | |  94 |  44 | 659+1/7 |
 * |   3 | C13 | 111+1/7 | |  26 | C27 | 249+1/7 | |  49 | C50 | 387+1/7 | |  72 |  22 | 527+1/7 | |  95 |  45 | 665+1/7 |
 * |   4 | C14 | 117+1/7 | |  27 | C28 | 255+1/7 | |  50 | C51 | 393+1/7 | |  73 |  23 | 533+1/7 | |  96 |  46 | 671+1/7 |
 * |   5 | C15 | 123+1/7 | |  28 | C29 | 261+1/7 | |  51 | C52 | 399+1/7 | |  74 |  24 | 539+1/7 | |  97 |  47 | 677+1/7 |
 * |   6 | C16 | 129+1/7 | |  29 | C30 | 267+1/7 | |  52 | C53 | 405+1/7 | |  75 |  25 | 545+1/7 | |  98 |  48 | 683+1/7 |
 * |   7 | C17 | 135+1/7 | |  30 | C31 | 273+1/7 | |  53 | C54 | 411+1/7 | |  76 |  26 | 551+1/7 | |  99 |  49 | 689+1/7 |
 * |   8 | C18 | 141+1/7 | |  31 | C32 | 279+1/7 | |  54 | C55 | 417+1/7 | |  77 |  27 | 557+1/7 | | 100 |  50 | 695+1/7 |
 * |   9 | C19 | 147+1/7 | |  32 | C33 | 285+1/7 | |  55 | C56 | 423+1/7 | |  78 |  28 | 563+1/7 | | 101 |  51 | 701+1/7 |
 * |  10 | C20 | 153+1/7 | |  33 | C34 | 291+1/7 | |  56 | C57 | 429+1/7 | |  79 |  29 | 569+1/7 | | 102 |  52 | 707+1/7 |
 * |  11 | C21 | 159+1/7 | |  34 | C35 | 297+1/7 | |  57 | C58 | 435+1/7 | |  80 |  30 | 575+1/7 | | 103 |  53 | 713+1/7 |
 * |  12 | C22 | 167+1/7 | |  35 | C36 | 303+1/7 | |  58 | C59 | 441+1/7 | |  81 |  31 | 581+1/7 | | 104 |  54 | 719+1/7 |
 * |  13 |   4 | 173+1/7 | |  36 | C37 | 309+1/7 | |  59 | C60 | 447+1/7 | |  82 |  32 | 587+1/7 | | 105 |  55 | 725+1/7 |
 * |  14 |   5 | 179+1/7 | |  37 | C38 | 315+1/7 | |  60 | C61 | 453+1/7 | |  83 |  33 | 593+1/7 | | 106 |  56 | 731+1/7 |
 * |  15 |   6 | 185+1/7 | |  38 | C39 | 321+1/7 | |  61 | C62 | 459+1/7 | |  84 |  34 | 599+1/7 | | 107 |  57 | 737+1/7 |
 * |  16 |   7 | 191+1/7 | |  39 | C40 | 327+1/7 | |  62 | C63 | 465+1/7 | |  85 |  35 | 605+1/7 | | 108 |  58 | 743+1/7 |
 * |  17 |   8 | 195+1/7 | |  40 | C41 | 333+1/7 | |  63 |  13 | 473+1/7 | |  86 |  36 | 611+1/7 | | 109 |  59 | 749+1/7 |
 * |  18 |   9 | 201+1/7 | |  41 | C42 | 339+1/7 | |  64 |  14 | 479+1/7 | |  87 |  37 | 617+1/7 | | 110 |  60 | 755+1/7 |
 * |  19 |  10 | 207+1/7 | |  42 | C43 | 345+1/7 | |  65 |  15 | 485+1/7 | |  88 |  38 | 623+1/7 | | 111 |  61 | 761+1/7 |
 * |  20 |  11 | 213+1/7 | |  43 | C44 | 351+1/7 | |  66 |  16 | 491+1/7 | |  89 |  39 | 629+1/7 | | 112 |  62 | 767+1/7 |
 * |  21 |  12 | 219+1/7 | |  44 | C45 | 357+1/7 | |  67 |  17 | 497+1/7 | |  90 |  40 | 635+1/7 | +-----+-----+---------+
 * |  22 | C23 | 225+1/7 | |  45 | C46 | 363+1/7 | |  68 |  18 | 503+1/7 | |  91 |  41 | 641+1/7 |
 * +-----+-----+---------+ +-----+-----+---------+ +-----+-----+---------+ +-----+-----+---------+
 * 
 * C24〜C27 は、ケーブルテレビ局により下記の周波数で送信されている場合があります。
 * +-----+---------+
 * | Ch. | f (MHz) |
 * +-----+---------+
 * | C24 | 233+1/7 |
 * | C25 | 239+1/7 |
 * | C26 | 245+1/7 |
 * | C27 | 251+1/7 |
 * +-----+---------+
 */
int PT1::setFrequency(uint32_t tuner, ISDB isdb, uint32_t channel, int32_t offset)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[4];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    scalarI_64[2] = channel;
    scalarI_64[3] = offset;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,            // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetFrequency, // selector of the function to be called via the user client.
                                   scalarI_64,          // array of scalar (64-bit) input values.
                                   4,                   // the number of scalar input values.
                                   &scalarO_64,         // array of scalar (64-bit) output values.
                                   &outputCount);       // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief 局発周波数の制御 取得
 */
int PT1::getFrequency(uint32_t tuner, ISDB isdb, uint32_t *channel, int32_t *offset)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    uint64_t scalarO_64[3];
    uint32_t outputCount = 3; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,            // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetFrequency, // selector of the function to be called via the user client.
                                   scalarI_64,          // array of scalar (64-bit) input values.
                                   2,                   // the number of scalar input values.
                                   scalarO_64,          // array of scalar (64-bit) output values.
                                   &outputCount);       // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *channel = (uint32_t)scalarO_64[1];
        *offset = (uint32_t)scalarO_64[2];
    }
    return result;
}

/**
 * @brief 周波数誤差を取得
 *
 * 値の意味は次の通りです。
 *   クロック周波数誤差: clock/100 (ppm)
 *   キャリア周波数誤差: carrier (Hz)
 *   放送波の周波数精度は十分に高い仮定すると、誤差が発生する要素として以下のようなものが考えられます。
 *   (ISDB-S) LNB での周波数変換精度 / 衛星側 PLL-IC に接続されている振動子の精度 / 復調 IC に接続されている振動子の精度
 *   (ISDB-T) 地上側 PLL-IC に接続されている振動子の精度 / 復調 IC に接続されている振動子の精度
 */
int PT1::getFrequencyOffset(uint32_t tuner, ISDB isdb, int32_t *clock, int32_t *carrier)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    uint64_t scalarO_64[3];
    uint32_t outputCount = 3; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetFrequencyOffset,   // selector of the function to be called via the user client.
                                   scalarI_64,                  // array of scalar (64-bit) input values.
                                   2,                           // the number of scalar input values.
                                   scalarO_64,                  // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *clock = (uint32_t)scalarO_64[1];
        *carrier = (uint32_t)scalarO_64[2];
    }
    return result;
}

/**
 * @brief C/N と AGC を取得
 *
 * C/N は短レイテンシで測定できるため、アンテナの向きを調整するのに便利です。
 *   値の意味は次の通りです。
 *   C/N 　           : cn100/100 (dB)
 *   現在の AGC 値     : currentAgc
 *   利得最大時の AGC 値: maxAgc
 *   currentAgc の範囲は 0 から maxAgc までです。
 */
int PT1::getCnAgc(uint32_t tuner, ISDB isdb, uint32_t *cn100, uint32_t *currentAgc, uint32_t *maxAgc)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    uint64_t scalarO_64[4];
    uint32_t outputCount = 4; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,        // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetCnAgc, // selector of the function to be called via the user client.
                                   scalarI_64,      // array of scalar (64-bit) input values.
                                   2,               // the number of scalar input values.
                                   scalarO_64,      // array of scalar (64-bit) output values.
                                   &outputCount);   // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *cn100 = (uint32_t)scalarO_64[1];
        *currentAgc = (uint32_t)scalarO_64[2];
        *maxAgc = (uint32_t)scalarO_64[3];
    }
    return result;
}

/**
 * @brief TS-ID を設定
 *
 * 設定値が復調IC の動作に反映されるまで時間が掛かります。
 * GetLayerS() を呼び出す前に、GetIdS() を使って切り替えが完了したことを確認してください。
 */
int PT1::setIdS(uint32_t tuner, uint32_t tsid)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = tsid;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,        // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetIdS,   // selector of the function to be called via the user client.
                                   scalarI_64,      // array of scalar (64-bit) input values.
                                   2,               // the number of scalar input values.
                                   &scalarO_64,     // array of scalar (64-bit) output values.
                                   &outputCount);   // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief 現在処理中の TS-ID を取得
 *
 * GetLayerS() で取得できるレイヤ情報は、この関数で示される TS-ID のものになります。
 */
int PT1::getIdS(uint32_t tuner, uint32_t *tsid)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = tuner;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,        // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetIdS,   // selector of the function to be called via the user client.
                                   &scalarI_64,     // array of scalar (64-bit) input values.
                                   1,               // the number of scalar input values.
                                   scalarO_64,      // array of scalar (64-bit) output values.
                                   &outputCount);   // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *tsid = (uint32_t)scalarO_64[1];
    }
    return result;
}

/**
 * @brief リードソロモン復号で訂正されたエラーレートを取得
 *
 * 測定に時間が掛かりますが、受信品質を正確に把握するには C/N ではなくこのエラーレートを参考にしてください。
 * ひとつの目安として 2×10^-4 以下であれば、リードソロモン復号後にほぼエラーフリーになるといわれています。
 * エラーレートの集計単位は次の通りです。
 * ISDB-S: 1024 フレーム
 * ISDB-T: 32 フレーム (モード 1,2) / 8 フレーム (モード 3)
 */
int PT1::getCorrectedErrorRate(uint32_t tuner, ISDB isdb, LayerIndex layerIndex, ErrorRate *errorRate)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[3];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    scalarI_64[2] = layerIndex;
    uint64_t scalarO_64;
    uint32_t outputCount = 1; 
    size_t structOSize = sizeof(ErrorRate);
    kern_return_t kr;
    kr = IOConnectCallMethod(_connect,                      // an io_connect_t returned from IOServiceOpen().
                             kMethodGetCorrectedErrorRate,  // selector of the function to be called via the user client.
                             scalarI_64,                    // array of scalar (64-bit) input values.
                             3,                             // the number of scalar input values.
                             NULL,                          // a pointer to the struct input parameter.
                             0,                             // the size of the input structure parameter.
                             &scalarO_64,                   // array of scalar (64-bit) output values.
                             &outputCount,                  // pointer to the number of scalar output values.
                             errorRate,                     // pointer to the struct output parameter.
                             &structOSize);                 // pointer to the size of the output structure parameter.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief リードソロモン復号で訂正されたエラーレートを計算するためのエラーカウンタを初期化
 *
 * 全階層のカウンタを初期化します。特定の階層のカウンタをリセットすることはできません。
 */
int PT1::resetCorrectedErrorCount(uint32_t tuner, ISDB isdb)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                        // an io_connect_t returned from IOServiceOpen().
                                   kMethodResetCorrectedErrorCount, // selector of the function to be called via the user client.
                                   scalarI_64,                      // array of scalar (64-bit) input values.
                                   2,                               // the number of scalar input values.
                                   &scalarO_64,                     // array of scalar (64-bit) output values.
                                   &outputCount);                   // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief リードソロモン復号で訂正しきれなかった TS パケット数を取得
 *
 * 下位24ビットのみ有効です（回路規模を小さくするため回路番号01 にてビット数を縮小）。
 * 0x??ffffff の次は 0x??000000 になります。
 * TS パケットの 2nd Byte MSB を数えても同じ数値になります。
 * このカウンタは DMA 転送開始時に初期化されます。
 */
int PT1::getErrorCount(uint32_t tuner, ISDB isdb, uint32_t *count)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetErrorCount,    // selector of the function to be called via the user client.
                                   scalarI_64,              // array of scalar (64-bit) input values.
                                   2,                       // the number of scalar input values.
                                   scalarO_64,              // array of scalar (64-bit) output values.
                                   &outputCount);           // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *count = (uint32_t)scalarO_64[1];
    }
    return result;
}

/**
 * @brief ISDB-S の TMCC 情報を取得
 */
int PT1::getTmccS(uint32_t tuner, TmccS *tmcc)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = tuner;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    size_t structOSize = sizeof(TmccS);
    kern_return_t kr;
    kr = IOConnectCallMethod(_connect,          // an io_connect_t returned from IOServiceOpen().
                             kMethodGetTmccS,   // selector of the function to be called via the user client.
                             &scalarI_64,       // array of scalar (64-bit) input values.
                             1,                 // the number of scalar input values.
                             NULL,              // a pointer to the struct input parameter.
                             0,                 // the size of the input structure parameter.
                             &scalarO_64,       // array of scalar (64-bit) output values.
                             &outputCount,      // pointer to the number of scalar output values.
                             tmcc,              // pointer to the struct output parameter.
                             &structOSize);     // pointer to the size of the output structure parameter.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief ISDB-S のレイヤ情報を取得
 */
int PT1::getLayerS(uint32_t tuner, LayerS *layer)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = tuner;
    uint64_t scalarO_64;
    uint32_t outputCount = 1; 
    size_t structOSize = sizeof(LayerS);
    kern_return_t kr;
    kr = IOConnectCallMethod(_connect,          // an io_connect_t returned from IOServiceOpen().
                             kMethodGetLayerS,  // selector of the function to be called via the user client.
                             &scalarI_64,       // array of scalar (64-bit) input values.
                             1,                 // the number of scalar input values.
                             NULL,              // a pointer to the struct input parameter.
                             0,                 // the size of the input structure parameter.
                             &scalarO_64,       // array of scalar (64-bit) output values.
                             &outputCount,      // pointer to the number of scalar output values.
                             layer,             // pointer to the struct output parameter.
                             &structOSize);     // pointer to the size of the output structure parameter.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief ISDB-T の TMCC 情報を取得
 */
int PT1::getTmccT(uint32_t tuner, TmccT *tmcc)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = tuner;
    uint64_t scalarO_64;
    uint32_t outputCount = 1; 
    size_t structOSize = sizeof(TmccT);
    kern_return_t kr;
    kr = IOConnectCallMethod(_connect,          // an io_connect_t returned from IOServiceOpen().
                             kMethodGetTmccT,   // selector of the function to be called via the user client.
                             &scalarI_64,       // array of scalar (64-bit) input values.
                             1,                 // the number of scalar input values.
                             NULL,              // a pointer to the struct input parameter.
                             0,                 // the size of the input structure parameter.
                             &scalarO_64,       // array of scalar (64-bit) output values.
                             &outputCount,      // pointer to the number of scalar output values.
                             tmcc,              // pointer to the struct output parameter.
                             &structOSize);     // pointer to the size of the output structure parameter.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief ISDB-T ロック判定を取得
 *
 * レイヤが存在し、なおかつそのレイヤがエラーフリーであるときに true になります。
 */
int PT1::getLockedT(uint32_t tuner, bool locked[LAYER_COUNT_T])
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = tuner;
    uint64_t scalarO_64;
    uint32_t outputCount = 1; 
    size_t structOSize = sizeof(bool[LAYER_COUNT_T]);
    kern_return_t kr;
    kr = IOConnectCallMethod(_connect,          // an io_connect_t returned from IOServiceOpen().
                             kMethodGetLockedT, // selector of the function to be called via the user client.
                             &scalarI_64,       // array of scalar (64-bit) input values.
                             1,                 // the number of scalar input values.
                             NULL,              // a pointer to the struct input parameter.
                             0,                 // the size of the input structure parameter.
                             &scalarO_64,       // array of scalar (64-bit) output values.
                             &outputCount,      // pointer to the number of scalar output values.
                             locked,            // pointer to the struct output parameter.
                             &structOSize);     // pointer to the size of the output structure parameter.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief 受信階層の設定
 *
 * ISDB-S の低階層を受信しないように設定することはできません。
 */
int PT1::setLayerEnable(uint32_t tuner, ISDB isdb, LayerMask layerMask)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[3];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    scalarI_64[2] = layerMask;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetLayerEnable,   // selector of the function to be called via the user client.
                                   scalarI_64,              // array of scalar (64-bit) input values.
                                   3,                       // the number of scalar input values.
                                   &scalarO_64,             // array of scalar (64-bit) output values.
                                   &outputCount);           // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief 受信階層の取得
 */
int PT1::getLayerEnable(uint32_t tuner, ISDB isdb, LayerMask *layerMask)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2; 
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetLayerEnable,   // selector of the function to be called via the user client.
                                   scalarI_64,              // array of scalar (64-bit) input values.
                                   2,                       // the number of scalar input values.
                                   scalarO_64,              // array of scalar (64-bit) output values.
                                   &outputCount);           // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *layerMask = (LayerMask)scalarO_64[1];
    }
    return result;
}

/**
 * @brief DMA バッファの確保・解放
 *
 * DMA バッファを開放するには SetBufferInfo(NULL) とします。
 * バッファが確保されていないときに GetBufferInfo() を呼び出すと、bufferInfo が指す全てのメンバは 0 になります。
 * バッファの構成を変更する場合は、現在のバッファを解放してから改めて確保します。
 */
int PT1::setBufferInfo(const BufferInfo *bufferInfo)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64;
    uint32_t outputCount = 1; 
    kern_return_t kr;
    kr = IOConnectCallMethod(_connect,              // an io_connect_t returned from IOServiceOpen().
                             kMethodSetBufferInfo,  // selector of the function to be called via the user client.
                             NULL,                  // array of scalar (64-bit) input values.
                             0,                     // the number of scalar input values.
                             bufferInfo,            // a pointer to the struct input parameter.
                             sizeof(BufferInfo),    // the size of the input structure parameter.
                             &scalarO_64,           // array of scalar (64-bit) output values.
                             &outputCount,          // pointer to the number of scalar output values.
                             NULL,                  // pointer to the struct output parameter.
                             NULL);                 // pointer to the size of the output structure parameter.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
        _bufferInfo = *bufferInfo;
    }
    return result;
}

/**
 *
 */
int PT1::getBufferInfo(BufferInfo *bufferInfo)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    size_t structOSize = sizeof(BufferInfo);
    kern_return_t kr;
    kr = IOConnectCallMethod(_connect,              // an io_connect_t returned from IOServiceOpen().
                             kMethodGetBufferInfo,  // selector of the function to be called via the user client.
                             NULL,                  // array of scalar (64-bit) input values.
                             0,                     // the number of scalar input values.
                             NULL,                  // a pointer to the struct input parameter.
                             0,                     // the size of the input structure parameter.
                             &scalarO_64,           // array of scalar (64-bit) output values.
                             &outputCount,          // pointer to the number of scalar output values.
                             bufferInfo,            // pointer to the struct output parameter.
                             &structOSize);         // pointer to the size of the output structure parameter.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief DMA 
 *
 * index で指定した DMA バッファのポインタを取得します。index の範囲は 0 から BufferInfo::VirtualCount-1 です。
 */
int PT1::getBufferPtr(uint32_t index, void **ptr)
{
    uint32_t result = kStatusSuccess;

//    @synchronized(self)
//    {
        if (_buffer == (vm_address_t)nil)
        {
            kern_return_t kr;
            kr = IOConnectMapMemory(_connect,
                                    kIODefaultMemoryType,
                                    mach_task_self(),
                                    &_buffer,
                                    &_bufsize,
                                    kIOMapAnywhere);
            result = (kr == KERN_SUCCESS) ? kStatusSuccess : kStatusIOError;
        }
        if (result == kStatusSuccess)
        {
            *ptr = (void *)(_buffer + 4096 * 511 * _bufferInfo.VirtualSize * index);
        }
//    }

    return result;
}

/**
 * @brief 転送カウンタをリセット
 */
int PT1::resetTransferCounter()
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodResetTransferCounter, // selector of the function to be called via the user client.
                                   NULL,                        // array of scalar (64-bit) input values.
                                   0,                           // the number of scalar input values.
                                   &scalarO_64,                 // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief 転送カウンタをインクリメント
 */
int PT1::incrementTransferCounter()
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                        // an io_connect_t returned from IOServiceOpen().
                                   kMethodIncrementTransferCounter, // selector of the function to be called via the user client.
                                   NULL,                            // array of scalar (64-bit) input values.
                                   0,                               // the number of scalar input values.
                                   &scalarO_64,                     // array of scalar (64-bit) output values.
                                   &outputCount);                   // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief ストリームごとの転送制御 設定
 *
 * 各ストリームを転送するかどうかを設定することができます。
 * 必要のないストリームをオフにすることで PCI バスの帯域を無駄に使うことがなくなります。
 * DMA 転送動作中にも変更可能です。
 */
int PT1::setStreamEnable(uint32_t tuner, ISDB isdb, bool enable)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[3];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    scalarI_64[2] = enable;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetStreamEnable,  // selector of the function to be called via the user client.
                                   scalarI_64,              // array of scalar (64-bit) input values.
                                   3,                       // the number of scalar input values.
                                   &scalarO_64,             // array of scalar (64-bit) output values.
                                   &outputCount);           // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief ストリームごとの転送制御 取得
 */
int PT1::getStreamEnable(uint32_t tuner, ISDB isdb, bool *enable)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetStreamEnable,  // selector of the function to be called via the user client.
                                   scalarI_64,              // array of scalar (64-bit) input values.
                                   2,                       // the number of scalar input values.
                                   scalarO_64,              // array of scalar (64-bit) output values.
                                   &outputCount);           // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *enable = (bool)scalarO_64[1];
    }
    return result;
}

/**
 * @brief ストリームごとの 3 ビット補助データの設定
 *
 * 1 TS パケット(188バイト) は 63 マイクロパケットを使って転送されますが、
 * 3バイト×63マイクロパケット=189バイトとなり、末尾のマイクロパケットには未使用部分が 1 バイトあります。
 * このバイトの下位 3 ビットをユーザーが自由に設定することができます。
 * 復調IC からの信号を FPGA 内にデータを取り込んでからできるだけ早い時刻に 3 ビットのデータが
 * 付加されますので、タイムスタンプ代わりに利用することができます。
 * FPGA 内では値の書き込みは PCI クロックに同期し、値の読み出しは TS クロックに同期しています。
 * このため、設定する数列はグレイコードなどのハミング距離が 1 のものを使ってください。
 */
int PT1::setStreamGray(uint32_t tuner, ISDB isdb, uint32_t gray)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[3];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    scalarI_64[2] = gray;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetStreamGray,    // selector of the function to be called via the user client.
                                   scalarI_64,              // array of scalar (64-bit) input values.
                                   3,                       // the number of scalar input values.
                                   &scalarO_64,             // array of scalar (64-bit) output values.
                                   &outputCount);           // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief ストリームごとの 3 ビット補助データの取得
 */
int PT1::getStreamGray(uint32_t tuner, ISDB isdb, uint32_t *gray)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64[2];
    scalarI_64[0] = tuner;
    scalarI_64[1] = isdb;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetStreamGray,    // selector of the function to be called via the user client.
                                   scalarI_64,              // array of scalar (64-bit) input values.
                                   2,                       // the number of scalar input values.
                                   scalarO_64,              // array of scalar (64-bit) output values.
                                   &outputCount);           // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *gray = (uint32_t)scalarO_64[1];
    }
    return result;
}

/**
 * @brief DMA 開始・停止の制御 設定
 *
 * DMA 転送は全く CPU を介在することなく動作します。
 * GetTransferEnable() で true  が得られるときに SetTransferEnable(true ) としたり、
 * GetTransferEnable() で false が得られるときに SetTransferEnable(false) とするとエラーになります。
 *
 * GetTransferEnable() で取得できる値は、単に SetTransferEnable() で最後に設定された値と同じです。
 * 転送カウンタが 0 になるなど、ハードウェア側で DMA 転送が自動的に停止する要因がいくつかありますが、
 * その場合でも GetTransferEnable() で得られる値は変わりません。
 */
int PT1::setTransferEnable(bool enable)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarI_64 = enable;
    uint64_t scalarO_64;
    uint32_t outputCount = 1;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodSetTransferEnable,    // selector of the function to be called via the user client.
                                   &scalarI_64,                 // array of scalar (64-bit) input values.
                                   1,                           // the number of scalar input values.
                                   &scalarO_64,                 // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

/**
 * @brief DMA 開始・停止の制御 取得
 */
int PT1::getTransferEnable(bool *enable)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64[2];
    uint32_t outputCount = 2;
    kern_return_t kr;
    kr = IOConnectCallScalarMethod(_connect,                    // an io_connect_t returned from IOServiceOpen().
                                   kMethodGetTransferEnable,    // selector of the function to be called via the user client.
                                   NULL,                        // array of scalar (64-bit) input values.
                                   0,                           // the number of scalar input values.
                                   scalarO_64,                  // array of scalar (64-bit) output values.
                                   &outputCount);               // pointer to the number of scalar output values.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64[0];
        *enable = (bool)scalarO_64[1];
    }
    return result;
}

/**
 * @brief DMA 状態の取得
 */
int PT1::getTransferInfo(TransferInfo *transferInfo)
{
    uint32_t result = kStatusIOError;
    uint64_t scalarO_64;
    uint32_t outputCount = 1; 
    size_t structOSize = sizeof(TransferInfo);
    kern_return_t kr;
    kr = IOConnectCallMethod(_connect,                  // an io_connect_t returned from IOServiceOpen().
                             kMethodGetTransferInfo,    // selector of the function to be called via the user client.
                             NULL,                      // array of scalar (64-bit) input values.
                             0,                         // the number of scalar input values.
                             NULL,                      // a pointer to the struct input parameter.
                             0,                         // the size of the input structure parameter.
                             &scalarO_64,               // array of scalar (64-bit) output values.
                             &outputCount,              // pointer to the number of scalar output values.
                             transferInfo,              // pointer to the struct output parameter.
                             &structOSize);             // pointer to the size of the output structure parameter.
    if (kr == KERN_SUCCESS)
    {
        result = (uint32_t)scalarO_64;
    }
    return result;
}

    // マイクロパケットの構造
    // +------------+----+----+----+----+----+----+----+----+----+----+----+
    // | ビット位置 | 31 | 30 | 29 | 28 | 27 | 26 | 25 | 24 | 23 | .. |  0 |
    // +------------+----+----+----+----+----+----+----+----+----+----+----+
    // |    内容    |      id      |    counter   | st | er |     data     |
    // +------------+--------------+--------------+----+----+--------------+
    // id     : ストリームID
    // counter: ストリームごとのカウンタ
    // st     : TS パケット開始位置フラグ
    // er     : エラーフラグ (TransferCounter0 と TransferCounter1 と BufferOverflow の論理和)
    // data   : データ

    // ストリームID
    // +----+------------------------+
    // | id | 説明                   |
    // +----+------------------------+
    // |  0 | 禁止                   |
    // |  1 | チューナー番号0 ISDB-S   |
    // |  2 | チューナー番号0 ISDB-T   |
    // |  3 | チューナー番号1 ISDB-S   |
    // |  4 | チューナー番号1 ISDB-T   |
    // |  5 | 予約                   |
    // |  6 | 予約                   |
    // |  7 | 予約                   |
    // +----+------------------------+
    // ストリームID が 0 になることは絶対にありません。
    // DMA 転送がどこまで進んでいるのかを調べるには、転送前に ストリームID を 0 に設定して、
    // その箇所が 0 以外になったかどうかを調べます。
    // 実用上は転送前に 4 バイトのマイクロパケット領域に 0 を書き込み、0 以外になったかどうかを調べることになります。

    // マイクロパケットから TS パケットを再構成する方法についてはサンプルコードをご参照ください。
    // 次の関数を呼び出した直後に 188 バイトに満たないパケットが発生することがあり、切捨て処理が必要です。
    // ・SetTunerSleep()
    // ・SetFrequency()
    // ・SetIdS()
    // ・SetLayerEnable()
    // ・SetStreamEnable()
    // ・SetTransferEnable(true)


} // PTx


