/*
 *  psychlops_FFTW_bridge.cpp
 *  Psychlops Standard Library (Universal)
 *
 *  Last Modified 2010/03/05 by Kenchi HOSOKAWA
 *  (C) 2010 Kenchi HOSOKAWA, Kazushi MARUYA, Takao SATO
 */


#include "../../../psychlops_core.h"
#include "psychlops_cv2_bridge.h"
#include <opencv2/opencv.hpp>


//AppState::ImageByteAlignment = 4;

namespace Psychlops {

void Matrix::from(const cv::Mat &target)
{
	convertFromCvMat(*this, target);
}
void Matrix::convertFromCvMat(Matrix &gray, const cv::Mat &target)
{
	if(CV_64FC3 != target.type() ) {
		throw Exception("Type of pixel did not matched.");
	}
	if(! (target.cols==gray.getCols() && target.rows==gray.getRows()) ) {
		gray.release();
		gray.set(target.cols, target.rows);
	}
	//if(target.isContinuous()) { throw Exception("cv::Mat is is continuous"); }
	const double *img;

	for(int y=0; y<gray.getRows(); y++) {
		img = target.ptr<double>(y);
		for(int x=0; x<gray.getCols(); x++) {
			gray(y,x) = img[x];
		}
	}
}
void Matrix::convertFromCvMat(Matrix &r, Matrix &g, Matrix &b, const cv::Mat &target)
{
	if(!(
		   target.cols==r.getCols() && target.rows==r.getRows()
		&& target.cols==g.getCols() && target.rows==g.getRows()
		&& target.cols==b.getCols() && target.rows==b.getRows()
		)) {
		throw Exception("Size of matrices did not matched.");
	}
	if(CV_64FC3 != target.type() ) {
		throw Exception("Type of pixel did not matched.");
	}
	if(! (target.cols==r.getCols() && target.rows==r.getRows()) )
		{ r.release(); r.set(target.cols, target.rows); }
	if(! (target.cols==g.getCols() && target.rows==g.getRows()) )
		{ g.release(); g.set(target.cols, target.rows); }
	if(! (target.cols==b.getCols() && target.rows==b.getRows()) )
		{ b.release(); b.set(target.cols, target.rows); }
	//if(target.isContinuous()) { throw Exception("cv::Mat is is continuous"); }
	const double *img;

	for(int y=0; y<r.getRows(); y++) {
		img = target.ptr<double>(y);
		for(int x=0; x<r.getCols(); x++) {
			r(y,x) = img[x*3    ];
			g(y,x) = img[x*3 + 1];
			b(y,x) = img[x*3 + 2];
		}
	}
}
void Matrix::convertFromCvMat(Matrix &r, Matrix &g, Matrix &b, Matrix &a, const cv::Mat &target)
{
	if(!(
		   target.cols==r.getCols() && target.rows==r.getRows()
		&& target.cols==g.getCols() && target.rows==g.getRows()
		&& target.cols==b.getCols() && target.rows==b.getRows()
		&& target.cols==a.getCols() && target.rows==a.getRows()
		)) {
		throw Exception("Size of matrices did not matched.");
	}
	if(CV_64FC4 != target.type() ) {
		throw Exception("Type of pixel did not matched.");
	}
	if(! (target.cols==r.getCols() && target.rows==r.getRows()) )
		{ r.release(); r.set(target.cols, target.rows); }
	if(! (target.cols==g.getCols() && target.rows==g.getRows()) )
		{ g.release(); g.set(target.cols, target.rows); }
	if(! (target.cols==b.getCols() && target.rows==b.getRows()) )
		{ b.release(); b.set(target.cols, target.rows); }
	if(! (target.cols==a.getCols() && target.rows==a.getRows()) )
		{ a.release(); a.set(target.cols, target.rows); }
	//if(target.isContinuous()) { throw Exception("cv::Mat is is continuous"); }
	const double *img;

	for(int y=0; y<r.getRows(); y++) {
		img = target.ptr<double>(y);
		for(int x=0; x<r.getCols(); x++) {
			r(y,x) = img[x*4    ];
			g(y,x) = img[x*4 + 1];
			b(y,x) = img[x*4 + 2];
			a(y,x) = img[x*4 + 3];
		}
	}
}

void Matrix::to(cv::Mat &target) const
{
	convertToCvMat(*this, target);
}

void Matrix::convertToCvMat(const Matrix &gray, cv::Mat &target)
{
	if(! (target.cols==gray.getCols() && target.rows==gray.getRows()) ) {
		if(CV_64FC3 != target.type() ) {
			target = cv::Mat(cvSize(gray.getCols(), gray.getRows()), CV_64FC3);
		}
	}
	//if(target.isContinuous()) { throw Exception("cv::Mat is is continuous"); }
	double *img;

	for(int y=0; y<gray.getRows(); y++) {
		img = target.ptr<double>(y);
		for(int x=0; x<gray.getCols(); x++) {
			img++;
			*img = gray(y,x);
		}
	}
}

void Matrix::convertToCvMat(const Matrix &r, const Matrix &g, const Matrix &b, cv::Mat &target)
{
	if(!(
		   target.cols==r.getCols() && target.rows==r.getRows()
		&& target.cols==g.getCols() && target.rows==g.getRows()
		&& target.cols==b.getCols() && target.rows==b.getRows()
		)) {
		throw Exception("Size of matrices did not matched.");
	}
	if(! (target.cols==r.getCols() && target.rows==r.getRows()) ) {
		if(CV_64FC3 != target.type() ) {
			target = cv::Mat(cvSize(r.getCols(), r.getRows()), CV_64FC3);
		}
	}
	//if(target.isContinuous()) { throw Exception("cv::Mat is is continuous"); }
	double *img;

	for(int y=0; y<r.getRows(); y++) {
		img = target.ptr<double>(y);
		for(int x=0; x<r.getCols(); x++) {
			img[x*3    ] = r(y,x);
			img[x*3 + 1] = g(y,x);
			img[x*3 + 2] = b(y,x);
		}
	}
}

void Matrix::convertToCvMat(const Matrix &r, const Matrix &g, const Matrix &b, const Matrix &a, cv::Mat &target)
{
	double *img;
	if(!(
		   target.cols==r.getCols() && target.rows==r.getRows()
		&& target.cols==g.getCols() && target.rows==g.getRows()
		&& target.cols==b.getCols() && target.rows==b.getRows()
		&& target.cols==a.getCols() && target.rows==a.getRows()
		)) {
		throw Exception("Size of matrices did not matched.");
	}
	if(! (target.cols==r.getCols() && target.rows==r.getRows()) ) {
		if(CV_64FC3 != target.type() ) {
			target = cv::Mat(cvSize(r.getCols(), r.getRows()), CV_64FC4);
		}
	}
	//if(target.isContinuous()) { throw Exception("cv::Mat is is continuous"); }

	for(int y=0; y<r.getRows(); y++) {
		img = target.ptr<double>(y);
		for(int x=0; x<r.getCols(); x++) {
			img[x*4    ] = r(y,x);
			img[x*4 + 1] = g(y,x);
			img[x*4 + 2] = b(y,x);
			img[x*4 + 3] = a(y,x);
		}
	}
}


void Image::to(cv::Mat &target) const
{
	char pix[4];
	Color c;

	unsigned char *img;

	switch(this->getComponentKind()) {
		case Image::GRAY:
			target = cv::Mat(cvSize(this->getWidth(), this->getHeight()), CV_8UC1);
			for(int y=0; y<this->getHeight(); y++) {
				img = target.ptr<unsigned char>(y);
				for(int x=0; x<this->getWidth(); x++) {
					c = this->getPix(x,y);
					img[x] = (unsigned char)(c.getR()*255.0+.5);
				}
			}
			break;
		case Image::RGB:
			target = cv::Mat(cvSize(this->getWidth(), this->getHeight()), CV_8UC3 );
			for(int y=0; y<this->getHeight(); y++) {
				img = target.ptr<unsigned char>(y);
				for(int x=0; x<this->getWidth(); x++) {
					c = this->getPix(x,y);
					img[x*3    ] = (unsigned char)(c.getB()*255.0+.5);
					img[x*3 + 1] = (unsigned char)(c.getG()*255.0+.5);
					img[x*3 + 2] = (unsigned char)(c.getR()*255.0+.5);
				}
			}
			break;
		case Image::RGBA:
			target = cv::Mat(cvSize(this->getWidth(), this->getHeight()), CV_8UC4 );
			for(int y=0; y<this->getHeight(); y++) {
				img = target.ptr<unsigned char>(y);
				for(int x=0; x<this->getWidth(); x++) {
					c = this->getPix(x,y);
					img[x*4    ] = (unsigned char)(c.getB()*255.0+.5);
					img[x*4 + 1] = (unsigned char)(c.getG()*255.0+.5);
					img[x*4 + 2] = (unsigned char)(c.getR()*255.0+.5);
					img[x*4 + 3] = (unsigned char)(c.getA()*255.0+.5);
				}
			}
			break;
		default:
			break;
	}

}


void Image::from(cv::Mat &source)
{

	size_t widthStep = source.step;
	size_t lineBytes;
	unsigned char *img_b = (unsigned char *)source.data;
	unsigned char *img = img_b;
	unsigned char *timg;
	unsigned char *timg_b;
	unsigned char pix[4];
	Color c;

	switch(source.channels()) {
		case 1:
			if(this->getComponentKind() != Image::GRAY || this->getWidth() != source.cols || this->getHeight() != source.rows ) {
				this->release();
				this->set(source.cols, source.rows, Image::GRAY);
			}
			lineBytes = this->getLineBytes();
			timg_b = this->getElementPtr();
			for(int y=0; y<this->getHeight(); y++) {
				img = img_b + widthStep*y;
				timg = timg_b + lineBytes*((source.rows-y)-1);
				for(int x=0; x<this->getWidth(); x++) {
					*(timg++) = (unsigned char)img[0];
					img += 1;
				}
			}
			break;
		case 3:
			if(this->getComponentKind() != Image::RGB || this->getWidth() != source.cols || this->getHeight() != source.rows ) {
				this->release();
				this->set(source.cols, source.rows, Image::RGB);
			}
			lineBytes = this->getLineBytes();
			timg_b = this->getElementPtr();
			for(int y=0; y<this->getHeight(); y++) {
				img = img_b + widthStep*y;
				timg = timg_b + lineBytes*((source.rows-y)-1);
				for(int x=0; x<this->getWidth(); x++) {
					*(timg++) = (unsigned char)img[2];
					*(timg++) = (unsigned char)img[1];
					*(timg++) = (unsigned char)img[0];
					img += 3;
				}
			}
			break;
		case 4:
			if(this->getComponentKind() != Image::RGBA || this->getWidth() != source.cols || this->getHeight() != source.rows ) {
				this->release();
				this->set(source.cols, source.rows, Image::RGBA);
			}
			lineBytes = this->getLineBytes();
			timg_b = this->getElementPtr();
			for(int y=0; y<this->getHeight(); y++) {
				img = img_b + widthStep*y;
				timg = timg_b + lineBytes*((source.rows-y)-1);
				for(int x=0; x<this->getWidth(); x++) {
					*(timg++) = (unsigned char)img[2];
					*(timg++) = (unsigned char)img[1];
					*(timg++) = (unsigned char)img[0];
					*(timg++) = (unsigned char)img[3];
					img += 4;
				}
			}
			break;
		default:
			break;
	}
}


}	/*	<- namespace Psycholops 	*/
