﻿// Copyright (C) 2010, 2013 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: ProtraConfig.cs 504 2014-02-11 09:49:09Z panacoran $

using System.Collections.Generic;
using System.Drawing;
using Protra.Lib.Data;

namespace Protra.Lib.Config
{
    /// <summary>
    /// チャートの設定を格納するクラス。
    /// </summary>
    public class ChartConfig
    {
        /// <summary>
        /// 高さを取得または設定する。
        /// </summary>
        public int Height { get; set; }

        /// <summary>
        /// 日足のチャートを取得または設定する。
        /// </summary>
        public string DailyFile { get; set; }

        /// <summary>
        /// 週足のチャートを取得または設定する。
        /// </summary>
        public string WeeklyFile { get; set; }
    }

    /// <summary>
    /// Protraの設定を読み書きするクラス。
    /// </summary>
    public class ProtraConfig : ConfigBase
    {
        // フォーム

        /// <summary>
        /// ウィンドウの大きさを取得または設定する。
        /// </summary>
        public Size Size { get; set; }

        /// <summary>
        /// ウィンドウの位置を取得または設定する。
        /// </summary>
        public Point Location { get; set; }

        /// <summary>
        /// 幅を取得または設定する。(廃止予定)
        /// </summary>
        public int Width { get; set; }

        /// <summary>
        /// 高さを取得または設定する。(廃止予定)
        /// </summary>
        public int Height { get; set; }

        /// <summary>
        /// パネルの左側の幅を取得または設定する。
        /// </summary>
        public int PanelLeftWidth { get; set; }

        /// <summary>
        /// 最大化しているかを取得または設定する。
        /// </summary>
        public bool Maximized { get; set; }

        /// <summary>
        /// 選択されている証券コードを取得または設定する。
        /// </summary>
        public string SelectedCode { get; set; }

        /// <summary>
        /// 選択されている銘柄リスト名を取得または設定する。
        /// </summary>
        public string BrandListName { get; set; }

        // チャート

        /// <summary>
        /// チャートの表示モードを取得または設定する。(廃止予定)
        /// </summary>
        public int ChartMode { get; set; }

        /// <summary>
        /// チャートのタイムフレームを取得または設定する。
        /// </summary>
        public TimeFrame TimeFrame { get; set; }

        /// <summary>
        /// チャートの前面色を取得または設定する。
        /// </summary>
        public int ChartForeColor { get; set; }

        /// <summary>
        /// チャートの背面色を取得または設定する。
        /// </summary>
        public int ChartBackColor { get; set; }

        /// <summary>
        /// 不完全な週足を描画するかを取得または設定する。
        /// </summary>
        public bool DrawLastWeek { get; set; }

        /// <summary>
        /// 日足と週足で異なるチャートを使用するかを取得または設定する。
        /// </summary>
        public bool UseDifferentChart { get; set; }

        /// <summary>
        /// チャートの個数を取得または設定する。
        /// </summary>
        public int ChartNum { get; set; }

        /// <summary>
        /// チャートの設定のリストを取得または設定する。
        /// </summary>
        public List<ChartConfig> ChartList { get; set; }

        /// <summary>
        /// 設定ファイルの名前を取得する。
        /// </summary>
        protected override string ConfigName
        {
            get { return "protra"; }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ProtraConfig()
        {
            Location = new Point(int.MinValue, int.MinValue);
            ChartNum = 3;
            ChartForeColor = Color.FromKnownColor(KnownColor.ControlText).ToArgb();
            ChartBackColor = Color.FromKnownColor(KnownColor.Control).ToArgb();
            ChartList = new List<ChartConfig>();
        }

        /// <summary>
        /// 廃止予定のプロパティの値を新しいプロパティに移す。
        /// </summary>
        protected override void Upgrade()
        {
            Size = new Size(Width, Height);
            TimeFrame = (TimeFrame)ChartMode;
        }

        /// <summary>
        /// 新しいプロパティの値を廃止予定のプロパティに戻す。
        /// </summary>
        protected override void Downgrade()
        {
            Width = Size.Width;
            Height = Size.Height;
            ChartMode = (int)TimeFrame;
        }
    }
}