// Copyright (C) 2008 panacorn <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: YahooFinanceUpdator.cs,v 1.1 2008-02-03 13:27:08 panacoran Exp $

using System;
using System.Collections;
using System.ComponentModel;
using System.Text;
using System.Text.RegularExpressions;
using System.Globalization;
using System.IO;
using System.Net;
using Protra.Lib.Db;

namespace Protra.Lib.Update
{
	/// <summary>
	/// Yahoo!t@CiX𗘗pĊf[^XVNXB
	/// </summary>
    public class YahooFinanceUpdator: PriceDataUpdator
    {
        Regex regex;

        /// <summary>
        /// f[^݂ŏ̓t擾B
        /// </summary>
        public static DateTime DataSince
        {
            get { return new DateTime(1991, 1, 4); }
        }

        /// <summary>
        /// RXgN^B
        /// </summary>
        public YahooFinanceUpdator()
        {
            regex = new Regex("<td><b>(?:\\(\\))?(?<name>[^(<]+)(?:\\(\\))?</b>y(?<market>.*)\\n" +
                ": (?<code>[0-9]+)\\.[TOQNJ]z.*" +
                "<td><small>(?<year>\\d{4})N(?<month>1?\\d)(?<day>\\d?\\d)</small></td>" +
                "<td><small>(?<open>[0-9,.]+)</small></td>" +
                "<td><small>(?<high>[0-9,.]+)</small></td>" +
                "<td><small>(?<low>[0-9,.]+)</small></td>" +
                "<td><small><b>(?<close>[0-9,.]+)</b></small></td>" +
                "(?:<td><small>(?<volume>[0-9,]+)</small></td>)?",
                RegexOptions.Compiled | RegexOptions.Singleline);
        }

        /// <summary>
        /// f[^XV܂B
        /// </summary>
        /// <param name="worker">BackgroundWorker</param>
        /// <param name="e">DoWorkCxg̈</param>
        public override void Update(BackgroundWorker worker, DoWorkEventArgs e)
        {
            DateTime begin = (DateTime)e.Argument;
            if (begin < DataSince)
                begin = DataSince;
            DateTime today = DateTime.Now;
            // Vf[^û͑Ăߌ7ȍ~
            if (today.Hour < 19)
                today = today.AddDays(-1);

            for (Start(begin, today); ShouldContinue(); NextDate())
            {
                UpdateProgress(worker, e);

                TotalRecords = 9999 - 1300 + 2;
                for (; NumRecords < TotalRecords; IncrementRecords(), UpdateProgress(worker, e))
                {
                    if (worker.CancellationPending)
                    {
                        e.Cancel = true;
                        PriceTable.Delete(Date);
                        return;
                    }
                    int code;
                    if (NumRecords == 0)
                        code = 998407; // o225
                    else if (NumRecords == 1)
                        code = 998405; // TOPIX
                    else
                        code = NumRecords - 2 + 1301;
                    PriceData data = ParsePage(GetPage(code));
                    if (data == null)
                    {
                        if (NumRecords == 0) // oςȂB
                            break;
                        else
                            continue;
                    }
                    Brand brand = BrandTable.GetRecordOrCreate(data.MarketId, data.Code, data.Name);
                    PriceTable.Add(brand.Id, data.Date, data.Open, data.High, data.Low, data.Close, data.Volume);
                }
                if (NumRecords == TotalRecords)
                    IncrementDays();
                else
                    DecrementTotalDays();
            }

            UpdateSplitData(worker, e);
            worker.ReportProgress(100, "");
        }

        private string GetPage(int code)
        {
            DownloadUtil.Url =
                "http://table.yahoo.co.jp/t?c=" + Date.Year + "&a=" + Date.Month + "&b=" + Date.Day +
                "&f=" + Date.Year + "&d=" + Date.Month + "&e=" + Date.Day + "&g=d" + "&s=" + code +
                "&y=0" + "&z=" + code;
            for (int i = 0; i < 10; i++)
            {
                try
                {
                    Stream stream = DownloadUtil.GetResponse();
                    if (stream == null)
                        return null;
                    StreamReader reader = new StreamReader(stream, Encoding.GetEncoding("euc-jp"));
                    string buf = reader.ReadToEnd();
                    reader.Close();
                    return buf;
                }
                catch (WebException e)
                {
                    switch (e.Status)
                    {
                        case WebExceptionStatus.Timeout:
                        case WebExceptionStatus.ConnectionClosed:
                        case WebExceptionStatus.ReceiveFailure:
                            break;
                        default:
                            throw;
                    }
                }
            }
            return null;
        }

        private PriceData ParsePage(string buf)
        {
            if (buf == null)
                return null;
            Match m = regex.Match(buf);
            if (!m.Success)
                return null;
            PriceData r = new PriceData();
            try
            {
                r.Date = new DateTime(int.Parse(m.Groups["year"].Value),
                    int.Parse(m.Groups["month"].Value),
                    int.Parse(m.Groups["day"].Value));
                r.Code = int.Parse(m.Groups["code"].Value);
                r.Name = m.Groups["name"].Value;
                if (r.Name.EndsWith("z[fBOX"))
                    r.Name = r.Name.Substring(0, r.Name.Length - ((r.Name.EndsWith("E")) ? 9 : 8)) + "gc";
                NumberStyles s = NumberStyles.AllowDecimalPoint | NumberStyles.AllowThousands;
                r.Open = (int)double.Parse(m.Groups["open"].Value, s);
                r.High = (int)double.Parse(m.Groups["high"].Value, s);
                r.Low = (int)double.Parse(m.Groups["low"].Value, s);
                r.Close = (int)double.Parse(m.Groups["close"].Value, s);
                if (m.Groups["volume"].Value != "")
                    r.Volume = double.Parse(m.Groups["volume"].Value, s) / 1000;
                switch (m.Groups["market"].Value)
                {
                    case "1":
                    case "":
                        r.MarketId = MarketId.Tokyo1;
                        break;
                    case "2":
                        r.MarketId = MarketId.Tokyo2;
                        break;
                    case "}U[Y":
                        r.MarketId = MarketId.Mothers;
                        break;
                    case "؊O":
                        r.MarketId = MarketId.TokyoForeign;
                        break;
                    case "":
                    case "1":
                        r.MarketId = MarketId.Osaka1;
                        break;
                    case "2":
                        r.MarketId = MarketId.Osaka2;
                        break;
                    case "wNX":
                        r.MarketId = MarketId.Hercules;
                        break;
                    case "":
                    case "1":
                        r.MarketId = MarketId.Nagoya1;
                        break;
                    case "2":
                        r.MarketId = MarketId.Nagoya2;
                        break;
                    case "JASDAQ":
                        r.MarketId = MarketId.Jasdaq;
                        break;
                    default:
                        return null;
                }
            }
            catch (FormatException)
            {
                return null;
            }
            // sɍ킹B
            if (r.Code == 998407)
            {
                r.Code = 1001;
                r.MarketId = MarketId.Tokyo1;
            }
            else if (r.Code == 998405)
                r.Code = 1002;
            return r;
        }
    }
}
