﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008, 2013 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: ArithmeticTermNode.cs 472 2013-06-26 15:26:49Z panacoran $

using System.Collections.Generic;

namespace Protra.Lib.Lang
{
    /// <summary>
    /// 算術項のノードを表すクラス。
    /// </summary>
    public class ArithmeticTermNode : ExpressionNode
    {
        /// <summary>
        /// ノードのリスト
        /// </summary>
        private List<ExpressionNode> _nodeList;

        /// <summary>
        /// 演算の種類を表すキャラクタのリスト
        /// </summary>
        private List<char> _opCharList;

        /// <summary>
        /// 構文解析する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.ParseException">
        /// 構文解析中にエラーが発生したときにthrowされる。
        /// </exception>
        /// <return>
        /// 解析した結果のノード。
        /// </return>
        public override ExpressionNode Parse(Scanner scanner)
        {
            Token = scanner.Token;
            var node = new PrefixedExpressionNode().Parse(scanner);
            var token = scanner.Token.Value;
            if (!(token == "*" || token == "/" || token == "%"))
                return node;
            _nodeList = new List<ExpressionNode>();
            _opCharList = new List<char>();
            _nodeList.Add(node);
            do
            {
                _opCharList.Add(token[0]);
                scanner.Scan();
                _nodeList.Add(new PrefixedExpressionNode().Parse(scanner));
                token = scanner.Token.Value;
            } while (token == "*" || token == "/" || token == "%");
            return this;
        }

        /// <summary>
        /// 式を評価する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// 評価中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <param name="resource">リソース</param>
        /// <param name="at">int型@作用素の値</param>
        /// <param name="ats">string型@作用素の値</param>
        /// <returns>値</returns>
        public override Value Evaluate(Resource resource, int at, string ats)
        {
            try
            {
                var val1 = _nodeList[0].Evaluate(resource, at, ats);
                for (var i = 1; i < _nodeList.Count; i++)
                {
                    var val2 = _nodeList[i].Evaluate(resource, at, ats);
                    switch (_opCharList[i - 1])
                    {
                        case '*':
                            val1 = val1 * val2;
                            break;
                        case '/':
                            val1 = val1 / val2;
                            break;
                        default:
                            val1 = val1 % val2;
                            break;
                    }
                }
                return val1;
            }
            catch (RuntimeException e)
            {
                throw new RuntimeException(e.Message, e.Token ?? Token, e);
            }
        }
    }
}