﻿// Copyright (C) 2010, 2013 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: ConfigBase.cs 442 2013-03-04 11:56:25Z panacoran $

using System;
using System.IO;
using System.Windows.Forms;
using System.Xml.Serialization;
using Protra.Lib.Dialogs;

namespace Protra.Lib.Config
{
    /// <summary>
    /// 設定ファイルを読み書きする抽象クラス。
    /// </summary>
    abstract public class ConfigBase
    {
        /// <summary>
        /// 古い設定ファイルを読み込む。
        /// </summary>
        abstract protected void ReadOldConfig();

        /// <summary>
        /// 設定ファイルの名前を取得する。
        /// </summary>
        abstract protected string ConfigName { get; }

        private readonly string filename;
        private readonly XmlSerializer serializer;
        private readonly FileChangeWatcher watcher;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected ConfigBase()
        {
// ReSharper disable DoNotCallOverridableMethodsInConstructor
            filename = Path.Combine(Global.DirConf, ConfigName + ".xml");
// ReSharper restore DoNotCallOverridableMethodsInConstructor
            serializer = new XmlSerializer(GetType());
            if (!Directory.Exists(Global.DirConf))
                Directory.CreateDirectory(Global.DirConf);
            watcher = new FileChangeWatcher(filename) {ProcessFile = Load};
        }

        /// <summary>
        /// 設定ファイルを読み込む。
        /// </summary>
        public virtual void Load()
        {
            try
            {
                using (var reader = File.OpenText(filename))
                {
                    var tmp = serializer.Deserialize(reader);
                    foreach (var property in GetType().GetProperties())
                        if (property.GetCustomAttributes(typeof(XmlIgnoreAttribute), false).GetLength(0) == 0)
                            property.SetValue(this, property.GetValue(tmp, null), null);
                }
            }
            catch (FileNotFoundException)
            {
                ReadOldConfig();
                Save();
            }
            catch (InvalidOperationException e)
            {
                MessageBox.Show("設定ファイルが壊れています。\n" + filename + "\n" + e.Message, "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            // シリアライズで生成したオブジェクトのwatcherが有効にならないように、
            // コンストラクタの外で有効にする。
            watcher.Enabled = true;
        }

        /// <summary>
        /// 設定ファイルを書き込む。
        /// </summary>
        public virtual void Save()
        {
            try
            {
                if (watcher != null)
                    watcher.Enabled = false;
                using (var file = File.CreateText(filename))
                    serializer.Serialize(file, this);
            }
            catch (Exception e)
            {
                MessageBox.Show(e.Message, "エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            finally
            {
                if (watcher != null)
                    watcher.Enabled = true;
            }
        }
    }
}
