#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Miscellaneous functions -
# Copyright (C) 2013-2017 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Variables =============
MISC_IS_READONLY_MODE=no
MISC_DEFAULT_CONSOLE_SIZE='25 80'

# ============= Check whether the current process is by the superuser privilege =============
misc_is_superuser_privilege ()
{
	[ `id -u` -eq 0 -a "$MISC_IS_READONLY_MODE" = no ]
}

# ============= Lock for duplicated executions =============
misc_lock_duplicated_executions ()
{
	local lockfile
	lockfile=$1
	MISC_IS_READONLY_MODE=no
	[ `id -u` -eq 0 ] || return 0
	if mktemp -q "$lockfile" > /dev/null
	then
		echo $$ > $lockfile
	elif [ x`cat "$lockfile" 2> /dev/null` != x$$ ]
	then
		message_echo "WARNING: Another ${APPNAME} process is running; commands requiring the superuser privilege is unavailable." >&2
		message_echo "If this report is incorrect, remove an obsolete lock file ($lockfile)." >&2
		message_echo >&2
		MISC_IS_READONLY_MODE=yes
	fi
	:
}

# ============= Check the superuser privilege and abort if not =============
misc_chk_privilege ()
{
	misc_is_superuser_privilege && return
	message_echo "ERROR: The superuser privilege is required for this command." >&2
	exit 1
}

# ============= Get all shell variable definitions (without functions) =============
# Actually, this filtering is unnecessary for Bourne shell and only required for Bash.
# Since the overhead is not harmful, I will keep this for correspondence to possible future changes.
misc_get_all_vardefs ()
{
	set | sed -E '/^[^ ]+ \(\) *$/,/^\}$/d'
}

# ============= Initialize shell variable definitions =============
misc_init_vardefs ()
{
	eval `misc_get_all_vardefs | grep -E '^[a-z_][a-zA-Z0-9_]+=' | sed 's/=.*//;s/^/unset /'`
}

# ============= Get the size of the current console =============
misc_get_console_size ()
{
	local size
	size=`stty -f /dev/stdin size 2> /dev/null || :`
	if [ -n "$size" ]
	then
		echo "$size" > ${TMPDIR}/misc_get_console_size::size
	else
		if [ ! -e "${TMPDIR}/misc_get_console_size::size" ]
		then
			size=`stty -f /dev/console size 2> /dev/null || :`
		else
			size=`cat "${TMPDIR}/misc_get_console_size::size"`
		fi
	fi
	[ -n "$size" ] || size=$MISC_DEFAULT_CONSOLE_SIZE
	echo "$size"
}

# ============= Get the number of columns for the current console =============
misc_get_console_column_size ()
{
	misc_get_console_size | sed -E 's/^[^ ]+ +([^ ]+)/\1/'
}

# ============= Get the number of rows for the current console =============
misc_get_console_row_size ()
{
	misc_get_console_size | sed -E 's/^([^ ]+) +[^ ]+/\1/'
}

# ============= Selection of removing leaf ports =============
# Box options for dialog(1) are given via stdin.
misc_dialog_checklist ()
{
	local title desc dstfile itemlist cmdscript LINES COLUMNS hight width width_desc lines_desc hight_list width_list nlines iline len_tag tag len_tag_cur len_item_max len_item_trim
	title=$1
	desc=$2
	dstfile=$3
	itemlist=$4
	cmdscript=${TMPDIR}/misc_dialog_checklist::command
	if [ `wc -l < $itemlist` -eq 0 ]
	then
		cp /dev/null "$dstfile"
		return 0
	fi
	echo -n 'dialog --title "$title" --checklist "$desc" $hight $width $hight_list ' > $cmdscript
	COLUMNS=`misc_get_console_column_size`
	LINES=`misc_get_console_row_size`
	hight=$(($LINES-3))
	[ $hight -gt 0 ] || hight=$LINES
	width=$(($COLUMNS-5))
	[ $width -gt 0 ] || width=$COLUMNS
	width_desc=$(($width-4))
	[ $width_desc -gt 0 ] || width_desc=$COLUMNS
	lines_desc=`echo "$desc" | sed -E 's/\\n/\
/g' | fold -s -w $width_desc | wc -l`
	hight_list=$(($hight-$lines_desc-6))
	[ $hight_list -gt 0 ] || hight_list=$LINES
	width_list=$(($width-6))
	[ $width_list -gt 0 ] || width_list=$COLUMNS
	nlines=`wc -l < $itemlist`
	iline=1
	len_tag=0
	while [ $iline -le $nlines ]
	do
		tag=`sed -n "${iline}p" "$itemlist" | cut -f 1`
		iline=$(($iline+1))
		len_tag_cur=`echo -n "$tag" | wc -c`
		[ $len_tag_cur -gt $len_tag ] && len_tag=$len_tag_cur
	done
	iline=1
	len_item_max=$(($width_list-$len_tag-6))
	[ $len_item_max -gt 0 ] || len_item_max=$width_list
	len_item_trim=$(($len_item_max-3))
	[ $len_item_trim -gt 0 ] || len_item_trim=$len_item_max
	while [ $iline -le $nlines ]
	do
		tag=`sed -n "${iline}p" "$itemlist" | cut -f 1`
		item=`sed -n "${iline}p" "$itemlist" | cut -f 2`
		status=`sed -n "${iline}p" "$itemlist" | cut -f 3`
		iline=$(($iline+1))
		len_item=`echo -n "$item" | wc -c`
		if [ $len_item -gt $len_item_max ]
		then
			item=`echo -n "$item" | head -c $len_item_trim`...\"
		fi
		echo -n "$tag $item $status "
	done >> $cmdscript
	echo ' 2> ${TMPDIR}/misc_dialog_checklist::selected_items || :' >> $cmdscript
	. "$cmdscript"
	tr -d \" < ${TMPDIR}/misc_dialog_checklist::selected_items | tr ' ' '\n' > $dstfile
	echo >> $dstfile
}

