package org.phosphoresce.commons.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;
import java.util.Properties;

/**
 * プロパティファイル操作におけるユーティリティクラス。<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者			更新内容
 * 2007/02/02	Kitagawa		新規作成
 * 2009/04/03	Kitagawa		loadメソッド追加
 * 2009/04/21	Kitagawa		loadメソッドで指定したリソースをファイルシステムに限らず、クラスローダが提供するリソースも検索するように修正
 * 2010/07/05	Kitagawa		ロケールを考慮してロードするload(String, boolean)を追加
 *-->
 */
public final class PropertiesUtil {

	/**
	 * コンストラクタ<br>
	 */
	private PropertiesUtil() {
		//
	}

	/**
	 * 指定されたファイルパスのPropertiesファイルを読み込みます。<br>
	 * @param filename ファイルパス
	 * @param localeBundle 指定ファイルに対してロケールを考慮したリソースを検索する場合にtrueを指定
	 * @return Propertiesオブジェクト
	 * @throws IOException 読込み時に入出力例外が発生した場合にスローされます
	 */
	public static Properties load(String filename, boolean localeBundle) throws IOException {
		//FileInputStream stream = new FileInputStream(filename);
		String basename = filename.substring(0, filename.lastIndexOf("."));
		String target = null;
		if (localeBundle) {
			target = basename + "_" + Locale.getDefault().getLanguage() + "_" + Locale.getDefault().getCountry() + ".properties";
			if (!ResourceUtil.isExistResource(target)) {
				target = basename + "_" + Locale.getDefault().getLanguage() + ".properties";
			}
			if (!ResourceUtil.isExistResource(target)) {
				target = basename + "_" + Locale.getDefault().getCountry() + ".properties";
			}
			if (!ResourceUtil.isExistResource(target)) {
				target = basename + ".properties";
			}
		} else {
			target = filename;
		}
		InputStream stream = ResourceUtil.getInputStream(target);
		Properties properties = new Properties();
		properties.load(stream);
		stream.close();
		return properties;
	}

	/**
	 * 指定されたファイルパスのPropertiesファイルを読み込みます。<br>
	 * @param filename ファイルパス
	 * @return Propertiesオブジェクト
	 * @throws IOException 読込み時に入出力例外が発生した場合にスローされます
	 */
	public static Properties load(String filename) throws IOException {
		return load(filename, false);
	}

	/**
	 * 指定されたキーのプロパティ値を文字列として取得します。<br>
	 * 正常にプロパティ値を取得できない場合は空文字を返却します。<br>
	 * @param properties 取得対象Propertiesオブジェクト
	 * @param key プロパティキー
	 * @return プロパティ値
	 */
	public static String getStringValue(Properties properties, String key) {
		String value = properties == null ? "" : properties.getProperty(key);
		return value == null ? "" : value;
	}

	/**
	 * 指定されたキーのプロパティ値をint型として取得します。<br>
	 * 正常にプロパティ値を取得できない場合は0を返却します。<br>
	 * @param properties 取得対象Propertiesオブジェクト
	 * @param key プロパティキー
	 * @return プロパティ値
	 */
	public static int getPrimitiveIntValue(Properties properties, String key) {
		String value = getStringValue(properties, key);
		return StringUtil.isEmpty(value) || !StringUtil.isNumeric(value) ? 0 : Integer.parseInt(value);
	}

	/**
	 * 指定されたキーのプロパティ値をlong型として取得します。<br>
	 * 正常にプロパティ値を取得できない場合は0を返却します。<br>
	 * @param properties 取得対象Propertiesオブジェクト
	 * @param key プロパティキー
	 * @return プロパティ値
	 */
	public static long getPrimitiveLongValue(Properties properties, String key) {
		String value = getStringValue(properties, key);
		return StringUtil.isEmpty(value) || !StringUtil.isNumeric(value) ? 0 : Long.parseLong(value);
	}

	/**
	 * 指定されたキーのプロパティ値をboolean型として取得します。<br>
	 * 正常にプロパティ値を取得できない場合はfalseを返却します。<br>
	 * @param properties 取得対象Propertiesオブジェクト
	 * @param key プロパティキー
	 * @return プロパティ値
	 */
	public static boolean getPrimitiveBooleanValue(Properties properties, String key) {
		String value = getStringValue(properties, key);
		return "true".equalsIgnoreCase(value);
	}
}
