package org.phosphoresce.common.graphics.util;

import java.awt.Color;

/**
 * カラーユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/06/05	Kitagawa		新規作成
 *-->
 */
public final class ColorUtil {

	/**
	 * コンストラクタ<br>
	 */
	private ColorUtil() {
		super();
	}

	/**
	 * 指定された色の成分を、HSBモデルの3つの成分である色相、彩度、明度の値の対応するセットに変換します。<br>
	 * @param color カラーオブジェクト
	 * @return 指定された赤色、緑色、および青色の成分を持つ色の、色相、彩度、明度をこの順序で格納する3つの要素の配列
	 */
	public static float[] getHSB(Color color) {
		if (color == null) {
			return new float[] { 0.0f, 0.0f, 0.0f };
		} else {
			return Color.RGBtoHSB(color.getRed(), color.getGreen(), color.getBlue(), new float[3]);
		}
	}

	/**
	 * 指定された色の色相を取得します。<br>
	 * @param color カラーオブジェクト
	 * @return 色相値
	 */
	public static float getHue(Color color) {
		return getHSB(color)[0];
	}

	/**
	 * 指定された色の彩度を取得します。<br>
	 * @param color カラーオブジェクト
	 * @return 彩度値
	 */
	public static float getSaturation(Color color) {
		return getHSB(color)[1];
	}

	/**
	 * 指定された色の明度を取得します。<br>
	 * @param color カラーオブジェクト
	 * @return 明度値
	 */
	public static float getBrightness(Color color) {
		return getHSB(color)[2];
	}

	/**
	 * 指定されたカラーオブジェクトの輝度を指定値分変更します。<br>
	 * 正数を指定することで明るい色に変更され、負数を指定することで暗い色に変更されます。<br>
	 * @param color 対象カラーオブジェクト
	 * @param value 輝度変更値(-1.0f～+1.0f)
	 * @return 輝度を変更されたカラーオブジェクト
	 */
	public static Color brightness(Color color, float value) {
		if (color == null) {
			return null;
		}
		float[] hsb = getHSB(color);
		float newvalue = hsb[2] + value;
		if (newvalue > 1.0f) {
			newvalue = 1.0f;
		} else if (newvalue < 0.0f) {
			newvalue = 0.0f;
		}
		Color render = Color.getHSBColor(hsb[0], hsb[1], newvalue);
		return new Color(render.getRed(), render.getGreen(), render.getBlue(), color.getAlpha());
	}

	/**
	 * 指定されたカラーオブジェクトの透明度を指定値分変更します。<br>
	 * 正数を指定することで不透明に変更され、負数を指定することで透明に変更されます。<br>
	 * @param color 対象カラーオブジェクト
	 * @param value 透明度変更値(-255～255)
	 * @return 透明度を変更されたカラーオブジェクト
	 */
	public static Color alphaness(Color color, int value) {
		if (color == null) {
			return null;
		}
		int r = color.getRed();
		int g = color.getGreen();
		int b = color.getBlue();
		int a = color.getAlpha();
		a += value;
		a = a > 255 ? 255 : a;
		a = a < 0 ? 0 : a;
		return new Color(r, g, b, a);
	}

	/**
	 * 指定されたカラーオブジェクトの色調を指定値分変更します。<br>
	 * 正数を指定することで階調が強調され、負数を指定することで階調が弱くなります。<br>
	 * @param color 対象カラーオブジェクト
	 * @param red 赤階調
	 * @param green 緑階調
	 * @param blue 青階調
	 * @return 階調を変更されたカラーオブジェクト
	 */
	public static Color tonness(Color color, int red, int green, int blue) {
		if (color == null) {
			return null;
		}
		int r = color.getRed();
		int g = color.getGreen();
		int b = color.getBlue();
		int a = color.getAlpha();
		r += red;
		g += green;
		b += blue;
		r = r > 255 ? 255 : r;
		g = g > 255 ? 255 : g;
		b = b > 255 ? 255 : b;
		r = r < 0 ? 0 : r;
		g = g < 0 ? 0 : g;
		b = b < 0 ? 0 : b;
		return new Color(r, g, b, a);
	}

	/**
	 * 指定されたカラーオブジェクトの色調を指定値分変更します。<br>
	 * 正数を指定することで階調が強調され、負数を指定することで階調が弱くなります。<br>
	 * @param color 対象カラーオブジェクト
	 * @param hue 色相
	 * @param saturation 彩度
	 * @param brightness 明度
	 * @return 階調を変更されたカラーオブジェクト
	 */
	public static Color tonness(Color color, float hue, float saturation, float brightness) {
		if (color == null) {
			return null;
		}
		float[] hsb = getHSB(color);

		hsb[0] += hue;
		hsb[1] += saturation;
		hsb[2] += brightness;

		hsb[0] = hsb[0] > 1.0f ? 1.0f : hsb[0] < 0.0f ? 0.0f : hsb[0];
		hsb[1] = hsb[1] > 1.0f ? 1.0f : hsb[1] < 0.0f ? 0.0f : hsb[1];
		hsb[2] = hsb[2] > 1.0f ? 1.0f : hsb[2] < 0.0f ? 0.0f : hsb[2];

		Color render = Color.getHSBColor(hsb[0], hsb[1], hsb[2]);

		return new Color(render.getRed(), render.getGreen(), render.getBlue(), color.getAlpha());
	}

	/**
	 * 指定されたカラーオブジェクトをマージします。<br>
	 * @param color1 カラー
	 * @param color2 カラー
	 * @param alpha 透明度(0.0f～1.0f)
	 * @param percent マージ割合(0.0f～1.0f)
	 * @return マージされたカラーオブジェクト
	 */
	public static Color merge(Color color1, Color color2, float alpha, float percent) {
		if (color1 == null && color2 == null) {
			return null;
		} else if (color1 == null) {
			return color2;
		} else if (color2 == null) {
			return color1;
		} else {

			int red = (int) (((color1.getRed() * 1.0f - percent) + (color2.getRed() * percent)) / 2);
			int green = (int) (((color1.getGreen() * 1.0f - percent) + (color2.getGreen() * percent)) / 2);
			int blue = (int) (((color1.getBlue() * 1.0f - percent) + (color2.getBlue() * percent)) / 2);
			int iAlpha = (int) (alpha * 255);
			red = red > 255 ? 255 : red;
			green = green > 255 ? 255 : green;
			blue = blue > 255 ? 255 : blue;
			iAlpha = iAlpha > 255 ? 255 : iAlpha;
			return new Color(red, green, blue, iAlpha);
		}
	}

	/**
	 * 指定されたカラーオブジェクトをマージします。<br>
	 * @param color1 カラー
	 * @param color2 カラー
	 * @return マージされたカラーオブジェクト
	 */
	public static Color merge(Color color1, Color color2) {
		return merge(color1, color2, 255, 0.5f);
	}
}
