package org.phosphoresce.common.graphics.paint.util;

import java.awt.Color;

import org.phosphoresce.common.graphics.paint.GradientAlphaDefine;
import org.phosphoresce.common.graphics.paint.GradientColorDefine;
import org.phosphoresce.common.graphics.paint.LinearGradientPaint;
import org.phosphoresce.common.graphics.paint.RadialGradientPaint;
import org.phosphoresce.common.graphics.util.ColorUtil;

/**
 * グラデーションペイントユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/06/09	Kitagawa		新規作成
 *-->
 */
public final class GradientPaintUtil {

	/**
	 * コンストラクタ<br>
	 */
	private GradientPaintUtil() {
		super();
	}

	/**
	 * 2つのカラーオブジェクトと回転角から線形グラデーションを生成します。<br>
	 * @param color1 開始色
	 * @param color2 終了色
	 * @param angle 回転角
	 * @return 線形グラデーションペイントオブジェクト
	 */
	public static LinearGradientPaint createSimpleLinearGradientPaint(Color color1, Color color2, double angle) {
		GradientColorDefine colorDefine = new GradientColorDefine(color1, 0.0f);
		colorDefine.add(new GradientColorDefine(color2, 1.0f));

		GradientAlphaDefine alphaDefine = new GradientAlphaDefine((float) color1.getAlpha() / 255f, 0.0f);
		alphaDefine.add(new GradientAlphaDefine((float) color2.getAlpha() / 255f, 1.0f));

		LinearGradientPaint paint = new LinearGradientPaint(colorDefine, alphaDefine, angle);
		return paint;
	}

	/**
	 * 2つのカラーオブジェクトから回転角0度の線形グラデーションを生成します。<br>
	 * @param color1 開始色
	 * @param color2 終了色
	 * @return 線形グラデーションペイントオブジェクト
	 */
	public static LinearGradientPaint createSimpleLinearGradientPaint(Color color1, Color color2) {
		return createSimpleLinearGradientPaint(color1, color2, 0.0d);
	}

	/**
	 * 2つのカラーオブジェクトと円形グラデーションを生成します。<br>
	 * @param color1 開始色
	 * @param color2 終了色
	 * @param strech ストレッチフラグ
	 * @return 線形グラデーションペイントオブジェクト
	 */
	public static RadialGradientPaint createSimpleRadialGradientPaint(Color color1, Color color2, boolean strech) {
		GradientColorDefine colorDefine = new GradientColorDefine(color1, 0.0f);
		colorDefine.add(new GradientColorDefine(color2, 1.0f));

		GradientAlphaDefine alphaDefine = new GradientAlphaDefine((float) color1.getAlpha() / 255f, 0.0f);
		alphaDefine.add(new GradientAlphaDefine((float) color2.getAlpha() / 255f, 1.0f));

		RadialGradientPaint paint = new RadialGradientPaint(colorDefine, alphaDefine);
		return paint;
	}

	/**
	 * 2つのカラーオブジェクトと円形グラデーションを生成します。<br>
	 * @param color1 開始色
	 * @param color2 終了色
	 * @return 線形グラデーションペイントオブジェクト
	 */
	public static RadialGradientPaint createSimpleRadialGradientPaint(Color color1, Color color2) {
		return createSimpleRadialGradientPaint(color1, color2, false);
	}

	/**
	 * 指定された色をベースとしたグロッシーグラデーションペイントを生成します。<br>
	 * @param color ベース色
	 * @return 線形グラデーションペイントオブジェクト
	 */
	public static LinearGradientPaint createGlossyLinearGradientPaint(Color color) {
		float[] hsb = ColorUtil.getHSB(color);

		/*
		 * Sample Glossy Gradient Value
		 * (1) Locate:  0% - RGB( 37, 84,138) / HSB(212, 73, 54) / HSB(0.5888889f, 0.7300000f, 0.5400000f) / #25548A
		 * (2) Locate: 29% - RGB( 75,184,240) / HSB(200, 69, 94) / HSB(0.5555556f, 0.6900000f, 0.9400000f) / #4BB8F0
		 * (3) Locate: 52% - RGB( 58,131,194) / HSB(208, 70, 76) / HSB(0.5777778f, 0.7000000f, 0.7600000f) / #3A83C2
		 * (4) Locate: 52% - RGB( 96,152,199) / HSB(207, 52, 78) / HSB(0.5750000f, 0.5200000f, 0.7800000f) / #6098C7
		 * (5) Locate:100% - RGB(207,220,233) / HSB(210, 11, 91) / HSB(0.5833333f, 0.1100000f, 0.9100000f) / #CFDCE9
		 */

		// グラデーション切り替え部分を中心色とする
		//Color color1 = ColorUtil.tonness(color, +0.0111111f, +0.0300000f, -0.2200000f);
		//Color color2 = ColorUtil.tonness(color, -0.0222222f, -0.0100000f, +0.1800000f);
		//Color color3 = ColorUtil.tonness(color, +0.0000000f, +0.0000000f, +0.0000000f);
		//Color color4 = ColorUtil.tonness(color, -0.0027778f, -0.1800000f, +0.0200000f);
		//Color color5 = ColorUtil.tonness(color, +0.0055555f, -0.5900000f, +0.1500000f);
		Color color1 = ColorUtil.tonness(color, +0.0111111f, +0.0000000f, -0.2200000f);
		Color color2 = ColorUtil.tonness(color, -0.0222222f, -0.0000000f, +0.1800000f);
		Color color3 = ColorUtil.tonness(color, +0.0000000f, +0.0000000f, +0.0000000f);
		Color color4 = ColorUtil.tonness(color, -0.0027778f, -0.0000000f, +0.0200000f);
		Color color5 = ColorUtil.tonness(color, +0.0055555f, -0.0000000f, +0.1500000f);

		// カラーグラデーション生成
		GradientColorDefine colorDefine = new GradientColorDefine(color1, 0.0f);
		colorDefine.add(new GradientColorDefine(color2, 0.2900000f));
		colorDefine.add(new GradientColorDefine(color3, 0.5200000f));
		colorDefine.add(new GradientColorDefine(color4, 0.5200001f));
		colorDefine.add(new GradientColorDefine(color5, 1.0000000f));

		// アルファグラデーション生成
		GradientAlphaDefine alphaDefine = new GradientAlphaDefine(1.0f, 0.0f);
		alphaDefine.add(new GradientAlphaDefine(1.0f, 1.0f));

		// 線形グラデーション生成
		LinearGradientPaint paint = new LinearGradientPaint(colorDefine, alphaDefine, 90.0d, 1.0f, 2.0f);

		return paint;
	}
}
