package org.phosphoresce.common.graphics.paint;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GradientPaint;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;

import org.phosphoresce.common.graphics.util.ImageGraphicsUtil;

/**
 * 線形ペイントコンテキストクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/06/08	Kitagawa		新規作成
 *-->
 */
public class RadialGradientPaintContext extends AbstractGradientPaintContext {

	/** ストレッチフラグ */
	private boolean strech = false;

	/**
	 * コンストラクタ<br>
	 * @param model Paintデータを受け取るColorModelオブジェクト
	 * @param deviceBounds 描画されるグラフィックスプリミティブのデバイス空間でのバウンディングボックス
	 * @param userBounds 描画されるグラフィックスプリミティブのユーザ空間でのバウンディングボックス
	 * @param transform ユーザ空間からデバイス空間へのAffineTransform
	 * @param hints コンテキストオブジェクトが描画の選択肢を選択するときに使用するヒント
	 * @param colorDefine グラデーションカラー定義
	 * @param alphaDefine グラデーションアルファ定義
	 * @param smoothness 滑らかさ(0.0f～1.0f)
	 * @param raito 比率(0.0f～100.0f)
	 * @param strech ストレッチフラグ
	 */
	public RadialGradientPaintContext(ColorModel model, Rectangle deviceBounds, Rectangle2D userBounds, AffineTransform transform, RenderingHints hints, //
			GradientColorDefine colorDefine, GradientAlphaDefine alphaDefine, float smoothness, float raito, //
			boolean strech) {
		super(model, deviceBounds, userBounds, transform, hints, 0.0d, colorDefine, alphaDefine, smoothness, raito);
		this.strech = strech;
	}

	/**
	 * 指定されたグラフィックスオブジェクトに対して回転角0度時のグラデーションイメージを描画します。<br>
	 * @param graphics グラフィックスオブジェクト
	 * @param dimension 描画サイズ
	 * @param gradient グラデーション定義
	 * @see org.phosphoresce.common.graphics.paint.AbstractGradientPaintContext#fillGradientImage(java.awt.Graphics2D, java.awt.Dimension, org.phosphoresce.common.graphics.paint.GradientDefine)
	 */
	protected void fillGradientImage(Graphics2D graphics, Dimension dimension, GradientDefine gradient) {
		// グラデーション定義取得
		GradientDefine[] defines = (GradientDefine[]) gradient.getAllDefines();

		final int QUALITY = 1;

		/*
		 * 線形グラデーション作成
		 */
		// 線形グラデーションサイズ算出
		int lineImageWidth = (int) ((dimension.getWidth() > dimension.getHeight() ? dimension.getWidth() : dimension.getHeight()) * getRatio() * QUALITY);
		int lineImageHeight = 1;

		// 線形グラデーションイメージ生成
		BufferedImage lineImage = ImageGraphicsUtil.createEmptyBufferedImage(lineImageWidth, lineImageHeight);

		// 線形グラデーションイメージ描画
		Color headColor = null;
		Color tailColor = null;
		{
			Graphics2D g = (Graphics2D) lineImage.getGraphics();
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

			for (int i = 0; i <= defines.length - 2; i++) {
				GradientDefine define1 = defines[i];
				GradientDefine define2 = defines[i + 1];
				Color color1 = define1.getGradientColor();
				Color color2 = define2.getGradientColor();

				// グラデーション描画後の余白ベタ塗用のカラーを退避
				if (i == 0) {
					headColor = color1;
					tailColor = color2;
				} else {
					tailColor = color2;
				}

				// 描画情報算出
				float x1 = define1.getLocation() * lineImageWidth;
				float y1 = 0.0f;
				float x2 = define2.getLocation() * lineImageWidth;
				float y2 = 0.0f;
				float smooth = ((x2 - x1) / 8) * getSmoothness();

				// 各種描画オブジェクト生成
				Point2D point1 = new Point2D.Float(x1 + smooth, y1);
				Point2D point2 = new Point2D.Float(x2 - smooth, y2);
				Shape shape = new Rectangle2D.Float(x1, 0, x2, 1);
				GradientPaint paint = new GradientPaint(point1, color1, point2, color2);

				// 線形グラデーションイメージ描画
				g.setPaint(paint);
				g.fill(shape);
			}
		}

		/*
		 * バッファイメージ作成
		 */
		// バッファイメージサイズ算出
		int bufferImageWidth = (int) (dimension.getWidth() * getRatio() * QUALITY);
		int bufferImageHeight = (int) (dimension.getHeight() * getRatio() * QUALITY);
		bufferImageWidth = bufferImageWidth <= 0 ? 1 : bufferImageWidth;
		bufferImageHeight = bufferImageHeight <= 0 ? 1 : bufferImageHeight;

		// バッファイメージ生成
		BufferedImage bufferImage = ImageGraphicsUtil.createEmptyBufferedImage(bufferImageWidth, bufferImageHeight);

		// バッファイメージ描画
		{
			Graphics2D g = (Graphics2D) bufferImage.getGraphics();
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

			// 余白フィルペイント
			{
				g.setColor(tailColor);
				g.fillRect(0, 0, bufferImage.getWidth(), bufferImage.getHeight());
			}

			for (int i = 0; i <= lineImageWidth - 1; i++) {
				int x = bufferImageWidth > bufferImageHeight ? i / 2 : (int) (i * ((double) bufferImageWidth / (double) bufferImageHeight)) / 2;
				int y = bufferImageWidth < bufferImageHeight ? i / 2 : (int) (i * ((double) bufferImageHeight / (double) bufferImageWidth)) / 2;
				int width = bufferImageWidth - (x * 2);
				int height = bufferImageHeight - (y * 2);
				//int[] pixel = lineImage.getRaster().getPixel(i, 0, new int[4]);
				int[] pixel = lineImage.getRaster().getPixel(lineImageWidth - 1 - i, 0, new int[4]);
				Color color = new Color(pixel[0], pixel[1], pixel[2], pixel[3]);
				g.setColor(color);
				if (strech) {
					g.fillArc(x + 1, y + 1, width - 2, height - 2, 0, 360);
				} else {
					int size = width > height ? height : width;
					g.fillArc(x + 1 + ((width - size) / 2), y + 1 + ((height - size) / 2), size, size, 0, 360);
				}
			}
		}

		/*
		 * グラフィックスオブジェクト描画
		 */
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		// 余白を塗りつぶす
		{
			graphics.setColor(tailColor);
			graphics.fillRect(0, 0, dimension.width, dimension.height);
		}

		int x = -(dimension.width - bufferImage.getWidth() / QUALITY) / 2;
		int y = -(dimension.height - bufferImage.getHeight() / QUALITY) / 2;
		int width = bufferImage.getWidth() / QUALITY;
		int height = bufferImage.getHeight() / QUALITY;
		graphics.drawImage(bufferImage, x, y, width, height, null);
		graphics.drawImage(bufferImage, x, y, width, height, null);
	}
}
