package org.phosphoresce.common.graphics.paint;

import java.awt.Graphics2D;
import java.awt.PaintContext;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.DirectColorModel;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.lang.ref.WeakReference;

import org.phosphoresce.common.graphics.util.ImageGraphicsUtil;

/**
 * ペイントコンテキスト上位抽象クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/06/09	Kitagawa		新規作成
 *-->
 */
public abstract class AbstractPaintContext implements PaintContext {

	/** カラーモデルオブジェクト */
	private ColorModel model = null;

	/** デバイス空間バウンディングボックス */
	private Rectangle deviceBounds;

	/** ユーザー空間バウンディングボックス */
	private Rectangle2D userBounds;

	/** ユーザ空間からデバイス空間へのAffineTransform */
	private AffineTransform transform;

	/** コンテキストオブジェクトが描画の選択肢を選択するときに使用するヒント */
	private RenderingHints hints;

	/** 角度 */
	private double angle;

	/** ペイントイメージ */
	private BufferedImage paintImage;

	/** アクティブRaster */
	private WritableRaster raster;

	/** キャッシュカラーモデル */
	private static ColorModel cachedModel;

	/** キャッシュRaster */
	private static WeakReference cachedRaster;

	/**
	 * キャッシュされているRasterオブジェクトを取得します。<br>
	 * 当メソッドはjava.awt.GradientPaintContextよりレスポンスを考慮して流用されました。<br>
	 * @param model カラーモデルオブジェクト
	 * @param width Raster幅
	 * @param height Raster高さ
	 * @return
	 */
	private final static synchronized Raster getCachedRaster(ColorModel model, int width, int height) {
		if (model == cachedModel) {
			if (cachedRaster != null) {
				Raster raster = (Raster) cachedRaster.get();
				if (raster != null && raster.getWidth() >= width && raster.getHeight() >= height) {
					cachedRaster = null;
					return raster;
				}
			}
		}
		return model.createCompatibleWritableRaster(width, height);
	}

	/**
	 * Rasterキャッシュを設定します。<br>
	 * @param model カラーモデルオブジェクト
	 * @param raster Rasterオブジェクト
	 */
	private final static synchronized void setCachedRaster(ColorModel model, Raster raster) {
		if (cachedRaster != null) {
			Raster cached = (Raster) cachedRaster.get();
			if (cached != null) {
				int cWidth = cached.getWidth();
				int cHeight = cached.getHeight();
				int iWidth = raster.getWidth();
				int iHeight = raster.getHeight();
				if (cWidth >= iWidth && cHeight >= iHeight) {
					return;
				}
				if (cWidth * cHeight >= iWidth * iHeight) {
					return;
				}
			}
		}
		cachedModel = model;
		cachedRaster = new WeakReference(raster);
	}

	/**
	 * コンストラクタ<br>
	 */
	private AbstractPaintContext() {
		super();
	}

	/**
	 * コンストラクタ<br>
	 * @param model Paintデータを受け取るColorModelオブジェクト
	 * @param deviceBounds 描画されるグラフィックスプリミティブのデバイス空間でのバウンディングボックス
	 * @param userBounds 描画されるグラフィックスプリミティブのユーザ空間でのバウンディングボックス
	 * @param transform ユーザ空間からデバイス空間へのAffineTransform
	 * @param hints コンテキストオブジェクトが描画の選択肢を選択するときに使用するヒント
	 * @param angle 回転角(0～360)
	 */
	public AbstractPaintContext(ColorModel model, Rectangle deviceBounds, Rectangle2D userBounds, AffineTransform transform, RenderingHints hints, double angle) {
		super();
		//this.model = model;
		this.model = new DirectColorModel(32, 0x00ff0000, 0x0000ff00, 0x000000ff, 0xff000000);
		this.deviceBounds = deviceBounds;
		this.userBounds = userBounds;
		this.transform = transform;
		this.hints = hints;
		this.angle = angle;
	}

	/**
	 * 操作に割り当てられているリソースを解放します。<br>
	 * @see java.awt.PaintContext#dispose()
	 */
	public void dispose() {
		if (raster != null) {
			setCachedRaster(model, raster);
			raster = null;
		}
	}

	/**
	 * 出力ColorModelを取得します。<br>
	 * @return Colorモデルオブジェクト
	 * @see java.awt.PaintContext#getColorModel()
	 */
	public final ColorModel getColorModel() {
		return model;
	}

	/**
	 * ペイントイメージを新たに生成します。<br>
	 * イメージ自体の描画処理はサブクラスによって実装されるfillPaintImageに委譲します。<br>
	 * @return ペイントイメージ
	 */
	protected final BufferedImage createPaintImage() {
		// 回転後描画バウンディングボックス生成
		Rectangle bounds = createRotatedRectangle((int) userBounds.getWidth(), (int) userBounds.getHeight(), angle);

		// ペイントイメージ生成
		BufferedImage buffer = ImageGraphicsUtil.createEmptyBufferedImage(bounds.getSize());
		fillPaintImage(buffer);

		// グラデーションイメージマージ
		BufferedImage image = ImageGraphicsUtil.createEmptyBufferedImage((int) userBounds.getWidth(), (int) userBounds.getHeight());
		Graphics2D graphics = (Graphics2D) image.getGraphics();
		graphics.setTransform(AffineTransform.getRotateInstance(Math.toRadians(-angle), userBounds.getWidth() / 2, userBounds.getHeight() / 2));
		graphics.drawImage(buffer, bounds.x, bounds.y, bounds.width, bounds.height, null);

		return image;
	}

	/**
	 * 指定されたイメージオブジェクトに対して回転角0度時のグラデーションイメージを描画します。<br>
	 * <br>
	 * ここで指定されるイメージサイズは回転後のバウンディングボックスサイズが考慮されたサイズとなっており、
	 * 実際の描画バウンディングボックスサイズとは異なることに注意してください。<br>
	 * @param image 描画バッファリングイメージ
	 */
	protected abstract void fillPaintImage(BufferedImage image);

	/**
	 * グラフィックス操作用に生成された色を格納するRasterを取得します。<br>
	 * @param x X座標
	 * @param y Y座標
	 * @param width デバイス空間での領域の幅
	 * @param height デバイス空間での領域の高さ
	 * @return グラフィックス操作用に生成された色を格納するRasterオブジェクト
	 * @see java.awt.PaintContext#getRaster(int, int, int, int)
	 */
	public final Raster getRaster(int x, int y, int width, int height) {
		//WritableRaster raster = getColorModel().createCompatibleWritableRaster(width, height);
		WritableRaster raster = this.raster;
		if (raster == null || raster.getWidth() < width || raster.getHeight() < height) {
			raster = (WritableRaster) getCachedRaster(model, width, height);
			this.raster = raster;
		}
		int size = width * height * 4;
		int[] data = new int[size];
		data = getPaintImage().getRaster().getPixels(x - deviceBounds.x, y - deviceBounds.y, width, height, data);
		raster.setPixels(0, 0, width, height, data);
		return raster;
	}

	/**
	 * 指定された長方形範囲を指定角回転させた際のバウンズボックスを取得します。<br>
	 * このバウンディングボックス範囲に対して、サブクラスにペイントイメージを描画させます。<br>
	 * @param width 幅
	 * @param height 高さ
	 * @param angle 回転角
	 * @return バウンズボックス
	 */
	protected final Rectangle createRotatedRectangle(int width, int height, double angle) {
		// ラジアン変換
		double radian = Math.toRadians(-angle);

		// 回転後の必要サイズ計算
		double renderWidth = Math.abs((width * Math.cos(radian))) + Math.abs((height * Math.sin(radian)));
		double renderHeight = Math.abs((height * Math.cos(radian))) + Math.abs((width * Math.sin(radian)));

		// 不動少数によるずれを補正
		renderWidth += 2;
		renderHeight += 2;

		// サイズ差取得
		double widthDiff = renderWidth - width;
		double heightDiff = renderHeight - height;

		// 範囲ボックス生成
		return new Rectangle((int) (-widthDiff / 2), (int) (-heightDiff / 2), (int) renderWidth, (int) renderHeight);
	}

	/**
	 * ペイントイメージを取得します。<br>
	 * @return ペイントイメージ
	 */
	protected BufferedImage getPaintImage() {
		if (paintImage == null) {
			paintImage = createPaintImage();
		}
		return paintImage;
	}

	/**
	 * デバイス空間バウンディングボックスを取得します。<br>
	 * @return デバイス空間バウンディングボックス
	 */
	public final Rectangle getDeviceBounds() {
		return deviceBounds;
	}

	/**
	 * ユーザー空間バウンディングボックスを取得します。<br>
	 * @return ユーザー空間バウンディングボックス
	 */
	public final Rectangle2D getUserBounds() {
		return userBounds;
	}

	/**
	 * コンテキストオブジェクトが描画の選択肢を選択するときに使用するヒントを取得します。<br>
	 * @return コンテキストオブジェクトが描画の選択肢を選択するときに使用するヒント
	 */
	public final RenderingHints getHints() {
		return hints;
	}

	/**
	 * ユーザ空間からデバイス空間へのAffineTransformを取得します。<br>
	 * @return ユーザ空間からデバイス空間へのAffineTransform
	 */
	public final AffineTransform getTransform() {
		return transform;
	}

	/**
	 * 角度を取得します。<br>
	 * @return 角度
	 */
	public final double getAngle() {
		return angle;
	}
}
