﻿using System;
using System.Threading.Tasks;
using System.Windows.Input;
using Progressive.PecaStarter.Model;
using System.Threading;

namespace Progressive.PecaStarter.ViewModel.Command
{
    public abstract class PeercastCommandBase : ICommand
    {
        protected readonly Peercast peercast;
        protected readonly MainWindowViewModel viewModel;

        public PeercastCommandBase(Peercast peercast, MainWindowViewModel viewModel)
        {
            this.peercast = peercast;
            this.viewModel = viewModel;
        }

        #region ICommand メンバー
        public event EventHandler CanExecuteChanged
        {
            add { CommandManager.RequerySuggested += value; }
            remove { CommandManager.RequerySuggested -= value; }
        }

        public bool CanExecute(object parameter)
        {
            if (viewModel == null || viewModel.IsBusy)
            {
                return false;
            }
            return CanExecute();
        }
        public async void Execute(object parameter)
        {
            while (viewModel.IsBusy)
            {
                Thread.Sleep(1000);
            }
            viewModel.IsBusy = true;
            try
            {
                await Execute();
            }
            finally
            {
                viewModel.IsBusy = false;
                GC.Collect();
            }
        }
        #endregion

        protected abstract bool CanExecute();
        protected abstract Task Execute();

        protected string GetErrorMessage(ResultInfo<string> info)
        {
            switch (info.Result)
            {
                case Result.YellowPagesFailure:
                    return "イエローページサーバのエラー\nエラーメッセージ: " + info.Value;
                case Result.ConnectFailure:
                    return string.IsNullOrEmpty(info.Value) ? "接続に失敗" : "接続に失敗\n" + info.Value;
                default:
                    return GetErrorMessage(info.Result);
            }
        }
        protected string GetErrorMessage(Result result)
        {
            switch (result)
            {
                case Result.ChannelDuplicated:
                    return "チャンネルの重複";
                case Result.ConnectFailure:
                    return "接続に失敗";
                case Result.CreatingChannelFailure:
                    return "チャンネルの作成に失敗";
                case Result.UpdatingChannelFailure:
                    return "チャンネル情報の更新に失敗";
                case Result.DeletingChannelFailure:
                    return "チャンネルの切断に失敗";
                case Result.GettingSettingsHtmlFailure:
                    return "設定へのアクセスに失敗";
                default:
                    throw new NotSupportedException();
            }
        }
    }
}
