package jp.sourceforge.pdt_tools.codeChecker.visitor;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.sourceforge.pdt_tools.codeChecker.CodeCheckerPlugin;
import jp.sourceforge.pdt_tools.codeChecker.ProblemMarker;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.dltk.ast.ASTListNode;
import org.eclipse.dltk.ast.ASTNode;
import org.eclipse.dltk.ast.Modifiers;
import org.eclipse.dltk.ast.declarations.MethodDeclaration;
import org.eclipse.dltk.ast.declarations.ModuleDeclaration;
import org.eclipse.dltk.ast.declarations.TypeDeclaration;
import org.eclipse.dltk.ast.expressions.Expression;
import org.eclipse.dltk.ast.references.ConstantReference;
import org.eclipse.dltk.ast.references.SimpleReference;
import org.eclipse.dltk.ast.references.TypeReference;
import org.eclipse.dltk.ast.references.VariableReference;
import org.eclipse.dltk.ast.statements.Block;
import org.eclipse.dltk.ast.statements.Statement;
import org.eclipse.dltk.compiler.problem.ProblemSeverities;
import org.eclipse.dltk.core.IBuildpathEntry;
import org.eclipse.dltk.core.IField;
import org.eclipse.dltk.core.IMethod;
import org.eclipse.dltk.core.IModelElement;
import org.eclipse.dltk.core.IParameter;
import org.eclipse.dltk.core.IProjectFragment;
import org.eclipse.dltk.core.IScriptProject;
import org.eclipse.dltk.core.ISourceModule;
import org.eclipse.dltk.core.ISourceRange;
import org.eclipse.dltk.core.IType;
import org.eclipse.dltk.core.ITypeHierarchy;
import org.eclipse.dltk.core.ModelException;
import org.eclipse.dltk.core.SourceParserUtil;
import org.eclipse.dltk.core.environment.EnvironmentPathUtils;
import org.eclipse.dltk.core.index2.search.ISearchEngine.MatchRule;
import org.eclipse.dltk.core.search.IDLTKSearchScope;
import org.eclipse.dltk.core.search.SearchEngine;
import org.eclipse.dltk.internal.core.SourceField;
import org.eclipse.dltk.internal.core.SourceMethod;
import org.eclipse.dltk.internal.core.SourceType;
import org.eclipse.dltk.ti.types.IEvaluatedType;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.osgi.util.NLS;
import org.eclipse.php.core.compiler.PHPFlags;
import org.eclipse.php.internal.core.PHPVersion;
import org.eclipse.php.internal.core.compiler.ast.nodes.ArrayCreation;
import org.eclipse.php.internal.core.compiler.ast.nodes.ArrayElement;
import org.eclipse.php.internal.core.compiler.ast.nodes.ArrayVariableReference;
import org.eclipse.php.internal.core.compiler.ast.nodes.Assignment;
import org.eclipse.php.internal.core.compiler.ast.nodes.BackTickExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.BreakStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.CastExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.CatchClause;
import org.eclipse.php.internal.core.compiler.ast.nodes.ClassDeclaration;
import org.eclipse.php.internal.core.compiler.ast.nodes.ClassInstanceCreation;
import org.eclipse.php.internal.core.compiler.ast.nodes.CloneExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.Comment;
import org.eclipse.php.internal.core.compiler.ast.nodes.ConditionalExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.ConstantDeclaration;
import org.eclipse.php.internal.core.compiler.ast.nodes.ContinueStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.DeclareStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.Dispatch;
import org.eclipse.php.internal.core.compiler.ast.nodes.DoStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.EchoStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.EmptyStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.ExpressionStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.FieldAccess;
import org.eclipse.php.internal.core.compiler.ast.nodes.ForEachStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.ForStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.FormalParameter;
import org.eclipse.php.internal.core.compiler.ast.nodes.FormalParameterByReference;
import org.eclipse.php.internal.core.compiler.ast.nodes.FullyQualifiedReference;
import org.eclipse.php.internal.core.compiler.ast.nodes.GlobalStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.GotoLabel;
import org.eclipse.php.internal.core.compiler.ast.nodes.GotoStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.IfStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.IgnoreError;
import org.eclipse.php.internal.core.compiler.ast.nodes.Include;
import org.eclipse.php.internal.core.compiler.ast.nodes.InfixExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.InstanceOfExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.InterfaceDeclaration;
import org.eclipse.php.internal.core.compiler.ast.nodes.LambdaFunctionDeclaration;
import org.eclipse.php.internal.core.compiler.ast.nodes.ListVariable;
import org.eclipse.php.internal.core.compiler.ast.nodes.NamespaceDeclaration;
import org.eclipse.php.internal.core.compiler.ast.nodes.NamespaceReference;
import org.eclipse.php.internal.core.compiler.ast.nodes.PHPCallArgumentsList;
import org.eclipse.php.internal.core.compiler.ast.nodes.PHPCallExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.PHPDocBlock;
import org.eclipse.php.internal.core.compiler.ast.nodes.PHPDocTag;
import org.eclipse.php.internal.core.compiler.ast.nodes.PHPFieldDeclaration;
import org.eclipse.php.internal.core.compiler.ast.nodes.PHPMethodDeclaration;
import org.eclipse.php.internal.core.compiler.ast.nodes.PostfixExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.PrefixExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.Quote;
import org.eclipse.php.internal.core.compiler.ast.nodes.ReferenceExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.ReflectionArrayVariableReference;
import org.eclipse.php.internal.core.compiler.ast.nodes.ReflectionCallExpression;
import org.eclipse.php.internal.core.compiler.ast.nodes.ReflectionStaticMethodInvocation;
import org.eclipse.php.internal.core.compiler.ast.nodes.ReflectionVariableReference;
import org.eclipse.php.internal.core.compiler.ast.nodes.ReturnStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.Scalar;
import org.eclipse.php.internal.core.compiler.ast.nodes.StaticConstantAccess;
import org.eclipse.php.internal.core.compiler.ast.nodes.StaticDispatch;
import org.eclipse.php.internal.core.compiler.ast.nodes.StaticFieldAccess;
import org.eclipse.php.internal.core.compiler.ast.nodes.StaticMethodInvocation;
import org.eclipse.php.internal.core.compiler.ast.nodes.StaticStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.SwitchCase;
import org.eclipse.php.internal.core.compiler.ast.nodes.SwitchStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.ThrowStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.TryStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.UnaryOperation;
import org.eclipse.php.internal.core.compiler.ast.nodes.UsePart;
import org.eclipse.php.internal.core.compiler.ast.nodes.UseStatement;
import org.eclipse.php.internal.core.compiler.ast.nodes.WhileStatement;
import org.eclipse.php.internal.core.includepath.IncludePath;
import org.eclipse.php.internal.core.includepath.IncludePathManager;
import org.eclipse.php.internal.core.language.LanguageModelInitializer;
import org.eclipse.php.internal.core.language.PHPVariables;
import org.eclipse.php.internal.core.model.PhpModelAccess;
import org.eclipse.php.internal.core.project.ProjectOptions;
import org.eclipse.php.internal.core.typeinference.BindingUtility;
import org.eclipse.php.internal.core.typeinference.PHPModelUtils;
import org.eclipse.php.internal.ui.editor.highlighter.ModelUtils;

public class CodeVisitor extends BaseVisitor {

	private ISourceModule sourceModule;
	private PHPVersion phpVersion;
	private Options options;
	private List<ProblemMarker> problems;
	private int errors;
	private int warnings;
	private BindingUtility bindingUtility;

	private static final String[] LanguageConstructs = { "die", "empty",
			"exit", "eval", "isset", "print", "unset" };
	private Map<String, Boolean> globalMap;
	private Map<String, Boolean> localMap;
	private boolean unusedLocal;
	private int currentScope;
	private static final int SCOPE_MAIN = 0;
	private static final int SCOPE_CLASS = 1;
	private static final int SCOPE_INTERFACE = 2;

	public CodeVisitor(ISourceModule module) {
		super();
		sourceModule = module;
		phpVersion = ProjectOptions.getPhpVersion(sourceModule);
		options = new Options(sourceModule);
		problems = new LinkedList<ProblemMarker>();
		errors = 0;
		warnings = 0;
		bindingUtility = new BindingUtility(sourceModule);
		globalMap = new HashMap<String, Boolean>();
		localMap = null;
		unusedLocal = false;
		currentScope = SCOPE_MAIN;
	}

	public List<ProblemMarker> getProblems() {
		return problems;
	}

	public int getErrors() {
		return errors;
	}

	public int getWarnings() {
		return warnings;
	}

	private void createMarker(int severity, String msg, String[] info,
			ISourceRange range) {
		createMarker(severity, msg, info, range.getOffset(), range.getOffset()
				+ range.getLength());
	}

	private void createMarker(int severity, String msg, String[] info,
			ASTNode node) {
		createMarker(severity, msg, info, node.sourceStart(), node.sourceEnd());
	}

	private void createMarker(int severity, String msg, String[] info,
			int start, int end) {
		int markerSeverity;
		if (severity == ProblemSeverities.Error.ordinal()) {
			markerSeverity = IMarker.SEVERITY_ERROR;
			errors++;
		} else if (severity == ProblemSeverities.Warning.ordinal()) {
			markerSeverity = IMarker.SEVERITY_WARNING;
			warnings++;
		} else {
			return;
		}
		if (info != null && info.length > 0) {
			msg = NLS.bind(msg, info);
		}
		try {
			IDocument document = new Document(sourceModule.getSource());
			int line = document.getLineOfOffset(start) + 1;
			ProblemMarker problemMarker = new ProblemMarker(markerSeverity,
					msg, start, end, line, sourceModule.getResource());
			for (ProblemMarker problem : problems) {
				if (problem.equals(problemMarker)) {
					return;
				}
			}
			problems.add(problemMarker);
		} catch (ModelException e) {
			CodeCheckerPlugin.log(e);
		} catch (BadLocationException e) {
			CodeCheckerPlugin.log(e);
		}
	}

	private boolean detect(int option) {
		return option != ProblemSeverities.Ignore.ordinal();
	}

	@Override
	public boolean visit(ArrayCreation s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ArrayElement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ArrayVariableReference s) throws Exception {
		if (detect(options.unused_local) && !unusedLocal) {
			unusedLocal();
			unusedLocal = true;
		}
		if (detect(options.uninitialized_variable)) {
			uninitializedVariable(s);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(Assignment s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(BackTickExpression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(BreakStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(CastExpression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(CatchClause s) throws Exception {
		ClassContext classContext = new ClassContext(s.getClassName());
		if (detect(options.unresolved_types)) {
			unresolvedClass(classContext);
		}
		if (detect(options.deprecation)) {
			deprecated(classContext);
		}
		if (detect(options.misc_error)) {
			errorneousClass(classContext);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(ConstantDeclaration s) throws Exception {
		if (detect(options.misc_compile_error)) {
			inspectClassConstants(s);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(ClassDeclaration s) throws Exception {
		currentScope = SCOPE_CLASS;
		ClassContext classContext = new ClassContext(s.getRef());
		if (detect(options.misc_compile_error)) {
			if (!checkClassName(s.getRef())) {
				return false;
			}
		}
		TypeReference superClass = s.getSuperClass();
		if (detect(options.unresolved_types)) {
			if (superClass != null) {
				unresolvedClass(new ClassContext(superClass));
			}
			for (TypeReference identifier : s.getInterfaceList()) {
				unresolvedInterface(new ClassContext(identifier));
			}
		}
		if (detect(options.redeclared_types)) {
			redeclaredTypes(classContext);
		}
		if (detect(options.redeclared_methods)) {
			redeclaredMethods(classContext);
		}
		if (detect(options.redeclared_fields)) {
			redeclaredFields(classContext);
		}
		if (detect(options.unused_private)) {
			classContext.scanPrivate(s);
			unusedPrivate(classContext);
		}
		if (detect(options.uninitialized_variable)) {
			classContext.scanPrivate(s);
			uninitializedPrivate(classContext);
		}
		if (superClass != null) {
			if (detect(options.misc_compile_error)) {
				inspectInheritedClass(classContext);
				inspectInheritedFields(classContext);
				inspectInheritedConstructor(classContext);
			}
			if (detect(options.misc_error)) {
				errorneousClass(classContext);
			}
		}
		if (detect(options.misc_compile_error)) {
			inspectClassFields(classContext);
		}
		validateMethods(classContext);
		return super.visit(s);
	}

	@Override
	public boolean endvisit(ClassDeclaration s) throws Exception {
		currentScope = SCOPE_MAIN;
		return super.endvisit(s);
	}

	@Override
	public boolean visit(ClassInstanceCreation s) throws Exception {
		ClassContext classContext = new ClassContext(s.getClassName());
		if (detect(options.unresolved_types)) {
			unresolvedClass(classContext);
		}
		if (detect(options.deprecation)) {
			deprecated(classContext);
		}
		if (detect(options.misc_error)) {
			errorneousClass(classContext);
			instantiateAbstractClass(classContext);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(CloneExpression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(Comment s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ConditionalExpression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ConstantReference s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ContinueStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(DeclareStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(Dispatch s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(DoStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(EchoStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(EmptyStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ExpressionStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(FieldAccess s) throws Exception {
		ClassContext classContext = new ClassContext(s.getDispatcher());
		if (detect(options.unresolved_types)) {
			unresolvedClass(classContext);
		}
		Expression expression = s.getField();
		if (detect(options.unresolved_fields)) {
			unresolvedField(classContext, expression);
		}
		if (detect(options.static_access_receiver)) {
			accessingAsNonStatic(classContext, expression);
		}
		if (detect(options.deprecation)) {
			deprecated(classContext);
			deprecated(classContext, s);
		}
		if (detect(options.misc_error)) {
			errorneousClass(classContext);
			disallowedFieldAccessType(classContext, expression);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(ForEachStatement s) throws Exception {
		if (detect(options.misc_compile_error)) {
			inspectReference(s.getKey());
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(FormalParameter s) throws Exception {
		Expression expression = s.getParameterType();
		if (expression != null) {
			ClassContext classContext = new ClassContext(expression);
			if (detect(options.unresolved_types)) {
				if (expression instanceof TypeReference
						&& ((TypeReference) expression).getName()
								.equalsIgnoreCase("array")) {
				} else {
					unresolvedClass(classContext);
				}
			}
			if (detect(options.deprecation)) {
				deprecated(classContext);
			}
			if (detect(options.misc_error)) {
				if (expression instanceof TypeReference
						&& ((TypeReference) expression).getName()
								.equalsIgnoreCase("array")) {
				} else {
					errorneousClass(classContext);
				}
			}
			if (s.getInitialization() != null) {
				if (detect(options.misc_compile_error)) {
					inspectFormalParameter(s);
				}
			}
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(FormalParameterByReference s) throws Exception {
		if (detect(options.misc_compile_error)) {
			inspectAutoloadMethod2(s);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(ForStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(GlobalStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(IfStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(IgnoreError s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(Include s) throws Exception {
		if (detect(options.include_not_found)) {
			includeNotFound(s);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(InfixExpression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(InstanceOfExpression s) throws Exception {
		ClassContext classContext = new ClassContext(s.getClassName());
		if (detect(options.unresolved_types)) {
			unresolvedClass(classContext);
		}
		if (detect(options.deprecation)) {
			deprecated(classContext);
		}
		if (detect(options.misc_error)) {
			errorneousClass(classContext);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(InterfaceDeclaration s) throws Exception {
		currentScope = SCOPE_INTERFACE;
		ClassContext classContext = new ClassContext(s.getRef());
		if (detect(options.misc_compile_error)) {
			if (!checkInterfaceName(s.getRef())) {
				return false;
			}
		}
		if (detect(options.unresolved_types)) {
			ASTListNode supers = s.getSuperClasses();
			if (supers != null) {
				for (Object child : supers.getChilds()) {
					if (child instanceof Expression) {
						unresolvedInterface(new ClassContext((Expression) child));
					}
				}
			}
		}
		if (detect(options.redeclared_types)) {
			redeclaredTypes(classContext);
		}
		if (detect(options.redeclared_methods)) {
			redeclaredMethods(classContext);
		}
		if (detect(options.unresolved_types)) {
			inspectInheritedClass(classContext);
		}
		if (detect(options.misc_compile_error)) {
			inspectClassFields(classContext);
		}
		validateInterface(classContext);
		return super.visit(s);
	}

	@Override
	public boolean endvisit(InterfaceDeclaration s) throws Exception {
		currentScope = SCOPE_MAIN;
		return super.endvisit(s);
	}

	@Override
	public boolean visit(ListVariable s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(PHPCallArgumentsList s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(PHPCallExpression s) throws Exception {
		ASTNode dispatcher = s.getReceiver();
		if (dispatcher != null) {
			ClassContext classContext = new ClassContext(
					(Expression) dispatcher);
			if (detect(options.unresolved_types)) {
				unresolvedClass(classContext);
			}
			Expression expression = s.getCallName();
			if (detect(options.unresolved_methods)) {
				unresolvedMethod(classContext, expression);
			}
			if (detect(options.deprecation)) {
				deprecated(classContext);
				deprecated(classContext, s);
			}
			if (detect(options.missing_arguments)) {
				missingArguments(classContext, s);
			}
			if (detect(options.misc_compile_error)) {
				inspectCloneMethod(classContext, expression);
			}
			if (detect(options.misc_error)) {
				errorneousClass(classContext);
				disallowedMethodAccessType(classContext, expression);
			}
		} else {
			FunctionContext functionContext = new FunctionContext(s);
			if (detect(options.unresolved_functions)) {
				functionContext.resolve();
			}
			if (detect(options.deprecation)) {
				functionContext.deprecation();
			}
			if (detect(options.missing_arguments)) {
				functionContext.arguments();
			}
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(PHPDocBlock s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(PHPDocTag s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(PHPFieldDeclaration s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(PHPMethodDeclaration s) throws Exception {
		localMap = new HashMap<String, Boolean>();
		if (currentScope == SCOPE_MAIN) {
			if (detect(options.redeclared_functions)) {
				redeclaredFunctions(s);
			}
		} else {
			if (detect(options.misc_compile_error)) {
				inspectAbstractMethodBody(s);
			}
		}
		if (detect(options.unused_parameter)) {
			unusedParameter(s);
		}
		if (detect(options.unused_local)) {
			unusedLocal(s);
		}
		return super.visit(s);
	}

	@Override
	public boolean endvisit(PHPMethodDeclaration s) throws Exception {
		localMap = null;
		return super.endvisit(s);
	}

	@Override
	public boolean visit(PostfixExpression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(PrefixExpression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(Quote s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ReferenceExpression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ReflectionArrayVariableReference s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ReflectionCallExpression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ReflectionStaticMethodInvocation s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ReflectionVariableReference s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ReturnStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(Scalar s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(SimpleReference s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(StaticConstantAccess s) throws Exception {
		ClassContext classContext = new ClassContext(s.getDispatcher());
		if (detect(options.unresolved_types)) {
			unresolvedClass(classContext);
		}
		if (detect(options.unresolved_fields)) {
			unresolvedConstant(classContext, s.getConstant());
		}
		if (detect(options.deprecation)) {
			deprecated(classContext);
		}
		if (detect(options.misc_error)) {
			errorneousClass(classContext);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(StaticDispatch s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(StaticFieldAccess s) throws Exception {
		ClassContext classContext = new ClassContext(s.getDispatcher());
		if (detect(options.unresolved_types)) {
			unresolvedClass(classContext);
		}
		Expression expression = s.getField();
		if (detect(options.unresolved_fields)) {
			unresolvedStaticField(classContext, expression);
		}
		if (detect(options.deprecation)) {
			deprecated(classContext);
			deprecated(classContext, s);
		}
		if (detect(options.misc_error)) {
			errorneousClass(classContext);
			disallowedStaticFieldAccessType(classContext, expression);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(StaticMethodInvocation s) throws Exception {
		ClassContext classContext = new ClassContext(
				(Expression) s.getReceiver());
		if (detect(options.unresolved_types)) {
			unresolvedClass(classContext);
		}
		Expression expression = s.getCallName();
		if (detect(options.misc_error)) {
			errorneousClass(classContext);
			disallowedMethodAccessType(classContext, expression);
		}
		if (detect(options.unresolved_methods)) {
			unresolvedMethod(classContext, expression);
		}
		if (detect(options.non_static_method)) {
			nonStaticMethod(classContext, expression);
		}
		if (detect(options.deprecation)) {
			deprecated(classContext);
			deprecated(classContext, s);
		}
		if (detect(options.missing_arguments)) {
			missingArguments(classContext, s);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(StaticStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(SwitchCase s) throws Exception {
		if (detect(options.fall_through_case)) {
			fallThroughCase(s);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(SwitchStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(ThrowStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(TryStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(TypeReference s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(UnaryOperation s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(VariableReference s) throws Exception {
		if (detect(options.unused_local) && !unusedLocal) {
			unusedLocal();
			unusedLocal = true;
		}
		if (detect(options.uninitialized_variable)) {
			uninitializedVariable(s);
		}
		if (detect(options.misc_compile_error)) {
			checkVariable(s);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(WhileStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(UseStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(UsePart s) throws Exception {
		ClassContext classContext = new ClassContext(s.getNamespace());
		if (detect(options.unresolved_types)) {
			unresolvedClass(classContext);
		}
		if (detect(options.misc_error)) {
			errorneousClass(classContext);
		}
		if (detect(options.deprecation)) {
			deprecated(classContext);
		}
		return super.visit(s);
	}

	@Override
	public boolean visit(NamespaceDeclaration s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(NamespaceReference s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(FullyQualifiedReference s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(GotoLabel s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(GotoStatement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(LambdaFunctionDeclaration s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(Expression s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(MethodDeclaration s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(Statement s) throws Exception {
		return super.visit(s);
	}

	@Override
	public boolean visit(TypeDeclaration s) throws Exception {
		return super.visit(s);
	}

	/**
	 * Class Context Data
	 */
	private class ClassContext {
		protected Expression expression;
		protected IType sourceType;
		protected IField sourceField;
		protected IMethod sourceMethod;
		protected boolean isSelf;
		protected boolean isParent;
		protected boolean isStatic;
		protected String typeName;
		protected IType[] types;
		protected IField[] fields;
		protected IMethod[] methods;
		private TestPrivateVisitor privateVisitor;

		public ClassContext(Expression expression) {
			this.expression = expression;
		}

		public void resolveType() throws ModelException {
			if (expression == null || sourceType != null) {
				return;
			}
			typeName = null;
			types = new IType[0];
			if (expression instanceof FullyQualifiedReference) {
				typeName = ((FullyQualifiedReference) expression)
						.getFullyQualifiedName();
			} else if (expression instanceof VariableReference) {
				IEvaluatedType evaluatedType = bindingUtility
						.getType(expression);
				if (evaluatedType != null) {
					typeName = evaluatedType.getTypeName();
				}
			} else if (expression instanceof SimpleReference) {
				typeName = ((SimpleReference) expression).getName();
			} else if (expression instanceof FieldAccess) {
				IEvaluatedType evaluatedType = bindingUtility
						.getType(expression);
				if (evaluatedType != null) {
					typeName = evaluatedType.getTypeName();
				}
			} else if (expression instanceof StaticFieldAccess) {
				IEvaluatedType evaluatedType = bindingUtility
						.getType(expression);
				if (evaluatedType != null) {
					typeName = evaluatedType.getTypeName();
				}
			} else if (expression instanceof PHPCallExpression) {
				// XXX
				// inspect return type of expression ?
				String name = ((PHPCallExpression) expression).getCallName()
						.getName();
				IMethod[] methods = PHPModelUtils.getFunctions(name,
						sourceModule, 0, null);
				if (methods != null && methods.length > 0) {
					IEvaluatedType[] types = bindingUtility
							.getFunctionReturnType(methods[0]);
					if (types != null && types.length > 0) {
						typeName = types[0].getTypeName();
					}
				}
			} else {
				System.out.println(expression.getClass().getName());
				System.out.println(getSourceString(expression));
			}
			if (typeName != null) {
				types = PHPModelUtils.getTypes(typeName, sourceModule, 0, null);
				if (types != null && types.length > 0) {
					sourceType = types[0]; // XXX
				}
				if (sourceType == null) {
					if (typeName.equalsIgnoreCase("this")) {
						sourceType = PHPModelUtils.getCurrentType(sourceModule,
								expression.sourceStart());
					} else if (typeName.equalsIgnoreCase("self")) {
						isSelf = true;
						sourceType = PHPModelUtils.getCurrentType(sourceModule,
								expression.sourceStart());
					} else if (typeName.equalsIgnoreCase("static")) {
						isStatic = true;
						sourceType = PHPModelUtils.getCurrentType(sourceModule,
								expression.sourceStart());
					} else if (typeName.equalsIgnoreCase("parent")) {
						isParent = true;
						IType type = PHPModelUtils.getCurrentType(sourceModule,
								expression.sourceStart());
						if (type instanceof SourceType) {
							if (type.getSuperClasses().length > 0) {
								sourceType = (SourceType) type;
							}
						}
					}
				}
			}
		}

		public void resolveField(Expression expression) throws CoreException {
			if (sourceField != null) {
				return;
			}
			String name = null;
			Expression expr = expression;
			if (expr instanceof ReflectionArrayVariableReference) {
				expr = ((ReflectionArrayVariableReference) expr)
						.getExpression();
			}
			if (expr instanceof ReflectionVariableReference) {
				expr = ((ReflectionVariableReference) expr).getExpression();
			}
			if (expr instanceof ConstantReference) {
				name = ((ConstantReference) expr).getName();
			} else if (expr instanceof SimpleReference) {
				name = ((SimpleReference) expr).getName();
				if (!name.startsWith("$")) {
					name = "$" + name;
				}
			}
			if (name == null) {
				return; // XXX
			}
			if (sourceType == null) {
				resolveType();
				if (sourceType == null) {
					return;
				}
			}
			fields = sourceType.getFields();
			for (IField field : fields) {
				if (field.getElementName().equalsIgnoreCase(name)) {
					sourceField = field;
					break;
				}
			}
			if (sourceField == null) {
				String[] superClasses = sourceType.getSuperClasses();
				if (superClasses != null && superClasses.length > 0) {
					fields = PHPModelUtils.getTypeHierarchyField(sourceType,
							name, true, null);
					if (fields != null && fields.length > 0) {
						sourceField = fields[0]; // XXX
					}
				}
			}
		}

		public void resolveMethod(Expression expression) throws CoreException {
			if (sourceMethod != null) {
				return;
			}
			String name = null;
			if (expression instanceof SimpleReference) {
				name = ((SimpleReference) expression).getName();
			}
			if (name == null) {
				return;
			}
			if (sourceType == null) {
				resolveType();
				if (sourceType == null) {
					return;
				}
			}
			methods = sourceType.getMethods();
			for (IMethod method : methods) {
				if (method.getElementName().equalsIgnoreCase(name)) {
					sourceMethod = method;
					break;
				}
			}
			if (sourceMethod == null && sourceType.getSuperClasses() != null) {
				methods = PHPModelUtils.getTypeHierarchyMethod(sourceType,
						name, true, null);
				if (methods != null && methods.length > 0) {
					sourceMethod = methods[0]; // XXX
				}
			}
		}

		public void scanPrivate(ClassDeclaration classDeclaration)
				throws Exception {
			if (privateVisitor == null) {
				privateVisitor = new TestPrivateVisitor(sourceModule);
				classDeclaration.traverse(privateVisitor);
			}
		}

		public boolean isReadOnly(String name) {
			return privateVisitor.isReadOnly(name);
		}

		public boolean isWriteOnly(String name) {
			return privateVisitor.isWriteOnly(name);
		}

		public boolean isUsed(String name) {
			return privateVisitor.isUsed(name);
		}

		public List<Expression> getOccurrence(String field_name) {
			return privateVisitor.getOccurrence(field_name);
		}
	}

	protected void unresolvedClass(ClassContext context) throws ModelException {
		context.resolveType();
		if (context.sourceType == null) {
			String name = getSourceString(context.expression);
			if (context.expression instanceof VariableReference) {
				// XXX
				if (detect(options.misc_strict)) {
					createMarker(options.unresolved_types,
							Messages.Problem_unresolved_type,
							new String[] { name }, context.expression);
				}
			} else if (context.expression instanceof FieldAccess
					|| context.expression instanceof StaticFieldAccess) {
				// XXX
				if (detect(options.misc_strict)) {
					createMarker(options.unresolved_types,
							Messages.Problem_unresolved_type,
							new String[] { name }, context.expression);
				}
			} else if (context.expression instanceof PHPCallExpression) {
				ASTNode node = ((PHPCallExpression) context.expression)
						.getReceiver();
				if (node == null) {
					createMarker(options.unresolved_types,
							Messages.Problem_unresolved_type,
							new String[] { name }, context.expression);
				}
			} else {
				createMarker(options.unresolved_types,
						Messages.Class_0_not_found, new String[] { name },
						context.expression);
			}
			return;
		}
	}

	protected void errorneousClass(ClassContext context) throws ModelException {
		context.resolveType();
		if (context.isSelf || context.isStatic) {
			if (context.sourceType == null) {
				// Cannot access self:: when no class scope is active
				createMarker(options.misc_error,
						Messages.Cannot_access_0_when_no_class_scope_is_active,
						new String[] { context.isSelf ? "self" : "static" },
						context.expression);
			}
		} else if (context.isParent) {
			if (context.sourceType == null) {
				// Cannot access parent:: when current class
				// scope has no parent
				createMarker(
						options.misc_error,
						Messages.Cannot_access_parent_when_current_class_scope_has_no_parent,
						null, context.expression);
			}
		}
	}

	/**
	 * 未解決のインターフェースの検査
	 */
	protected void unresolvedInterface(ClassContext context)
			throws ModelException {
		context.resolveType();
		if (context.sourceType == null) {
			String name = getSourceString(context.expression);
			createMarker(options.unresolved_types,
					Messages.Interface_0_not_found, new String[] { name },
					context.expression);
		}
	}

	protected void unresolvedConstant(ClassContext context,
			Expression expression) throws CoreException {
		context.resolveField(expression);
		if (context.sourceType == null) {
			// cannot validate
			return;
		}
		if (context.sourceField == null) {
			String name = getSourceString(expression);
			createMarker(options.unresolved_fields,
					Messages.Problem_unresolved_constant, new String[] {
							context.sourceType.getElementName(), name },
					expression);
			return;
		}
	}

	/**
	 * 未解決のプロパティを検出する
	 */
	protected void unresolvedField(ClassContext context, Expression expression)
			throws CoreException {
		context.resolveField(expression);
		if (context.sourceType == null) {
			// cannot validate
			return;
		}
		if (context.sourceField != null
				&& !PHPFlags.isStatic(context.sourceField.getFlags())) {
			// normal
			return;
		}
		// read access -> notice: undefined property
		// write access -> no problem ...(new property)
		if (mode.peek() == WRITE) {
			return;
		}
		String name = null;
		int delta = 0;
		Expression expr = expression;
		if (expr instanceof ReflectionArrayVariableReference) {
			expr = ((ReflectionArrayVariableReference) expr).getExpression();
		}
		if (expr instanceof ReflectionVariableReference) {
			expr = ((ReflectionVariableReference) expr).getExpression();
		}
		if (expr instanceof SimpleReference) {
			name = ((SimpleReference) expr).getName();
			if (!name.startsWith("$")) {
				name = "$" + name;
				delta--;
			}
		} else {
			name = getSourceString(expr);
		}
		String message;
		if (expr instanceof VariableReference) {
			message = Messages.Problem_unresolved_property;
		} else {
			message = Messages.Problem_unresolved_field;
		}
		createMarker(options.unresolved_fields, message, new String[] {
				context.sourceType.getElementName(), name },
				expr.sourceStart(), expr.sourceStart() + name.length() + delta);
	}

	protected void unresolvedStaticField(ClassContext context,
			Expression expression) throws CoreException {
		context.resolveField(expression);
		if (context.sourceType == null) {
			// cannot validate
			return;
		}
		if (context.sourceField != null
				&& PHPFlags.isStatic(context.sourceField.getFlags())) {
			// normal
			return;
		}
		String name = getSourceString(expression);
		createMarker(options.unresolved_fields,
				Messages.Access_to_undeclared_static_property_0_1,
				new String[] { context.sourceType.getElementName(), name },
				expression);
	}

	protected void disallowedFieldAccessType(ClassContext context,
			Expression expression) throws CoreException {
		context.resolveField(expression);
		if (context.sourceField == null) {
			// cannot validate
			return;
		}
		boolean inhouse = false;
		IType declType = context.sourceField.getDeclaringType();
		IType type = PHPModelUtils.getCurrentType(sourceModule,
				expression.sourceStart());
		if (declType.equals(type)) {
			inhouse = true;
		} else if (type != null) {
			IType[] types = PHPModelUtils.getSuperClasses(type, null);
			for (IType superType : types) {
				if (declType.equals(superType)) {
					inhouse = true;
					break;
				}
			}
		}
		if (inhouse) {
			return;
		}
		String fieldType = context.sourceField.getDeclaringType()
				.getElementName();
		String fieldName = context.sourceField.getElementName();
		int flags = context.sourceField.getFlags();
		if (PHPFlags.isPrivate(flags)) {
			createMarker(options.misc_error,
					Messages.Cannot_access_private_property_0_1, new String[] {
							fieldType, fieldName }, expression);
		} else if (PHPFlags.isProtected(flags)) {
			createMarker(options.misc_error,
					Messages.Cannot_access_protected_property_0_1,
					new String[] { fieldType, fieldName }, expression);
		}
	}

	protected void disallowedStaticFieldAccessType(ClassContext context,
			Expression expression) throws CoreException {
		context.resolveField(expression);
		if (context.sourceField == null) {
			// cannot validate
			return;
		}
		// check if private/protected
		int flags = context.sourceField.getFlags();
		if (PHPFlags.isPrivate(flags)) {
			IType type = PHPModelUtils.getCurrentType(sourceModule,
					expression.sourceStart());
			if (!context.sourceField.getDeclaringType().equals(type)) {
				// Fatal error: Cannot access private property
				String class_name = context.sourceField.getDeclaringType()
						.getElementName();
				String field_name = context.sourceField.getElementName();
				createMarker(options.misc_error,
						Messages.Cannot_access_private_property_0_1,
						new String[] { class_name, field_name }, expression);
				return;
			}
		}
		if (PHPFlags.isProtected(flags)) {
			boolean inhouse = false;
			IType declType = context.sourceField.getDeclaringType();
			IType type = PHPModelUtils.getCurrentType(sourceModule,
					expression.sourceStart());
			if (declType.equals(type)) {
				inhouse = true;
			} else if (type != null) {
				ITypeHierarchy typeHierarchy = type.newTypeHierarchy(null);
				IType[] types = typeHierarchy.getAllSupertypes(type);
				inhouse = Arrays.asList(types).contains(type);
			}
			if (!inhouse) {
				// Fatal error: Cannot access protected property
				String class_name = context.sourceField.getDeclaringType()
						.getElementName();
				String field_name = context.sourceField.getElementName();
				createMarker(options.misc_error,
						Messages.Cannot_access_protected_property_0_1,
						new String[] { class_name, field_name }, expression);
				return;
			}
		}
	}

	/**
	 * 未解決のメソッドを検出する
	 */
	protected void unresolvedMethod(ClassContext context, Expression expression)
			throws CoreException {
		context.resolveMethod(expression);
		if (context.sourceType == null) {
			// cannot validate
			return;
		}
		if (context.sourceMethod == null) {
			String name = getSourceString(expression);
			createMarker(options.unresolved_methods,
					Messages.Problem_unresolved_method, new String[] {
							context.sourceType.getElementName(), name },
					expression);
			return;
		}
	}

	protected void disallowedMethodAccessType(ClassContext context,
			Expression expression) throws CoreException {
		context.resolveMethod(expression);
		if (context.sourceMethod == null) {
			// cannot validate
			return;
		}
		String declType = context.sourceMethod.getDeclaringType()
				.getElementName();
		String methodName = context.sourceMethod.getElementName();
		int flags = context.sourceMethod.getFlags();
		if (PHPFlags.isPrivate(flags)) {
			// must be in same class
			IType currType = PHPModelUtils.getCurrentType(sourceModule,
					expression.sourceStart());
			if (currType != null) {
				if (currType.getElementName().equals(declType)) {
					return;
				}
			}
			createMarker(options.misc_error,
					Messages.Call_to_private_method_0_1, new String[] {
							declType, methodName }, expression);
			return;
		}
		if (PHPFlags.isProtected(flags)) {
			// must be in same class or inherited
			IType currType = PHPModelUtils.getCurrentType(sourceModule,
					expression.sourceStart());
			if (currType != null) {
				if (currType.getElementName().equals(declType)) {
					return;
				}
				IMethod[] methods = PHPModelUtils.getSuperTypeHierarchyMethod(
						currType, methodName, true, null);
				if (methods != null && methods.length > 0) {
					return;
				}
			}
			createMarker(options.misc_error,
					Messages.Call_to_protected_method_0_1, new String[] {
							declType, methodName }, expression);
			return;
		}
		if (PHPFlags.isAbstract(flags)) {
			// Fatal error: Cannot call abstract method
			createMarker(options.misc_error,
					Messages.Cannot_call_abstract_method_0_1, new String[] {
							declType, methodName }, expression);
			return;
		}
	}

	/**
	 * プロジェクト内のクラス・インターフェース名の重複を検査する<br>
	 * 重複している場合、アクセス可能かどうかチェックする
	 */
	protected void redeclaredTypes(ClassContext context) throws ModelException {
		String declName = null;
		context.resolveType();
		if (context.sourceType != null) {
			declName = context.sourceType.getElementName();
		} else {
			if (context.expression instanceof SimpleReference) {
				declName = ((SimpleReference) context.expression).getName();
			}
		}
		if (declName == null) {
			return;
		}
		int trueFlag = 0;
		int falseFlag = Modifiers.AccNameSpace;
		// search declared name in the project
		IType[] types = findTypes(declName, trueFlag, falseFlag);
		if (types.length > 1) {
			// found more than 1 type
			boolean redeclared = false;
			boolean checkSystem = false;
			for (IType type : types) {
				if (checkSystem) {
					if (LanguageModelInitializer.isLanguageModelElement(type)) {
						redeclared = true;
						break;
					} else {
						continue;
					}
				}
				if (type.getSourceModule().equals(sourceModule)) {
					if (type.getNameRange().getOffset() == context.expression
							.sourceStart()) {
						// first occurrence
						checkSystem = true;
						if (context.sourceType == null) {
							context.sourceType = (SourceType) type;
						}
						continue;
					}
					// redeclared in the sourcemodule
					redeclared = true;
					break;
				} else {
					// external module, 'include' required
				}
			}
			if (redeclared) {
				createMarker(options.redeclared_types,
						Messages.Cannot_redeclare_class_0,
						new String[] { declName }, context.expression);
			}
		}
	}

	/**
	 * クラス内のメソッドの重複を検査する
	 */
	protected void redeclaredMethods(ClassContext context)
			throws ModelException {
		context.resolveType();
		if (context.sourceType == null) {
			// cannot validate
			return;
		}
		SourceType sourceType = (SourceType) context.sourceType;
		IMethod[] methods = sourceType.getMethods();
		for (int i = 0; i < methods.length; i++) {
			String methodName = methods[i].getElementName();
			for (int j = i + 1; j < methods.length; j++) {
				if (methods[j].getElementName().equals(methodName)) {
					if (methods[j] instanceof SourceMethod) {
						ISourceRange range = ((SourceMethod) methods[j])
								.getNameRange();
						String className = sourceType.getElementName();
						createMarker(options.redeclared_methods,
								Messages.Problem_redeclared_method,
								new String[] { className, methodName }, range);
						break;
					}
				}
			}
		}
	}

	/**
	 * クラス内のプロパティ・定数の重複を検査する
	 */
	protected void redeclaredFields(ClassContext context) throws ModelException {
		context.resolveType();
		if (context.sourceType == null) {
			// cannot validate
			return;
		}
		SourceType sourceType = (SourceType) context.sourceType;
		IField[] fields = sourceType.getFields();
		for (int i = 0; i < fields.length; i++) {
			String fieldName = fields[i].getElementName();
			boolean isConstant = PHPFlags.isConstant(fields[i].getFlags());
			for (int j = i + 1; j < fields.length; j++) {
				if (fields[j].getElementName().equals(fieldName)
						&& PHPFlags.isConstant(fields[j].getFlags()) == isConstant) {
					if (fields[j] instanceof SourceField) {
						SourceField sourceField = (SourceField) fields[j];
						String msg;
						if (isConstant) {
							msg = Messages.Problem_redeclared_constant;
						} else {
							msg = Messages.Problem_redeclared_field;
						}
						String className = sourceType.getElementName();
						ISourceRange range = sourceField.getNameRange();
						createMarker(options.redeclared_fields, msg,
								new String[] { className, fieldName }, range);
						break;
					}
				}
			}
		}
	}

	/**
	 * プロジェクト内の関数定義の重複を検査する<br>
	 * 重複している場合、アクセス可能かどうかチェックする
	 */
	protected void redeclaredFunctions(PHPMethodDeclaration methodDeclaration)
			throws ModelException {
		String declName = methodDeclaration.getName();
		int trueFlag = Modifiers.AccGlobal;
		int falseFlag = 0;
		// search declared name in the project
		IMethod[] methods = findMethods(declName, trueFlag, falseFlag);
		if (methods.length > 1) {
			// found more than 1 function
			boolean redeclared = false;
			for (IMethod method : methods) {
				if (method.getSourceModule().equals(sourceModule)) {
					ISourceRange range = method.getSourceRange();
					if (range.getOffset() == methodDeclaration.sourceStart()) {
						// first occurrence
						break;
					}
					// redeclared in the sourcemodule
					redeclared = true;
					break;
				} else {
					// external module, 'include' required
				}
			}
			if (redeclared) {
				createMarker(options.redeclared_functions,
						Messages.Problem_redeclared_function,
						new String[] { declName }, methodDeclaration.getRef());
			}
		}
	}

	/**
	 * ClassInstanceCreationの検査
	 */
	protected void instantiateAbstractClass(ClassContext context)
			throws ModelException {
		context.resolveType();
		if (context.sourceType == null) {
			// cannot validate
			return;
		}
		if (PHPFlags.isAbstract(context.sourceType.getFlags())) {
			createMarker(options.misc_error,
					Messages.Cannot_instantiate_abstract_class_0,
					new String[] { context.sourceType.getElementName() },
					context.expression);
			return;
		}
	}

	/**
	 * staticでないメソッドのstaticな呼び出しを検出する
	 */
	protected void nonStaticMethod(ClassContext context, Expression expression)
			throws CoreException {
		context.resolveMethod(expression);
		if (context.sourceMethod == null) {
			// cannot validate
			return;
		}
		String className = context.sourceMethod.getDeclaringType()
				.getElementName();
		String methodName = context.sourceMethod.getElementName();
		if (context.sourceMethod.isConstructor()) {
			createMarker(options.non_static_method,
					Messages.Non_static_method_0_1_cannot_be_called_statically,
					new String[] { className, methodName }, expression);
			return;
		}
		int flags = context.sourceMethod.getFlags();
		if (PHPFlags.isAbstract(flags)) {
			createMarker(options.non_static_method,
					Messages.Cannot_call_abstract_method_0_1, new String[] {
							className, methodName }, expression);
			return;
		}
		if (!PHPFlags.isStatic(flags)) {
			createMarker(
					options.non_static_method,
					Messages.Non_static_method_0_1_should_not_be_called_statically,
					new String[] { className, methodName }, expression);
			return;
		}
	}

	/**
	 * staticなプロパティに対するstaticでない参照を検出する
	 */
	protected void accessingAsNonStatic(ClassContext context,
			Expression expression) throws CoreException {
		context.resolveField(expression);
		if (context.sourceField == null) {
			// cannot validate
			return;
		}
		int flags = context.sourceField.getFlags();
		if (PHPFlags.isStatic(flags)) {
			String className = context.sourceField.getDeclaringType()
					.getElementName();
			String fieldName = context.sourceField.getElementName();
			createMarker(options.static_access_receiver,
					Messages.Accessing_static_property_0_1_as_non_static,
					new String[] { className, fieldName }, expression);
		}
	}

	protected void validateInterface(ClassContext context)
			throws ModelException {
		context.resolveType();
		if (context.sourceType == null) {
			return;
		}
		for (IMethod method : context.sourceType.getMethods()) {
			if (method instanceof SourceMethod) {
				context.sourceMethod = (SourceMethod) method;
				if (detect(options.misc_compile_error)) {
					inspectAbstractMethod(context.sourceMethod);
				}
			}
		}
	}

	protected void validateMethods(ClassContext context) throws CoreException {
		context.resolveType();
		if (context.sourceType == null) {
			return;
		}
		for (IMethod method : context.sourceType.getMethods()) {
			if (method instanceof SourceMethod) {
				context.sourceMethod = (SourceMethod) method;
				if (detect(options.override_final_method)) {
					overrideFinalMethod(context.sourceMethod);
				}
				if (detect(options.incompatible_method)) {
					incompatibleMethod(context.sourceMethod);
				}
				if (detect(options.misc_compile_error)) {
					inspectMagicMethodError(context.sourceMethod);
					inspectMethodAccessType(context.sourceMethod);
					inspectInheritedMethod(context.sourceMethod);
					inspectAbstractMethod(context.sourceMethod);
				}
				if (detect(options.misc_error)) {
					inspectAutoloadMethod(context.sourceMethod);
				}
				if (detect(options.misc_warning)) {
					inspectMagicMethodWarning(context.sourceMethod);
				}
				if (detect(options.misc_strict)) {
					inspectConstructor(context.sourceMethod);
					inspectStaticAbstractMethod(context.sourceMethod);
				}
			}
		}
	}

	/**
	 * finalメソッドのオーバーライドを検査する
	 */
	protected void overrideFinalMethod(IMethod sourceMethod)
			throws CoreException {
		String methodName = sourceMethod.getElementName();
		IMethod[] methods = PHPModelUtils.getSuperTypeHierarchyMethod(
				sourceMethod.getDeclaringType(), methodName, true, null);
		for (IMethod method : methods) {
			int flags = method.getFlags();
			if (PHPFlags.isFinal(flags)) {
				String className = method.getDeclaringType().getElementName();
				String[] info = { className, methodName };
				ISourceRange range = sourceMethod.getNameRange();
				createMarker(options.override_final_method,
						Messages.Cannot_override_final_method_0_1, info, range);
			}
		}
	}

	/**
	 * 引数に互換性の無いオーバーライドメソッドを検出する
	 */
	protected void incompatibleMethod(IMethod sourceMethod)
			throws CoreException {
		String methodName = sourceMethod.getElementName();
		String className = sourceMethod.getDeclaringType().getElementName();
		ISourceRange range = sourceMethod.getNameRange();
		IParameter[] parameters = sourceMethod.getParameters();

		IMethod[] methods = PHPModelUtils.getSuperTypeHierarchyMethod(
				sourceMethod.getDeclaringType(), methodName, true, null);
		for (IMethod method : methods) {
			String declName = method.getDeclaringType().getElementName();
			IParameter[] params = method.getParameters();
			if (parameters.length < params.length) {
				// E_COMPILE_ERROR when method is abstract
				// E_STRICT when method is not abstract
				createMarker(
						options.incompatible_method,
						Messages.Declaration_of_0_1_should_be_compatible_with_that_of_2_1,
						new String[] { className, methodName, declName }, range);
				break;
			}
			for (int i = 0; i < params.length; i++) {
				boolean detected = false;
				if (params[i].getDefaultValue() != null) {
					detected = parameters[i].getDefaultValue() == null;
				} else {
					// nothing to do
				}
				if (!detected) {
					if (params[i].getType() != null) {
						detected = !params[i].getType().equalsIgnoreCase(
								parameters[i].getType());
					} else {
						detected = parameters[i].getType() != null;
					}
				}
				if (detected) {
					createMarker(
							options.incompatible_method,
							Messages.Declaration_of_0_1_should_be_compatible_with_that_of_2_1,
							new String[] { className, methodName, declName },
							range);
					break;
				}
			}
		}
	}

	/**
	 * deprecatedクラスの検査
	 */
	protected void deprecated(ClassContext context) {
		if (ModelUtils.isDeprecated(context.sourceType)) {
			createMarker(options.deprecation, Messages.Problem_deprecation,
					new String[] { context.sourceType.getElementName() },
					context.expression);
		}
	}

	/**
	 * deprecatedメソッドの検査
	 */
	protected void deprecated(ClassContext context, PHPCallExpression expression)
			throws CoreException {
		if (context.sourceMethod == null) {
			context.resolveMethod(expression.getCallName());
			if (context.sourceMethod == null) {
				return;
			}
		}
		if (ModelUtils.isDeprecated(context.sourceMethod)) {
			StringBuffer name = new StringBuffer();
			name.append(
					context.sourceMethod.getDeclaringType().getElementName())
					.append("::").append(context.sourceMethod.getElementName())
					.append("()");
			createMarker(options.deprecation, Messages.Problem_deprecation,
					new String[] { name.toString() }, expression.getCallName());
		}
	}

	/**
	 * deprecatedメンバの検査
	 */
	protected void deprecated(ClassContext context, Expression expression)
			throws CoreException {
		if (context.sourceField == null) {
			context.resolveField(expression);
			if (context.sourceField == null) {
				return;
			}
		}
		if (ModelUtils.isDeprecated(context.sourceField)) {
			StringBuffer name = new StringBuffer();
			name.append(context.sourceType.getElementName()).append("::")
					.append(context.sourceField.getElementName());
			createMarker(options.deprecation, Messages.Problem_deprecation,
					new String[] { name.toString() }, expression);
		}
	}

	/**
	 * Includeファイルの存在チェック
	 */
	protected void includeNotFound(Include include) {
		String name = null;
		Expression expression = include.getExpr();
		if (expression instanceof Scalar) {
			name = ((Scalar) expression).getValue();
		}
		if (name != null) {
			if ((name.startsWith("\"") && name.endsWith("\""))
					|| (name.startsWith("'") && name.endsWith("'"))) {
				name = name.substring(1, name.length() - 1);
			}
			if (name.trim().equals("")) {
				createMarker(options.include_not_found,
						Messages.Filename_cannot_be_empty, null, include);
			} else {
				IPath path = getIncludeFilePath(name);
				if (path == null) {
					createMarker(options.include_not_found,
							Messages.Problem_include_not_found,
							new String[] { name }, include);
				}
			}
		}
	}

	/**
	 * 次のcase文になだれこむcase文を検出する
	 */
	protected void fallThroughCase(SwitchCase switchCase) {
		boolean fallThrough = true;
		for (Statement statement : switchCase.getActions()) {
			if (!fallThrough) {
				// XXX dead code
				if (detect(options.misc_strict)) {
					createMarker(options.fall_through_case, "Dead code", null,
							statement.sourceStart(), switchCase.sourceEnd());
				}
				break;
			}
			if (statement instanceof BreakStatement
					|| statement instanceof ReturnStatement
					|| statement instanceof ContinueStatement
					|| statement instanceof GotoStatement
					|| statement instanceof ThrowStatement) {
				fallThrough = false;
				// break;
			} else if (statement instanceof ExpressionStatement) {
				Expression expression = ((ExpressionStatement) statement)
						.getExpr();
				if (expression instanceof PHPCallExpression) {
					SimpleReference callName = ((PHPCallExpression) expression)
							.getCallName();
					String name = callName.getName();
					if (name.equalsIgnoreCase("exit")
							|| name.equalsIgnoreCase("die")) {
						fallThrough = false;
						// break;
					}
				}
			}
		}
		if (fallThrough) {
			Expression val = switchCase.getValue();
			int end = (val != null) ? val.sourceEnd() : switchCase.sourceEnd();
			createMarker(options.fall_through_case,
					Messages.Problem_fall_through_case, null,
					switchCase.sourceStart(), end);
		}
	}

	/**
	 * メソッドの引数の不足を検査する
	 */
	@SuppressWarnings("unchecked")
	protected void missingArguments(ClassContext context,
			PHPCallExpression expression) throws CoreException {
		if (context.sourceMethod == null) {
			context.resolveMethod(expression.getCallName());
			if (context.sourceMethod == null) {
				// cannot validate
				return;
			}
		}
		IParameter[] parameters = context.sourceMethod.getParameters();
		List<Expression> arguments = expression.getArgs().getChilds();
		if (parameters.length > arguments.size()) {
			for (int i = arguments.size(); i < parameters.length; i++) {
				if (parameters[i].getDefaultValue() == null) {
					String param = Integer.toString(arguments.size() + 1);
					StringBuffer name = new StringBuffer();
					name.append(
							context.sourceMethod.getDeclaringType()
									.getElementName()).append("::")
							.append(context.sourceMethod.getElementName());
					createMarker(options.missing_arguments,
							Messages.Missing_argument_0_for_1, new String[] {
									param, name.toString() }, expression);
					break;
				}
			}
		}
	}

	/**
	 * 未初期化変数を検出する
	 */
	protected void uninitializedVariable(VariableReference variable) {
		String name = variable.getName();
		if (name.equalsIgnoreCase("$this")
				&& !(variable instanceof ArrayVariableReference)) {
			return;
		}
		for (String var : PHPVariables.getVariables(phpVersion)) {
			if (name.equalsIgnoreCase(var)) {
				return;
			}
		}
		int idx = -1;
		ASTNode node = getNode(idx);
		if (node instanceof ReflectionVariableReference) {
			node = getNode(--idx);
		}
		if (node == null) {
			CodeCheckerPlugin.log(IStatus.ERROR, "invalid stack");
			return;
		}
		if (node instanceof FieldAccess) {
			return;
		} else if (node instanceof StaticFieldAccess) {
			return;
		} else if (node instanceof ForEachStatement) {
			if (!variable.equals(((ForEachStatement) node).getExpression())) {
				if (localMap != null) {
					localMap.put(name, true);
				} else {
					globalMap.put(name, true);
				}
				return;
			}
		} else if (node instanceof GlobalStatement) {
			Boolean initialized = globalMap.get(name);
			if (initialized == null) {
				globalMap.put(name, false);
				initialized = false;
			}
			if (localMap != null) {
				localMap.put(name, initialized);
			}
			return;
		} else if (node instanceof FormalParameter
				|| node instanceof CatchClause) {
			if (localMap != null) {
				localMap.put(name, true);
			} else {
				globalMap.put(name, true);
			}
			return;
		}
		Boolean initialized;
		if (localMap != null) {
			initialized = localMap.get(name);
		} else {
			initialized = globalMap.get(name);
		}
		if (mode.peek().equals(WRITE)) {
			initialized = true;
		} else if (initialized == null || !initialized) {
			createMarker(options.uninitialized_variable,
					Messages.Problem_uninitialized_variable,
					new String[] { name }, variable.sourceStart(),
					variable.sourceStart() + name.length());
			initialized = false;
		}
		if (localMap != null) {
			localMap.put(name, initialized);
		} else {
			globalMap.put(name, initialized);
		}
	}

	/**
	 * 参照されていないローカル変数を検出する
	 */
	protected void unusedLocal() throws Exception {
		TestGlobalVisitor visitor = new TestGlobalVisitor();
		ModuleDeclaration moduleDeclaration = SourceParserUtil
				.getModuleDeclaration(sourceModule);
		moduleDeclaration.traverse(visitor);
		for (VariableReference variable : visitor.getOccurrence()) {
			String name = variable.getName();
			if (visitor.isWriteOnly(name)) {
				createMarker(options.unused_local,
						Messages.Problem_unused_local, new String[] { name },
						variable.sourceStart(),
						variable.sourceStart() + name.length());
			}
		}
	}

	/**
	 * 関数内で参照されていないローカル変数を検出する
	 */
	protected void unusedLocal(PHPMethodDeclaration methodDeclaration)
			throws Exception {
		TestLocalVisitor visitor = new TestLocalVisitor();
		methodDeclaration.traverse(visitor);
		for (VariableReference variable : visitor.getOccurrence()) {
			String name = variable.getName();
			if (visitor.isWriteOnly(name)) {
				createMarker(options.unused_local,
						Messages.Problem_unused_local, new String[] { name },
						variable.sourceStart(),
						variable.sourceStart() + name.length());
			}
		}
	}

	/**
	 * 使用されていない引数を検出する
	 */
	protected void unusedParameter(PHPMethodDeclaration methodDeclaration)
			throws Exception {
		List<?> arguments = methodDeclaration.getArguments();
		if (arguments != null && !arguments.isEmpty()) {
			TestLocalVisitor visitor = new TestLocalVisitor();
			methodDeclaration.traverse(visitor);
			for (Object argument : arguments) {
				if (argument instanceof FormalParameter) {
					FormalParameter parameter = (FormalParameter) argument;
					String name = parameter.getName();
					if (visitor.isUnusedParam(name)) {
						createMarker(options.unused_parameter,
								Messages.Problem_unused_parameter,
								new String[] { name }, parameter);
					}
				}
			}
		}
	}

	/**
	 * 参照されていないprivateプロパティを検出する
	 */
	protected void unusedPrivate(ClassContext context) throws Exception {
		context.resolveType();
		if (context.sourceType == null) {
			return;
		}
		IField[] fields = context.sourceType.getFields();
		if (fields != null && fields.length > 0) {
			for (IField field : fields) {
				if (PHPFlags.isPrivate(field.getFlags())) {
					String field_name = field.getElementName();
					if (context.isWriteOnly(field_name)) {
						String class_name = field.getDeclaringType()
								.getElementName();
						createMarker(options.unused_private,
								Messages.Problem_unused_private, new String[] {
										class_name, field_name },
								field.getNameRange());
						for (Expression ex : context.getOccurrence(field_name)) {
							createMarker(options.unused_private,
									Messages.Problem_unused_private,
									new String[] { class_name, field_name }, ex);
						}
					} else if (!context.isUsed(field_name)) {
						String class_name = field.getDeclaringType()
								.getElementName();
						createMarker(options.unused_private,
								Messages.Problem_unused_private, new String[] {
										class_name, field_name },
								field.getNameRange());
					}
				}
			}
		}
	}

	/**
	 * 初期化されていないprivateプロパティを検出する
	 */
	protected void uninitializedPrivate(ClassContext context) throws Exception {
		context.resolveType();
		if (context.sourceType == null) {
			return;
		}
		IField[] fields = context.sourceType.getFields();
		if (fields != null && fields.length > 0) {
			for (IField field : fields) {
				if (PHPFlags.isPrivate(field.getFlags())) {
					String field_name = field.getElementName();
					if (context.isReadOnly(field_name)) {
						String class_name = field.getDeclaringType()
								.getElementName();
						String name = class_name + "::" + field_name;
						createMarker(options.uninitialized_variable,
								Messages.Problem_uninitialized_variable,
								new String[] { name }, field.getNameRange());
						for (Expression ex : context.getOccurrence(field_name)) {
							createMarker(options.unused_private,
									Messages.Problem_uninitialized_variable,
									new String[] { name }, ex);
						}
					}
				}
			}
		}
	}

	/**
	 * マジックメソッドの検査 (Compile Error)
	 */
	protected void inspectMagicMethodError(IMethod sourceMethod)
			throws ModelException {
		String methodName = sourceMethod.getElementName();
		int flags = sourceMethod.getFlags();
		if (sourceMethod.isConstructor()) {
			if (PHPFlags.isStatic(flags)) {
				// E_COMPILE_ERROR
				String[] info = {
						sourceMethod.getDeclaringType().getElementName(),
						methodName };
				createMarker(options.misc_compile_error,
						Messages.Constructor_0_1_cannot_be_static, info,
						sourceMethod.getNameRange());
			}
		} else if (methodName.equalsIgnoreCase("__desctruct")) {
			if (PHPFlags.isStatic(flags)) {
				// E_COMPILE_ERROR
				String[] info = {
						sourceMethod.getDeclaringType().getElementName(),
						methodName };
				createMarker(options.misc_compile_error,
						Messages.Destructor_0_1_cannot_be_static, info,
						sourceMethod.getNameRange());
			}
		} else if (methodName.equalsIgnoreCase("__clone")) {
			if (PHPFlags.isStatic(flags)) {
				// E_COMPILE_ERROR
				String[] info = {
						sourceMethod.getDeclaringType().getElementName(),
						methodName };
				createMarker(options.misc_compile_error,
						Messages.Clone_method_0_1_cannot_be_static, info,
						sourceMethod.getNameRange());
			}
		}
	}

	/**
	 * マジックメソッドの検査 (Warning)
	 */
	protected void inspectMagicMethodWarning(IMethod sourceMethod)
			throws ModelException {
		String methodName = sourceMethod.getElementName();
		ISourceRange range = sourceMethod.getNameRange();
		int flags = sourceMethod.getFlags();
		if (methodName.equalsIgnoreCase("__call")) {
			if ((flags & (Modifiers.AccPrivate | Modifiers.AccProtected | Modifiers.AccStatic)) != 0) {
				// E_WARNING
				createMarker(
						options.misc_warning,
						Messages.The_magic_method_call_must_have_public_visibility_and_cannot_be_static,
						null, range);
			}
		} else if (methodName.equalsIgnoreCase("__callstatic")) {
			if ((flags & (Modifiers.AccPrivate | Modifiers.AccProtected)) != 0
					|| (flags & Modifiers.AccStatic) == 0) {
				// E_WARNING
				createMarker(
						options.misc_warning,
						Messages.The_magic_method_callStatic_must_have_public_visibility_and_be_static,
						null, range);
			}
		} else if (methodName.equalsIgnoreCase("__get")) {
			if ((flags & (Modifiers.AccPrivate | Modifiers.AccProtected | Modifiers.AccStatic)) != 0) {
				// E_WARNING
				createMarker(
						options.misc_warning,
						Messages.The_magic_method_get_must_have_public_visibility_and_cannot_be_static,
						null, range);
			}
		} else if (methodName.equalsIgnoreCase("__set")) {
			if ((flags & (Modifiers.AccPrivate | Modifiers.AccProtected | Modifiers.AccStatic)) != 0) {
				// E_WARNING
				createMarker(
						options.misc_warning,
						Messages.The_magic_method_set_must_have_public_visibility_and_cannot_be_static,
						null, range);
			}
		} else if (methodName.equalsIgnoreCase("__unset")) {
			if ((flags & (Modifiers.AccPrivate | Modifiers.AccProtected | Modifiers.AccStatic)) != 0) {
				// E_WARNING
				createMarker(
						options.misc_warning,
						Messages.The_magic_method_unset_must_have_public_visibility_and_cannot_be_static,
						null, range);
			}
		} else if (methodName.equalsIgnoreCase("__isset")) {
			if ((flags & (Modifiers.AccPrivate | Modifiers.AccProtected | Modifiers.AccStatic)) != 0) {
				// E_WARNING
				createMarker(
						options.misc_warning,
						Messages.The_magic_method_isset_must_have_public_visibility_and_cannot_be_static,
						null, range);
			}
		} else if (methodName.equalsIgnoreCase("__tostring")) {
			if ((flags & (Modifiers.AccPrivate | Modifiers.AccProtected | Modifiers.AccStatic)) != 0) {
				// E_WARNING
				createMarker(
						options.misc_warning,
						Messages.The_magic_method_toString_must_have_public_visibility_and_cannot_be_static,
						null, range);
			}
		}
	}

	/**
	 * メソッドのアクセス修飾子の検査
	 */
	protected void inspectMethodAccessType(IMethod sourceMethod)
			throws ModelException {
		ISourceRange range = sourceMethod.getNameRange();
		int rangeStart = sourceMethod.getSourceRange().getOffset();
		int rangeEnd = range.getOffset() + range.getLength();
		int flags = sourceMethod.getFlags();
		int sum = (PHPFlags.isPrivate(flags) ? 1 : 0)
				+ (PHPFlags.isProtected(flags) ? 1 : 0)
				+ (PHPFlags.isPublic(flags) ? 1 : 0);
		if (sum > 1) {
			// E_COMPILE_ERROR
			createMarker(options.misc_compile_error,
					Messages.Multiple_access_type_modifiers_are_not_allowed,
					null, rangeStart, rangeEnd);
		}
		int len = sourceMethod.getNameRange().getOffset()
				- sourceMethod.getSourceRange().getOffset();
		String src = sourceMethod.getSource().substring(0, len);
		String[] modifiers = { "public", "protected", "private", "abstract",
				"static", "final" };
		for (String modifier : modifiers) {
			Pattern pattern = Pattern.compile(modifier,
					Pattern.CASE_INSENSITIVE);
			Matcher matcher = pattern.matcher(src);
			if (matcher.find()) {
				if (matcher.find(matcher.end() + 1)) {
					// E_COMPILE_ERROR
					String[] info = { modifier };
					createMarker(options.misc_compile_error,
							Messages.Multiple_0_modifiers_are_not_allowed,
							info, rangeStart, rangeEnd);
					break;
				}
			}
		}
		if (PHPFlags.isFinal(flags) && PHPFlags.isAbstract(flags)) {
			// E_COMPILE_ERROR
			createMarker(
					options.misc_compile_error,
					Messages.Cannot_use_the_final_modifier_on_an_abstract_class_member,
					null, rangeStart, rangeEnd);
		}
	}

	/**
	 * 継承されたメソッドの検査
	 */
	protected void inspectInheritedMethod(IMethod sourceMethod)
			throws CoreException {
		IType type = sourceMethod.getDeclaringType();
		String methodName = sourceMethod.getElementName();
		ISourceRange range = sourceMethod.getNameRange();
		IMethod[] methods = PHPModelUtils.getSuperTypeHierarchyMethod(type,
				methodName, true, null);
		if (methods == null || methods.length == 0) {
			// not inherited
			return;
		}
		int sflags = sourceMethod.getFlags();
		for (IMethod method : methods) {
			int flags = method.getFlags();
			if (PHPFlags.isAbstract(flags) && PHPFlags.isAbstract(sflags)) {
				// E_COMPILE_ERROR
				String[] info = { method.getDeclaringType().getElementName(),
						methodName, type.getElementName() };
				createMarker(
						options.misc_compile_error,
						Messages.Cannot_inherit_abstract_function_0_1_previously_declared_abstract_in_2,
						info, range);
			}
			if (PHPFlags.isFinal(flags)) {
				// E_COMPILE_ERROR
				String[] info = { method.getDeclaringType().getElementName(),
						methodName };
				createMarker(options.override_final_method,
						Messages.Cannot_override_final_method_0_1, info, range);
			}
			if (PHPFlags.isStatic(flags) != PHPFlags.isStatic(sflags)) {
				String[] info = { method.getDeclaringType().getElementName(),
						methodName, type.getElementName() };
				if (PHPFlags.isStatic(flags)) {
					// E_COMPILE_ERROR
					createMarker(
							options.misc_compile_error,
							Messages.Cannot_make_static_method_0_1_non_static_in_class_2,
							info, range);
				} else {
					// E_COMPILE_ERROR
					createMarker(
							options.misc_compile_error,
							Messages.Cannot_make_non_static_method_0_1_static_in_class_2,
							info, range);
				}
			}
			if (!PHPFlags.isAbstract(flags) && PHPFlags.isAbstract(sflags)) {
				// E_COMPILE_ERROR
				String[] info = { method.getDeclaringType().getElementName(),
						methodName, type.getElementName() };
				createMarker(
						options.misc_compile_error,
						Messages.Cannot_make_non_abstract_method_0_1_abstract_in_class_2,
						info, range);
			}
			int accPPP = Modifiers.AccPrivate | Modifiers.AccProtected
					| Modifiers.AccPublic;
			if ((sflags & accPPP) == 0) {
				sflags = Modifiers.AccPublic; // assumed public
			}
			if ((flags & accPPP) > (sflags & accPPP)) {
				// E_COMPILE_ERROR
				String accType = accessModifier(flags);
				String tail = PHPFlags.isPublic(flags) ? "" : " or weaker";
				String[] info = { type.getElementName(), methodName, accType,
						method.getDeclaringType().getElementName(), tail };
				createMarker(
						options.misc_compile_error,
						Messages.Access_level_to_0_1__must_be_2_as_in_class_3_4,
						info, range);
			}
		}
	}

	/**
	 * abstractメソッドの検査
	 */
	protected void inspectAbstractMethod(IMethod sourceMethod)
			throws ModelException {
		String methodType = "";
		IType type = sourceMethod.getDeclaringType();
		if (PHPFlags.isInterface(type.getFlags())) {
			methodType = "Interface";
		} else if (PHPFlags.isAbstract(sourceMethod.getFlags())) {
			methodType = "Abstract";
		}
		int flags = sourceMethod.getFlags();
		if (PHPFlags.isAbstract(flags)) {
			if (PHPFlags.isPrivate(flags)) {
				// E_COMPILE_ERROR
				String[] info = { methodType, type.getElementName(),
						sourceMethod.getElementName() };
				createMarker(options.misc_compile_error,
						Messages.Z0_function_1_2_cannot_be_declared_private,
						info, sourceMethod.getSourceRange());
			}
		}
	}

	protected void inspectAbstractMethodBody(PHPMethodDeclaration declaration)
			throws ModelException {
		Block block = declaration.getBody(); // always return "{...}"
		if (declaration.isAbstract() || currentScope == SCOPE_INTERFACE) {
			String source = getSourceString(block).trim();
			if (source.length() > 1) { // expected ";"
				// E_COMPILE_ERROR
				String[] info = {
						declaration.isAbstract() ? "Abstract" : "Interface",
						declaration.getDeclaringTypeName(),
						declaration.getName() };
				createMarker(options.misc_compile_error,
						Messages.Z0_function_1_2_cannot_contain_body, info,
						declaration);
			}
		} else {
			String source = getSourceString(block).trim();
			if (source.length() < 2) { // expected "{}"
				// E_COMPILE_ERROR
				String[] info = { declaration.getDeclaringTypeName(),
						declaration.getName() };
				createMarker(options.misc_compile_error,
						Messages.Non_abstract_method_0_1_must_contain_body,
						info, declaration);
			}
		}
	}

	/**
	 * __autoloadメソッドの検査
	 */
	protected void inspectAutoloadMethod(IMethod sourceMethod)
			throws ModelException {
		String methodName = sourceMethod.getElementName();
		int params = sourceMethod.getParameters().length;
		String[] info = { sourceMethod.getDeclaringType().getElementName(),
				methodName };
		ISourceRange range = sourceMethod.getNameRange();
		if (methodName.equalsIgnoreCase("__destruct")) {
			if (params != 0) {
				// zend API error: zend_error(error_type,
				createMarker(options.misc_error,
						Messages.Destructor_0_1_cannot_take_arguments, info,
						range);
				return;
			}
		} else if (methodName.equalsIgnoreCase("__clone")) {
			if (params != 0) {
				// zend_error(error_type,
				createMarker(options.misc_error,
						Messages.Method_0_1_cannot_accept_any_arguments, info,
						range);
			}
		} else if (methodName.equalsIgnoreCase("__get")) {
			if (params != 1) {
				// zend_error(error_type,
				createMarker(options.misc_error,
						Messages.Method_0_1_must_take_exactly_1_argument, info,
						range);
			}
		} else if (methodName.equalsIgnoreCase("__set")) {
			if (params != 2) {
				// zend_error(error_type,
				createMarker(options.misc_error,
						Messages.Method_0_1_must_take_exactly_2_arguments,
						info, range);
			}
		} else if (methodName.equalsIgnoreCase("__unset")
				|| methodName.equalsIgnoreCase("__isset")) {
			if (params != 1) {
				// zend_error(error_type,
				createMarker(options.misc_error,
						Messages.Method_0_1_must_take_exactly_1_argument, info,
						range);
			}
		} else if (methodName.equalsIgnoreCase("__call")
				|| methodName.equalsIgnoreCase("__callstatic")) {
			if (params != 2) {
				// zend_error(error_type,
				createMarker(options.misc_error,
						Messages.Method_0_1_must_take_exactly_2_arguments,
						info, range);
			}
		} else if (methodName.equalsIgnoreCase("__tostring")) {
			if (params != 0) {
				// zend_error(error_type,
				createMarker(options.misc_error,
						Messages.Method_0_1_cannot_take_arguments, info, range);
			}
		}
	}

	protected void inspectAutoloadMethod2(
			FormalParameterByReference formalParameterByReference) {
		IMethod method = PHPModelUtils.getCurrentMethod(sourceModule,
				formalParameterByReference.getNameStart());
		String name = method.getElementName();
		if (name.equalsIgnoreCase("__get") || name.equalsIgnoreCase("__set")
				|| name.equalsIgnoreCase("__unset")
				|| name.equalsIgnoreCase("__isset")
				|| name.equalsIgnoreCase("__call")
				|| name.equalsIgnoreCase("__callstatic")) {
			createMarker(options.misc_error,
					Messages.Method_0_1_cannot_take_arguments_by_reference,
					new String[] { method.getDeclaringType().getElementName(),
							name }, formalParameterByReference);
		}
	}

	/**
	 * constructorの検査
	 */
	protected void inspectConstructor(IMethod sourceMethod)
			throws ModelException {
		if (PHPFlags.isInterface(sourceMethod.getDeclaringType().getFlags())) {
			return;
		}
		if (sourceMethod.isConstructor()) {
			for (IMethod method : sourceMethod.getDeclaringType().getMethods()) {
				if (method.equals(sourceMethod)) {
					break;
				}
				if (method.isConstructor()) {
					// E_STRICT
					String className = sourceMethod.getDeclaringType()
							.getElementName();
					ISourceRange range = sourceMethod.getNameRange();
					createMarker(
							options.misc_strict,
							Messages.Redefining_already_defined_constructor_for_class_0,
							new String[] { className }, range);
					return;
				}
			}
		}
	}

	/**
	 * static+abstractの検査
	 */
	protected void inspectStaticAbstractMethod(IMethod sourceMethod)
			throws ModelException {
		if (PHPFlags.isInterface(sourceMethod.getDeclaringType().getFlags())) {
			return;
		}
		int flags = sourceMethod.getFlags();
		if (PHPFlags.isStatic(flags) && PHPFlags.isAbstract(flags)) {
			// E_STRICT
			ISourceRange range = sourceMethod.getNameRange();
			String[] info = { sourceMethod.getDeclaringType().getElementName(),
					sourceMethod.getElementName() };
			createMarker(options.misc_strict,
					Messages.Static_function_0_1_should_not_be_abstract, info,
					range);
		}
	}

	/**
	 * 継承されたクラスの検査
	 */
	protected void inspectInheritedClass(ClassContext context)
			throws ModelException {
		context.resolveType();
		if (context.sourceType == null) {
			return;
		}
		SourceType sourceType = (SourceType) context.sourceType;
		int flags = sourceType.getFlags();
		IType[] types = PHPModelUtils.getSuperClasses(sourceType, null);
		for (IType type : types) {
			int superflags = type.getFlags();
			if (PHPFlags.isFinal(superflags)) {
				// E_COMPILE_ERROR
				ISourceRange range = sourceType.getNameRange();
				String[] info = { sourceType.getElementName(),
						type.getElementName() };
				createMarker(options.misc_compile_error,
						Messages.Class_0_may_not_inherit_from_final_class_1,
						info, range);
			}
			if (PHPFlags.isInterface(flags)
					&& !PHPFlags.isInterface(superflags)) {
				// E_COMPILE_ERROR
				ISourceRange range = sourceType.getNameRange();
				String[] info = { sourceType.getElementName(),
						type.getElementName() };
				createMarker(options.misc_compile_error,
						// "Interface {0} may not inherit from class ({1})",
						Messages.Z0_cannot_implement_1_it_is_not_an_interface,
						info, range);
			}
		}
	}

	/**
	 * 継承されたConstructorの検査
	 */
	protected void inspectInheritedConstructor(ClassContext context)
			throws ModelException {
		context.resolveType();
		if (context.sourceType == null) {
			return;
		}
		SourceType sourceType = (SourceType) context.sourceType;
		IMethod constructor = null;
		for (IMethod method : sourceType.getMethods()) {
			if (method.isConstructor()) {
				constructor = method;
				break;
			}
		}
		if (constructor == null) {
			return;
		}
		IType[] types = PHPModelUtils.getSuperClasses(sourceType,
				sourceType.newSupertypeHierarchy(null));
		for (IType type : types) {
			for (IMethod method : type.getMethods()) {
				if (method.isConstructor()) {
					int flags = method.getFlags();
					if (PHPFlags.isFinal(flags)) {
						// E_ERROR
						String sClass = method.getDeclaringType()
								.getElementName();
						String sName = method.getElementName();
						String[] info = { sClass, sName,
								sourceType.getElementName(),
								constructor.getElementName() };
						ISourceRange range = constructor.getNameRange();
						createMarker(options.misc_compile_error,
								Messages.Cannot_override_final_0_1_with_2_3,
								info, range);
						return;
					}
					break;
				}
			}
		}
	}

	/**
	 * 継承されたプロパティを検査する
	 */
	protected void inspectInheritedFields(ClassContext context)
			throws CoreException {
		context.resolveType();
		if (context.sourceType == null) {
			return;
		}
		SourceType sourceType = (SourceType) context.sourceType;
		for (IField field : sourceType.getFields()) {
			int flags = field.getFlags();
			boolean fStatic = PHPFlags.isStatic(flags);
			IField[] sfields = PHPModelUtils.getSuperTypeHierarchyField(
					sourceType, field.getElementName(), true, null);
			for (IField sfield : sfields) {
				int superflags = sfield.getFlags();
				boolean sStatic = PHPFlags.isStatic(superflags);
				if (sStatic != fStatic) {
					// E_COMPILE_ERROR
					String sup = sStatic ? "static " : "non static ";
					String fld = fStatic ? "static " : "non static ";
					String name = field.getElementName();
					ISourceRange range = field.getSourceRange();
					String[] info = { sup,
							sfield.getDeclaringType().getElementName(), name,
							fld, sourceType.getElementName(), name };
					createMarker(options.misc_compile_error,
							Messages.Cannot_redeclare_0_1_2_as_3_4_5, info,
							range);
				}
				int accPPP = Modifiers.AccPrivate | Modifiers.AccProtected
						| Modifiers.AccPublic;
				if ((superflags & accPPP) < (flags & accPPP)) {
					// E_COMPILE_ERROR
					String accType = accessModifier(superflags);
					String tail = PHPFlags.isPublic(superflags) ? ""
							: " or weaker";
					ISourceRange range = field.getSourceRange();
					String[] info = {
							sfield.getDeclaringType().getElementName(),
							field.getElementName(), accType,
							field.getDeclaringType().getElementName(), tail };
					createMarker(
							options.misc_compile_error,
							Messages.Access_level_to_0_1_must_be_2_as_in_class_3_4,
							info, range);
				}
			}
		}
	}

	/**
	 * FormalParameterのdefaultValueの検査
	 */
	protected void inspectFormalParameter(FormalParameter formalParameter) {
		SimpleReference parameterType = formalParameter.getParameterType();
		ASTNode defaultValue = formalParameter.getInitialization();
		String typeName = parameterType.getName();
		if (typeName.equalsIgnoreCase("array")) {
			if (defaultValue instanceof ArrayCreation
					|| (defaultValue instanceof Scalar && ((Scalar) defaultValue)
							.getValue().equalsIgnoreCase("null"))) {
			} else {
				// E_COMPILE_ERROR
				createMarker(
						options.misc_compile_error,
						Messages.Default_value_for_parameters_with_array_type_hint_can_only_be_an_array_or_NULL,
						null, parameterType.sourceStart(),
						defaultValue.sourceEnd());
			}
		} else {
			if (defaultValue instanceof Scalar
					&& ((Scalar) defaultValue).getValue().equalsIgnoreCase(
							"null")) {
			} else {
				// E_COMPILE_ERROR
				createMarker(
						options.misc_compile_error,
						Messages.Default_value_for_parameters_with_a_class_type_hint_can_only_be_NULL,
						null, parameterType.sourceStart(),
						defaultValue.sourceEnd());
			}
		}
	}

	/**
	 * プロパティの検査
	 */
	protected void inspectClassFields(ClassContext context)
			throws ModelException {
		context.resolveType();
		if (context.sourceType == null) {
			return;
		}
		SourceType sourceType = (SourceType) context.sourceType;
		boolean isInterface = PHPFlags.isInterface(sourceType.getFlags());
		for (IField field : sourceType.getFields()) {
			ISourceRange range = field.getSourceRange();
			int flags = field.getFlags();
			if (PHPFlags.isConstant(flags)) {
				// array check => inspectClassConstant
				continue;
			}
			if (isInterface) {
				// E_COMPILE_ERROR
				createMarker(options.misc_compile_error,
						Messages.Interfaces_may_not_include_member_variables,
						null, range);
				continue;
			}
			if (PHPFlags.isAbstract(flags)) {
				// E_COMPILE_ERROR
				createMarker(options.misc_compile_error,
						Messages.Properties_cannot_be_declared_abstract, null,
						range);
			}
			if (PHPFlags.isFinal(flags)) {
				// E_COMPILE_ERROR
				createMarker(
						options.misc_compile_error,
						Messages.Cannot_declare_property_0_1_final,
						new String[] { sourceType.getElementName(),
								field.getElementName() }, range);
			}
			int sum = (PHPFlags.isPrivate(flags) ? 1 : 0)
					+ (PHPFlags.isProtected(flags) ? 1 : 0)
					+ (PHPFlags.isPublic(flags) ? 1 : 0);
			if (sum > 1) {
				// E_COMPILE_ERROR
				createMarker(
						options.misc_compile_error,
						Messages.Multiple_access_type_modifiers_are_not_allowed,
						null, range);
			}
			int start = field.getSourceRange().getOffset();
			int end = field.getNameRange().getOffset();
			String src = sourceModule.getSource().substring(start, end);
			String[] modifiers = { "public", "protected", "private",
					"abstract", "static", "final" };
			for (String modifier : modifiers) {
				Pattern pattern = Pattern.compile(modifier,
						Pattern.CASE_INSENSITIVE);
				Matcher matcher = pattern.matcher(src);
				if (matcher.find()) {
					if (matcher.find(matcher.end())) {
						// E_COMPILE_ERROR
						String[] info = { modifier };
						createMarker(options.misc_compile_error,
								Messages.Multiple_0_modifiers_are_not_allowed,
								info, range);
						break;
					}
				}
			}
		}
	}

	/**
	 * クラス定数の検査
	 */
	protected void inspectClassConstants(ConstantDeclaration constantDeclaration) {
		if (constantDeclaration.getConstantValue() instanceof ArrayCreation) {
			// E_COMPILE_ERROR
			createMarker(options.misc_compile_error,
					Messages.Arrays_are_not_allowed_in_class_constants, null,
					constantDeclaration);
		}
	}

	/**
	 * __clone()呼び出しの検査
	 */
	protected void inspectCloneMethod(ClassContext context,
			Expression expression) {
		if (expression instanceof VariableReference) {
			return;
		}
		String methodName = null;
		if (expression instanceof SimpleReference) {
			methodName = ((SimpleReference) expression).getName();
			if (methodName.equalsIgnoreCase("__clone")) {
				// E_COMPILE_ERROR
				createMarker(options.misc_compile_error,
						Messages.Cannot_call_clone_method_on_objects, null,
						expression);
			}
		}
	}

	/**
	 * foreach文のkeyがリファレンスかどうかの検査
	 */
	protected void inspectReference(Expression expression) {
		if (expression instanceof ReferenceExpression) {
			// E_COMPILE_ERROR
			createMarker(options.misc_compile_error,
					Messages.Key_element_cannot_be_a_reference, null,
					expression);
		}
	}

	/**
	 * Variableの検査
	 */
	protected void checkVariable(VariableReference variable) {
		String name = variable.getName();
		if (name.equalsIgnoreCase("$this")) {
			if (mode.peek().equals(WRITE)) {
				// E_COMPILE_ERROR
				createMarker(options.misc_compile_error,
						Messages.Cannot_re_assign_this, null, variable);
			}
		}
	}

	/**
	 * クラス名のチェック<br>
	 * "self", "parent"ならエラー
	 */
	protected boolean checkClassName(Expression expression) {
		return checkTypeName(expression, true);
	}

	/**
	 * インターフェース名のチェック<br>
	 * "self", "parent"ならエラー
	 */
	protected boolean checkInterfaceName(Expression expression) {
		return checkTypeName(expression, false);
	}

	/**
	 * クラス・インターフェース名の検査<br>
	 * "self", "parent"ならエラー
	 */
	private boolean checkTypeName(Expression expression, boolean isClass) {
		if (expression instanceof SimpleReference) {
			String name = ((SimpleReference) expression).getName();
			if (name.equalsIgnoreCase("self")
					|| name.equalsIgnoreCase("parent")) {
				// E_COMPILE_ERROR
				if (isClass) {
					createMarker(
							options.misc_compile_error,
							Messages.Cannot_use_0_as_class_name_as_it_is_reserved,
							new String[] { name }, expression);
				} else {
					createMarker(
							options.misc_compile_error,
							Messages.Cannot_use_0_as_interface_name_as_it_is_reserved,
							new String[] { name }, expression);
				}
				return false;
			}
		}
		return true;
	}

	private String accessModifier(int flags) {
		if (PHPFlags.isPrivate(flags)) {
			return "private";
		} else if (PHPFlags.isProtected(flags)) {
			return "protected";
		} else if (PHPFlags.isPublic(flags)) {
			return "public";
		}
		return "?";
	}

	private String getSourceString(Expression expression) throws ModelException {
		if (expression == null) {
			return "";
		}
		return sourceModule.getSource().substring(expression.sourceStart(),
				expression.sourceEnd());
	}

	/*
	 * @see org.eclipse.php.internal.core.codeassist.strategies.
	 * AbstractCompletionStrategy
	 */
	private IDLTKSearchScope createSearchScope() {
		IScriptProject scriptProject = sourceModule.getScriptProject();
		if (scriptProject != null) {
			return SearchEngine.createSearchScope(scriptProject);
		}
		IProjectFragment projectFragment = (IProjectFragment) sourceModule
				.getAncestor(IModelElement.PROJECT_FRAGMENT);
		if (projectFragment != null) {
			return SearchEngine.createSearchScope(projectFragment);
		}
		return SearchEngine.createSearchScope(sourceModule);
	}

	private IType[] findTypes(String name, int trueFlag, int falseFlag) {
		IDLTKSearchScope scope = createSearchScope();
		IType[] types = PhpModelAccess.getDefault().findTypes(name,
				MatchRule.EXACT, trueFlag, falseFlag, scope, null);
		return types;
	}

	private IMethod[] findMethods(String name, int trueFlag, int falseFlag) {
		IDLTKSearchScope scope = createSearchScope();
		IMethod[] methods = PhpModelAccess.getDefault().findMethods(name,
				MatchRule.EXACT, trueFlag, falseFlag, scope, null);
		return methods;
	}

	/**
	 * includeファイルの実pathを取得する
	 */
	private IPath getIncludeFilePath(String name) {
		IProject project = sourceModule.getScriptProject().getProject();
		IncludePath[] includePaths = IncludePathManager.getInstance()
				.getIncludePaths(project);
		for (IncludePath includePath : includePaths) {
			Object includePathEntry = includePath.getEntry();
			if (includePathEntry instanceof IResource) {
				IPath resourcePath = ((IResource) includePathEntry)
						.getFullPath().addTrailingSeparator().append(name);
				IResource resource = ResourcesPlugin.getWorkspace().getRoot()
						.findMember(resourcePath);
				if (resource != null) {
					return resourcePath;
				}
			} else if (includePathEntry instanceof IBuildpathEntry) {
				IBuildpathEntry buildpathEntry = (IBuildpathEntry) includePathEntry;
				switch (buildpathEntry.getEntryKind()) {
				case IBuildpathEntry.BPE_PROJECT:
					IPath resourcePath = buildpathEntry.getPath()
							.addTrailingSeparator().append(name);
					IResource resource = ResourcesPlugin.getWorkspace()
							.getRoot().findMember(resourcePath);
					if (resource != null) {
						return resourcePath;
					}
					break;
				case IBuildpathEntry.BPE_LIBRARY:
					IPath path = buildpathEntry.getPath()
							.addTrailingSeparator().append(name);
					resourcePath = EnvironmentPathUtils.getLocalPath(path);
					if (resourcePath.toFile().exists()) {
						return path;
					}
					break;
				}
			}
		}
		return null;
	}

	/**
	 * 
	 */
	private class FunctionContext {

		private PHPCallExpression expression;
		private String functionName;
		private IMethod sourceMethod;

		public FunctionContext(PHPCallExpression expression) {
			this.expression = expression;
			functionName = expression.getName();
		}

		public void resolve() throws ModelException {
			IMethod[] methods = PHPModelUtils.getFunctions(functionName,
					sourceModule, 0, null);
			if (methods.length == 1) {
				sourceMethod = methods[0];
			}
			if (sourceMethod != null) {
			} else {
				for (String languageConstruct : LanguageConstructs) {
					if (functionName.equalsIgnoreCase(languageConstruct)) {
						return;
					}
				}
				createMarker(options.unresolved_functions,
						Messages.Problem_unresolved_function,
						new String[] { functionName }, expression.getCallName());
			}
		}

		public void deprecation() throws ModelException {
			if (sourceMethod == null) {
				resolve();
				if (sourceMethod == null) {
					return;
				}
			}
			if (ModelUtils.isDeprecated(sourceMethod)) {
				createMarker(options.deprecation, Messages.Problem_deprecation,
						new String[] { sourceMethod.getElementName() + "()" },
						expression.getCallName());
			}
		}

		@SuppressWarnings("unchecked")
		public void arguments() throws ModelException {
			if (sourceMethod == null) {
				resolve();
				if (sourceMethod == null) {
					return;
				}
			}
			IParameter[] parameters = sourceMethod.getParameters();
			List<Expression> arguments = expression.getArgs().getChilds();
			if (parameters.length > arguments.size()) {
				for (int i = arguments.size(); i < parameters.length; i++) {
					if (parameters[i].getDefaultValue() == null) {
						String param = Integer.toString(arguments.size() + 1);
						createMarker(
								options.missing_arguments,
								Messages.Missing_argument_0_for_1,
								new String[] { param,
										sourceMethod.getElementName() },
								expression);
						break;
					}
				}
			}
		}
	}
}
