package jp.sourceforge.pdt_tools.codeChecker;

import java.util.List;

import jp.sourceforge.pdt_tools.codeChecker.visitor.CodeVisitor;

import org.eclipse.core.internal.content.ContentTypeManager;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.content.IContentType;
import org.eclipse.dltk.ast.declarations.ModuleDeclaration;
import org.eclipse.dltk.ast.parser.IModuleDeclaration;
import org.eclipse.dltk.compiler.env.IModuleSource;
import org.eclipse.dltk.core.DLTKCore;
import org.eclipse.dltk.core.IBuildpathEntry;
import org.eclipse.dltk.core.IModelElement;
import org.eclipse.dltk.core.IScriptProject;
import org.eclipse.dltk.core.ISourceModule;
import org.eclipse.dltk.core.ModelException;
import org.eclipse.dltk.core.ScriptModelUtil;
import org.eclipse.php.internal.core.PHPVersion;
import org.eclipse.php.internal.core.compiler.ast.parser.PHPSourceParserFactory;
import org.eclipse.php.internal.core.project.ProjectOptions;
import org.eclipse.ui.PlatformUI;
import org.eclipse.wst.validation.AbstractValidator;
import org.eclipse.wst.validation.ValidationResult;
import org.eclipse.wst.validation.ValidationState;

public class Validator extends AbstractValidator {

	public Validator() {
	}

	@Override
	public void clean(IProject project, ValidationState state,
			IProgressMonitor monitor) {
		try {
			IMarker[] markers = project.findMarkers(
					CodeCheckerPlugin.MARKER_ID, false,
					IResource.DEPTH_INFINITE);
			for (IMarker marker : markers) {
				marker.delete();
			}
		} catch (CoreException e) {
			CodeCheckerPlugin.log(e);
		}
	}

	@Override
	public ValidationResult validate(IResource resource, int kind,
			ValidationState state, final IProgressMonitor monitor) {
		PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {
			public void run() {
				if (PlatformUI.getWorkbench().getActiveWorkbenchWindow() == null) {
					monitor.setCanceled(true);
				}
			}
		});
		if (monitor.isCanceled()) {
			return null;
		}
		switch (kind) {
		case IResourceDelta.NO_CHANGE: // manual validation
		case IResourceDelta.ADDED:
		case IResourceDelta.CHANGED:
			if (shouldBeValidated(resource)) {
				return validate(resource, state, monitor);
			}
			break;
		}
		return super.validate(resource, kind, state, monitor);
	}

	private boolean shouldBeValidated(IResource resource) {
		if (resource instanceof IFile) {
			IContentType contentType = ContentTypeManager.getInstance()
					.findContentTypeFor(resource.getName());
			if (contentType != null
					&& contentType.getId().equals(
							"org.eclipse.php.core.phpsource")) {
				if (ProjectOptions.getPhpVersion((IFile) resource).equals(
						PHPVersion.PHP5_3)) {
					if (isInBuildPath(resource)) {
						return true;
					}
				}
			}
		}
		return false;
	}

	private boolean isInBuildPath(IResource resource) {
		IScriptProject project = DLTKCore.create(resource.getProject());
		try {
			IPath resourcePath = resource.getProjectRelativePath();
			IBuildpathEntry[] buildpath = project.getRawBuildpath();
			for (IBuildpathEntry buildpathEntry : buildpath) {
				if (buildpathEntry.getEntryKind() == IBuildpathEntry.BPE_SOURCE) {
					IPath[] exclusionPatterns = buildpathEntry
							.getExclusionPatterns();
					if (ScriptModelUtil.isExcludedPath(resourcePath,
							exclusionPatterns)) {
						return false;
					}
				}
			}
			return true;
		} catch (ModelException e) {
		}
		return false;
	}

	private ValidationResult validate(IResource resource,
			ValidationState state, IProgressMonitor monitor) {
		ValidationResult result = new ValidationResult();
		IModelElement modelElement = DLTKCore.create(resource);
		if (modelElement instanceof ISourceModule) {
			ISourceModule sourceModule = (ISourceModule) modelElement;
			IModuleDeclaration moduleDeclaration = new PHPSourceParserFactory()
					.parse((IModuleSource) sourceModule, null);
			CodeVisitor visitor = new CodeVisitor(sourceModule);
			try {
				((ModuleDeclaration) moduleDeclaration).traverse(visitor);
			} catch (ModelException e) {
				// ignore
				return result;
			} catch (Exception e) {
				CodeCheckerPlugin.log(e, "in validating "
						+ sourceModule.getPath().toString());
				return result;
			}
			final List<ProblemMarker> problems = visitor.getProblems();
			try {
				IMarker[] markers = sourceModule.getResource().findMarkers(
						CodeCheckerPlugin.MARKER_ID, false,
						IResource.DEPTH_INFINITE);
				for (int i = 0; i < markers.length; i++) {
					boolean keep = false;
					for (ProblemMarker problemMarker : problems) {
						if (problemMarker.equals(markers[i])) {
							problems.remove(problemMarker);
							keep = true;
							break;
						}
					}
					if (!keep) {
						markers[i].delete();
					}
				}
			} catch (CoreException e) {
				CodeCheckerPlugin.log(e);
			}
			PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {
				public void run() {
					for (ProblemMarker problemMarker : problems) {
						try {
							problemMarker.commit();
						} catch (CoreException e) {
							CodeCheckerPlugin.log(e);
						}
					}
				}
			});
			result.setValidated(new IResource[] { resource });
			result.setSeverityError(visitor.getErrors());
			result.setSeverityWarning(visitor.getWarnings());
		}
		return result;
	}
}
