package jp.sourceforge.pdt_tools.variablefilter.preferences;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import jp.sourceforge.pdt_tools.variablefilter.VariableFilter;
import jp.sourceforge.pdt_tools.variablefilter.VariableFilterPlugin;

import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.preferences.IPreferencesService;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckStateProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

public class VariableFilterPreferencePage extends PreferencePage implements
		IWorkbenchPreferencePage {

	private VariableFilter[] filters;

	private CheckboxTableViewer tableViewer;
	private Button newButton;
	private Button editButton;
	private Button removeButton;

	private final static int COLUMN_ENABLED = 0;
	private final static int COLUMN_TYPE = 1;
	private final static int COLUMN_PATTERN = 2;
	private final static String EMPTY_STRING = ""; //$NON-NLS-1$

	@Override
	protected Control createContents(Composite parent) {
		Composite base = new Composite(parent, SWT.NONE);
		GridLayout baseLayout = new GridLayout(2, false);
		baseLayout.horizontalSpacing = 2;
		baseLayout.verticalSpacing = 2;
		base.setLayout(baseLayout);

		Label description = new Label(base, SWT.NONE);
		description.setText(Messages.VariableFilterPreferencePage_description);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 2;
		description.setLayoutData(gridData);

		tableViewer = CheckboxTableViewer.newCheckList(base, SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.BORDER | SWT.FULL_SELECTION);
		tableViewer.setContentProvider(new ArrayContentProvider());
		tableViewer.setLabelProvider(new FilterLabelProvider());
		gridData = new GridData(SWT.FILL, SWT.FILL, true, true);
		Table table = tableViewer.getTable();
		table.setLayoutData(gridData);
		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		String[] columnHeaders = {
				Messages.VariableFilterPreferencePage_enabled_label,
				Messages.VariableFilterPreferencePage_type_label,
				Messages.VariableFilterPreferencePage_pattern_label };
		int[] columnWidths = { 20, 50, 300 };
		for (int i = 0; i < columnHeaders.length; i++) {
			TableColumn column = new TableColumn(table, SWT.NONE);
			column.setText(columnHeaders[i]);
			column.setWidth(columnWidths[i]);
		}
		tableViewer.addCheckStateListener(new ICheckStateListener() {
			public void checkStateChanged(CheckStateChangedEvent event) {
				Object element = event.getElement();
				if (element instanceof VariableFilter) {
					((VariableFilter) element).isEnabled(event.getChecked());
				}
			}
		});
		tableViewer.setCheckStateProvider(new ICheckStateProvider() {
			public boolean isGrayed(Object element) {
				return false;
			}

			public boolean isChecked(Object element) {
				if (element instanceof VariableFilter) {
					return ((VariableFilter) element).isEnabled();
				}
				return false;
			}
		});

		Composite buttons = new Composite(base, SWT.NONE);
		gridData = new GridData(SWT.CENTER, SWT.BEGINNING, false, true);
		gridData.widthHint = 80;
		buttons.setLayoutData(gridData);
		buttons.setLayout(new GridLayout(1, true));
		GridData buttonLayoutData = new GridData(SWT.FILL, SWT.FILL, true,
				false);
		newButton = new Button(buttons, SWT.PUSH);
		newButton
				.setText(Messages.VariableFilterPreferencePage_new_button_label);
		newButton.setLayoutData(buttonLayoutData);
		newButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				newFilter();
			}
		});
		editButton = new Button(buttons, SWT.PUSH);
		editButton
				.setText(Messages.VariableFilterPreferencePage_edit_button_label);
		editButton.setLayoutData(buttonLayoutData);
		editButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				editFilter();
			}
		});
		removeButton = new Button(buttons, SWT.PUSH);
		removeButton
				.setText(Messages.VariableFilterPreferencePage_remove_button_label);
		removeButton.setLayoutData(buttonLayoutData);
		removeButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				removeFilter();
			}
		});

		tableViewer.setInput(filters);

		return base;
	}

	public void init(IWorkbench workbench) {
		setPreferenceStore(VariableFilterPlugin.getDefault()
				.getPreferenceStore());
		IPreferencesService service = Platform.getPreferencesService();
		IScopeContext[] contexts = { InstanceScope.INSTANCE };
		List<VariableFilter> list = new LinkedList<VariableFilter>();
		int count = service.getInt(VariableFilterPlugin.PLUGIN_ID,
				PreferenceConstants.FILTERS, 0, contexts);
		for (int i = 0; i < count; i++) {
			String key = PreferenceConstants.FILTER + i;
			int type = service.getInt(VariableFilterPlugin.PLUGIN_ID, key
					+ PreferenceConstants.TYPE, 0, contexts);
			String pattern = service.getString(VariableFilterPlugin.PLUGIN_ID,
					key + PreferenceConstants.PATTERN, EMPTY_STRING, contexts);
			boolean regex = service.getBoolean(VariableFilterPlugin.PLUGIN_ID,
					key + PreferenceConstants.REGEX, true, contexts);
			boolean enabled = service.getBoolean(
					VariableFilterPlugin.PLUGIN_ID, key
							+ PreferenceConstants.ENABLED, true, contexts);
			if (type != 0 && pattern.length() > 0) {
				list.add(new VariableFilter(type, pattern, regex, enabled));
			}
		}
		filters = list.toArray(new VariableFilter[list.size()]);
	}

	@Override
	protected void performDefaults() {
		IPreferenceStore store = getPreferenceStore();
		List<VariableFilter> list = new LinkedList<VariableFilter>();
		int count = store.getDefaultInt(PreferenceConstants.FILTERS);
		for (int i = 0; i < count; i++) {
			String key = PreferenceConstants.FILTER + i;
			int type = store.getDefaultInt(key + PreferenceConstants.TYPE);
			String pattern = store.getDefaultString(key
					+ PreferenceConstants.PATTERN);
			boolean regex = store.getDefaultBoolean(key
					+ PreferenceConstants.REGEX);
			boolean enabled = store.getDefaultBoolean(key
					+ PreferenceConstants.ENABLED);
			list.add(new VariableFilter(type, pattern, regex, enabled));
		}
		filters = list.toArray(new VariableFilter[list.size()]);
		tableViewer.setInput(filters);
	}

	@Override
	public boolean performOk() {
		IPreferenceStore store = getPreferenceStore();
		int count = store.getInt(PreferenceConstants.FILTERS);
		for (int i = 0; i < count; i++) {
			String key = PreferenceConstants.FILTER + i;
			store.setToDefault(key + PreferenceConstants.TYPE);
			store.setToDefault(key + PreferenceConstants.PATTERN);
			store.setToDefault(key + PreferenceConstants.REGEX);
			store.setToDefault(key + PreferenceConstants.ENABLED);
		}
		count = filters.length;
		store.setValue(PreferenceConstants.FILTERS, count);
		for (int i = 0; i < count; i++) {
			String key = PreferenceConstants.FILTER + i;
			store.setValue(key + PreferenceConstants.TYPE, filters[i].getType());
			store.setValue(key + PreferenceConstants.PATTERN,
					filters[i].getPattern());
			store.setValue(key + PreferenceConstants.REGEX,
					Boolean.toString(filters[i].isRegEx()));
			store.setValue(key + PreferenceConstants.ENABLED,
					Boolean.toString(filters[i].isEnabled()));
		}
		return true;
	}

	private void newFilter() {
		VariableFilterEditDialog dialog = new VariableFilterEditDialog(
				getShell(),
				Messages.VariableFilterPreferencePage_dialog_title_new,
				new VariableFilter(VariableFilter.BY_NAME, EMPTY_STRING, true,
						true));
		int result = dialog.open();
		if (result == IDialogConstants.OK_ID) {
			VariableFilter newFilter = dialog.getFilter();
			List<VariableFilter> list = new LinkedList<VariableFilter>();
			list.addAll(Arrays.asList(filters));
			list.add(newFilter);
			filters = list.toArray(new VariableFilter[list.size()]);
			tableViewer.setInput(filters);
		}
	}

	private void editFilter() {
		ISelection selection = tableViewer.getSelection();
		if (selection instanceof StructuredSelection) {
			Object element = ((StructuredSelection) selection)
					.getFirstElement();
			if (element instanceof VariableFilter) {
				VariableFilter filter = (VariableFilter) element;
				VariableFilterEditDialog dialog = new VariableFilterEditDialog(
						getShell(),
						Messages.VariableFilterPreferencePage_dialog_title_edit,
						filter);
				int result = dialog.open();
				if (result == IDialogConstants.OK_ID) {
					tableViewer.update(element, null);
				}
			}
		}
	}

	private void removeFilter() {
		ISelection selection = tableViewer.getSelection();
		if (selection instanceof StructuredSelection) {
			Object element = ((StructuredSelection) selection)
					.getFirstElement();
			if (element instanceof VariableFilter) {
				List<VariableFilter> list = new LinkedList<VariableFilter>();
				for (VariableFilter filter : filters) {
					if (!filter.equals(element)) {
						list.add(filter);
					}
				}
				filters = list.toArray(new VariableFilter[list.size()]);
				tableViewer.setInput(filters);
			}
		}
	}

	/**
	 * 
	 */
	private class FilterLabelProvider extends LabelProvider implements
			ITableLabelProvider {

		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		public String getColumnText(Object element, int columnIndex) {
			if (element instanceof VariableFilter) {
				VariableFilter filter = (VariableFilter) element;
				switch (columnIndex) {
				case COLUMN_ENABLED:
					return EMPTY_STRING;
				case COLUMN_TYPE:
					return VariableFilter.getTypeName(filter.getType());
				case COLUMN_PATTERN:
					return filter.getPattern();
				}
			}
			return null;
		}
	}
}
