package jp.sourceforge.pdt_tools.codeFolding;

import java.util.List;
import java.util.Map;
import java.util.Stack;

import jp.sourceforge.pdt_tools.codeFolding.preferences.FoldingPreference;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.projection.ProjectionAnnotation;
import org.eclipse.php.internal.core.ast.nodes.ASTNode;
import org.eclipse.php.internal.core.ast.nodes.ArrayCreation;
import org.eclipse.php.internal.core.ast.nodes.Block;
import org.eclipse.php.internal.core.ast.nodes.CatchClause;
import org.eclipse.php.internal.core.ast.nodes.ClassDeclaration;
import org.eclipse.php.internal.core.ast.nodes.Comment;
import org.eclipse.php.internal.core.ast.nodes.DoStatement;
import org.eclipse.php.internal.core.ast.nodes.ForEachStatement;
import org.eclipse.php.internal.core.ast.nodes.ForStatement;
import org.eclipse.php.internal.core.ast.nodes.FunctionDeclaration;
import org.eclipse.php.internal.core.ast.nodes.IfStatement;
import org.eclipse.php.internal.core.ast.nodes.InterfaceDeclaration;
import org.eclipse.php.internal.core.ast.nodes.MethodDeclaration;
import org.eclipse.php.internal.core.ast.nodes.Quote;
import org.eclipse.php.internal.core.ast.nodes.SwitchCase;
import org.eclipse.php.internal.core.ast.nodes.SwitchStatement;
import org.eclipse.php.internal.core.ast.nodes.TryStatement;
import org.eclipse.php.internal.core.ast.nodes.WhileStatement;

@SuppressWarnings("restriction")
public class FoldingVisitor extends EmptyVisitor {

	private int foldClasses;
	private int foldInterfaces;
	private int foldMethods;
	private int foldFunctions;
	private int foldPHPDoc;
	private int foldComments;
	private int foldIf;
	private int foldFor;
	private int foldWhile;
	private int foldDo;
	private int foldSwitch;
	private int foldCase;
	private int foldTry;
	private int foldCatch;
	private int foldArray;
	private int foldHereDoc;
	private int foldRegion;
	private int foldBlocks;

	private IDocument document;
	private Map<Annotation, Position> map;
	private int currentOffset;
	private boolean isMethod;

	private Stack<ASTNode> regionStack;
	private String regionBeginning;
	private String regionEnding;

	public FoldingVisitor(IDocument document, int offset,
			Map<Annotation, Position> map) {
		this.document = document;
		this.map = map;
		currentOffset = offset;
		isMethod = false;

		IPreferenceStore pref = CodeFolding.getDefault().getPreferenceStore();
		foldClasses = pref.getInt(FoldingPreference.FOLD_CLASSES);
		foldInterfaces = pref.getInt(FoldingPreference.FOLD_INTERFACES);
		foldMethods = pref.getInt(FoldingPreference.FOLD_METHODS);
		foldFunctions = pref.getInt(FoldingPreference.FOLD_FUNCTIONS);
		foldPHPDoc = pref.getInt(FoldingPreference.FOLD_PHPDOC);
		foldComments = pref.getInt(FoldingPreference.FOLD_COMMENTS);
		foldIf = pref.getInt(FoldingPreference.FOLD_IF);
		foldFor = pref.getInt(FoldingPreference.FOLD_FOR);
		foldWhile = pref.getInt(FoldingPreference.FOLD_WHILE);
		foldDo = pref.getInt(FoldingPreference.FOLD_DO);
		foldSwitch = pref.getInt(FoldingPreference.FOLD_SWITCH);
		foldCase = pref.getInt(FoldingPreference.FOLD_CASE);
		foldTry = pref.getInt(FoldingPreference.FOLD_TRY);
		foldCatch = pref.getInt(FoldingPreference.FOLD_CATCH);
		foldArray = pref.getInt(FoldingPreference.FOLD_ARRAY);
		foldHereDoc = pref.getInt(FoldingPreference.FOLD_HEREDOC);
		foldRegion = pref.getInt(FoldingPreference.FOLD_REGION);
		if (foldRegion != FoldingPreference.FOLD_DISABLE) {
			regionStack = new Stack<ASTNode>();
			regionBeginning = pref
					.getString(FoldingPreference.REGION_BEGINNING);
			regionEnding = pref.getString(FoldingPreference.REGION_ENDING);
		}
		foldBlocks = pref.getInt(FoldingPreference.FOLD_BLOCKS);
	}

	private void createAnnotation(ASTNode node, int folding) {
		createAnnotation(node.getStart(), node.getLength(), folding);
	}

	private void createAnnotation(int offset, int length, int folding) {
		if (folding == FoldingPreference.FOLD_DISABLE) {
			return;
		}
		try {
			int startLine = document.getLineOfOffset(offset);
			int endLine = document.getLineOfOffset(offset + length);
			if (startLine == endLine) {
				return;
			}
			int startOffset = document.getLineOffset(startLine);
			int endOffset = document.getLineOffset(endLine)
					+ document.getLineLength(endLine);
			length = endOffset - startOffset;
			offset = startOffset;
			int docLength = document.getLength();
			if (offset < 0 || offset > docLength) {
				CodeFolding.log(IStatus.ERROR, "Invalid offset");
				return;
			}
			if (length < 0 || offset + length > docLength) {
				CodeFolding.log(IStatus.ERROR, "Invalid length");
				return;
			}
			Position position = new Position(offset, length);
			if (map.containsValue(position)) {
				return;
			}
			Boolean markCollapsed = false;
			if (folding == FoldingPreference.FOLD_INITIALLY) {
				if (currentOffset < offset || offset + length < currentOffset) {
					markCollapsed = true;
				}
			}
			ProjectionAnnotation annotation = new ProjectionAnnotation(
					markCollapsed);
			map.put(annotation, position);
		} catch (BadLocationException e) {
			CodeFolding.log(e);
		}
	}

	@Override
	public boolean visit(ArrayCreation arrayCreation) {
		createAnnotation(arrayCreation, foldArray);
		return super.visit(arrayCreation);
	}

	@Override
	public boolean visit(Block block) {
		if (block.isCurly()) {
			createAnnotation(block, foldBlocks);
		}
		return super.visit(block);
	}

	@Override
	public boolean visit(ClassDeclaration classDeclaration) {
		createAnnotation(classDeclaration, foldClasses);
		return super.visit(classDeclaration);
	}

	@Override
	public boolean visit(Comment comment) {
		switch (comment.getCommentType()) {
		case Comment.TYPE_PHPDOC:
			createAnnotation(comment, foldPHPDoc);
			break;
		case Comment.TYPE_MULTILINE:
			createAnnotation(comment, foldComments);
			break;
		}
		if (foldRegion != FoldingPreference.FOLD_DISABLE) {
			try {
				String commentBody = document.get(comment.getStart(),
						comment.getLength()).trim();
				if (commentBody.startsWith(regionBeginning)) {
					regionStack.push(comment);
				} else if (commentBody.startsWith(regionEnding)) {
					if (!regionStack.isEmpty()) {
						ASTNode beginning = regionStack.pop();
						int start = beginning.getStart();
						int last = comment.getEnd() - 1;
						createAnnotation(start, last - start, foldRegion);
					}
				}
			} catch (BadLocationException e) {
				CodeFolding.log(e);
			}
		}
		return super.visit(comment);
	}

	@Override
	public boolean visit(DoStatement doStatement) {
		createAnnotation(doStatement, foldDo);
		return super.visit(doStatement);
	}

	@Override
	public boolean visit(ForEachStatement forEachStatement) {
		createAnnotation(forEachStatement, foldFor);
		return super.visit(forEachStatement);
	}

	@Override
	public boolean visit(ForStatement forStatement) {
		createAnnotation(forStatement, foldFor);
		return super.visit(forStatement);
	}

	@Override
	public boolean visit(FunctionDeclaration functionDeclaration) {
		if (isMethod) {
			createAnnotation(functionDeclaration, foldMethods);
		} else {
			createAnnotation(functionDeclaration, foldFunctions);
		}
		return super.visit(functionDeclaration);
	}

	@Override
	public boolean visit(IfStatement ifStatement) {
		createAnnotation(ifStatement, foldIf);
		return super.visit(ifStatement);
	}

	@Override
	public boolean visit(InterfaceDeclaration interfaceDeclaration) {
		createAnnotation(interfaceDeclaration, foldInterfaces);
		return super.visit(interfaceDeclaration);
	}

	@Override
	public boolean visit(MethodDeclaration methodDeclaration) {
		isMethod = true;
		return super.visit(methodDeclaration);
	}

	@Override
	public void endVisit(MethodDeclaration methodDeclaration) {
		isMethod = false;
	}

	@Override
	public boolean visit(Quote quote) {
		if (quote.getQuoteType() == Quote.QT_HEREDOC) {
			createAnnotation(quote, foldHereDoc);
		}
		return super.visit(quote);
	}

	@Override
	public boolean visit(SwitchStatement switchStatement) {
		createAnnotation(switchStatement, foldSwitch);
		return super.visit(switchStatement);
	}

	@Override
	public boolean visit(SwitchCase switchCase) {
		createAnnotation(switchCase, foldCase);
		return super.visit(switchCase);
	}

	@Override
	public boolean visit(TryStatement tryStatement) {
		createAnnotation(tryStatement, foldTry);
		return super.visit(tryStatement);
	}

	@Override
	public boolean visit(CatchClause catchClause) {
		ASTNode node = catchClause.getParent();
		while (node.getType() != ASTNode.TRY_STATEMENT) {
			node = node.getParent();
			if (node == null || node.getType() == ASTNode.PROGRAM) {
				return super.visit(catchClause);
			}
		}
		List<CatchClause> list = ((TryStatement) node).catchClauses();
		try {
			for (int i = 0; i < list.size(); i++) {
				CatchClause clause = list.get(i);
				if (clause.equals(catchClause)) {
					if (i < list.size() - 1) {
						CatchClause clause2 = list.get(i + 1);
						int end = document.getLineOfOffset(clause.getEnd());
						int start = document
								.getLineOfOffset(clause2.getStart());
						if (end == start && end > 0) {
							IRegion region = document
									.getLineInformation(end - 1);
							int len = region.getOffset() + region.getLength()
									- catchClause.getStart();
							createAnnotation(catchClause.getStart(), len,
									foldCatch);
							return super.visit(catchClause);
						}
					}
					break;
				}
			}
		} catch (BadLocationException e) {
			// ignore
		}
		createAnnotation(catchClause, foldCatch);
		return super.visit(catchClause);
	}

	@Override
	public boolean visit(WhileStatement whileStatement) {
		createAnnotation(whileStatement, foldWhile);
		return super.visit(whileStatement);
	}

}
