﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Text;
using System.Runtime.InteropServices;
using System.Reflection;
using System.IO;
using System.Windows.Forms;
using System.Xml;

using Lugens.Utils;

namespace Lugens.Passer
{
    public class EngineInfo
    {
        /// <summary>
        /// 一意のID
        /// </summary>
        public string Id;

        /// <summary>
        /// コマンド名
        /// </summary>
        public string Name;

        /// <summary>
        /// タイプ(0.COMMAND 1.LOCAL_SEARCH 2.WEB_SEARCH)
        /// </summary>
        public int Type;

        /// <summary>
        /// URL
        /// </summary>
        public string Url;

        /// <summary>
        /// 検索URL
        /// </summary>
        public string SearchUrl;

        /// <summary>
        /// サジェストURL
        /// </summary>
        public string SuggestUrl;

        /// <summary>
        /// エンコーディング
        /// </summary>
        public string Encoding;

        /// <summary>
        /// アイコン
        /// </summary>
        //public Icon Icon;

        /// <summary>
        /// アイコン画像
        /// </summary>
        public Image IconImage;

        /// <summary>
        /// 表示名
        /// </summary>
        public string Display;

        /// <summary>
        /// ショートカットキー
        /// </summary>
        public int Shortcut;

        /// <summary>
        /// 削除したか
        /// </summary>
        public bool Removed;

        /// <summary>
        /// アイコンを変更したか
        /// </summary>
        public bool IconChanged;

    }

    public static partial class Program
    {
        private static string chars = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

        public static readonly string EngineIconDir = Application.StartupPath + @"\engine";
        public static readonly string EngineIconTmpDir = Application.StartupPath + @"\engine\tmp";
        public static readonly string EngineFileName = Application.StartupPath + @"\EngineList.xml";

        /// <summary>
        /// 検索エンジンリスト
        /// </summary>
        public static List<string> EngineList = new List<string>();
        public static Dictionary<string, EngineInfo> EngineDic = new Dictionary<string, EngineInfo>();

        /// <summary>
        /// 検索エンジンリスト
        /// </summary>
        public static List<string> DefaultEngineList = new List<string>();
        public static Dictionary<string, EngineInfo> DefaultEngineDic = new Dictionary<string, EngineInfo>();

        /// <summary>
        /// ホットキー押下時の検索エンジンリスト
        /// </summary>
        public static List<int> HotKeyEngineList = new List<int>();

        /// <summary>
        /// 検索エンジンのアイコン用テンポラリディレクトリ作成
        /// </summary>
        public static void CreateEngineIconTmpDir()
        {
            if(Directory.Exists(Program.EngineIconTmpDir))
                return;

            if (!Directory.Exists(Program.EngineIconDir))
                Directory.CreateDirectory(Program.EngineIconDir);

            if (!Directory.Exists(Program.EngineIconTmpDir))
                Directory.CreateDirectory(Program.EngineIconTmpDir);
        }

        /// <summary>
        /// 検索エンジンのインデックスを返す
        /// </summary>
        /// <param name="name"></param>
        /// <returns></returns>
        public static int GetEngineIndex(string name)
        {
            int i = 0;
            foreach (string key in Program.DefaultEngineList)
            {
                if(name.Equals(Program.DefaultEngineDic[key].Name))
                    return i;
                i++;
            }
            foreach (string key in Program.EngineList)
            {
                if(name.Equals(Program.EngineDic[key].Name))
                    return i;
                i++;
            }
            return -1;
        }

        /// <summary>
        /// 検索エンジンを返す
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public static EngineInfo GetEngineInfo(int index)
        {
            if(index < Program.DefaultEngineList.Count)
                return Program.DefaultEngineDic[Program.DefaultEngineList[index]];

            index -= Program.DefaultEngineList.Count;
            if (index < Program.EngineList.Count)
                return Program.EngineDic[Program.EngineList[index]];

            return null;
        }

        public static EngineInfo GetEngineInfoFromHotKey(int hotkey)
        {
            if (!Program.HotKeyEngineList.Contains(hotkey))
                return null;

            int index;
            int sindex = Program.MainForm.SelectedEngineIndex;
            int dcount = Program.DefaultEngineList.Count;
            int count = dcount + Program.EngineList.Count;
            EngineInfo info;
            for (int i = 1; i < count; i++)
            {
                index = sindex + i;
                if (index >= count)
                    index -= count;

                if (index < dcount)
                {
                    info = Program.DefaultEngineDic[Program.DefaultEngineList[index]];
                    if (info.Shortcut == hotkey)
                        return info;
                }
                else
                {
                    index -= dcount;
                    info = Program.EngineDic[Program.EngineList[index]];
                    if (info.Shortcut == hotkey)
                        return info;
                }
            }
            return null;
        }

        /// <summary>
        /// 検索エンジンの一意IDを作成
        /// </summary>
        /// <returns></returns>
        public static string GetEngineID()
        {
            string id;
            bool check;
            Random r = new Random();
            do
            {
                id = "";
                for (int i = 0; i < 4; i++)
                {
                    id += chars[r.Next(0, chars.Length - 1)];
                }

                check = false;
                foreach (EngineInfo info in Program.DefaultEngineDic.Values)
                {
                    if (id.Equals(info.Id))
                    {
                        check = true;
                        break;
                    }
                }

                if (check)
                    continue;

                foreach (EngineInfo info in Program.EngineDic.Values)
                {
                    if (id.Equals(info.Id))
                    {
                        check = true;
                        break;
                    }
                }

                if (check)
                    continue;

                foreach (EngineInfo info in Program.SettingForm.EngineDicTmp.Values)
                {
                    if (id.Equals(info.Id))
                    {
                        check = true;
                        break;
                    }
                }


            } while (check);

            return id;
        }


        /// <summary>
        /// 検索エンジンリスト読み込み
        /// </summary>
        /// <returns></returns>
        public static bool ReadEngineList()
        {
            SetDefaultEngineList();

            string file = null;
            if (Settings.ContainsKey(Settings.PASSER_ENGINELIST_DIR))
            {
                file = Settings.Get(Settings.PASSER_ENGINELIST_DIR);
                if (file[file.Length - 1] == '\\')
                    file = file.Substring(0, file.Length - 1);
                file += @"\EngineList.xml";
                if (!File.Exists(Program.EngineFileName))
                    file = null;
            }

            if (file == null)
                file = Program.EngineFileName;

            if (!File.Exists(file))
                return false;

            EngineInfo info = new EngineInfo();
            XmlTextReader reader = null;
            Program.EngineList.Clear();
            Program.EngineDic.Clear();

            try
            {
                using (reader = new XmlTextReader(file))
                {
                    while (reader.Read())
                    {
                        if (reader.IsStartElement() && !reader.IsEmptyElement)
                        {
                            if (reader.LocalName.Equals("Engine"))
                            {
                                info = new EngineInfo();
                                continue;
                            }
                            if (reader.LocalName.Equals("Id"))
                            {
                                reader.Read();
                                info.Id = reader.Value;
                                continue;
                            }
                            if (reader.LocalName.Equals("Name"))
                            {
                                reader.Read();
                                info.Name = reader.Value;
                                continue;
                            }
                            if (reader.LocalName.Equals("Url"))
                            {
                                reader.Read();
                                info.Url = reader.Value;
                                continue;
                            }
                            if (reader.LocalName.Equals("SearchUrl"))
                            {
                                reader.Read();
                                info.SearchUrl = reader.Value;
                                continue;
                            }
                            if (reader.LocalName.Equals("SuggestUrl"))
                            {
                                reader.Read();
                                info.SuggestUrl = reader.Value;
                                continue;
                            }
                            if (reader.LocalName.Equals("Encoding"))
                            {
                                reader.Read();
                                info.Encoding = reader.Value;
                                continue;
                            }
                            if (reader.LocalName.Equals("Display"))
                            {
                                reader.Read();
                                info.Display = reader.Value;
                                continue;
                            }
                            if (reader.LocalName.Equals("Shortcut"))
                            {
                                reader.Read();
                                info.Shortcut = int.Parse(reader.Value);
                                continue;
                            }
                        }
                        else if (reader.LocalName.Equals("Engine"))
                        {
                            info.Type = 2;
                            Bitmap bitmap = new Bitmap(Program.EngineIconDir + "\\" + info.Id);
                            info.IconImage = new Bitmap(bitmap);
                            bitmap.Dispose();

                            Program.EngineList.Add(info.Name);
                            Program.EngineDic.Add(info.Name, info);
                            if(info.Shortcut > 0 && !Program.HotKeyEngineList.Contains(info.Shortcut))
                                Program.HotKeyEngineList.Add(info.Shortcut);
                        }
                    }
                }
                return true;
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString());
                return false;
            }
        }

        /// <summary>
        /// 検索エンジンリスト書き込み
        /// </summary>
        public static void WriteEngineList()
        {
            Version version = Assembly.GetExecutingAssembly().GetName().Version;
            XmlTextWriter writer = null;

            using (writer = new XmlTextWriter(Program.EngineFileName, Encoding.UTF8))
            {
                writer.Formatting = Formatting.Indented;

                writer.WriteStartDocument();
                writer.WriteStartElement(Assembly.GetExecutingAssembly().GetName().Name);
                writer.WriteAttributeString("version", version.Major.ToString() + "." + version.Minor.ToString());
                writer.WriteStartElement("EngineList");

                foreach (string key in Program.EngineList)
                {
                    EngineInfo info = Program.EngineDic[key];
                    if (info.Removed)
                    {
                        string file = Program.EngineIconDir + "\\" + info.Id;
                        if (File.Exists(file))
                            File.Delete(file);

                        Program.EngineDic.Remove(key);
                        continue;
                    }
                    writer.WriteStartElement("Engine");

                    writer.WriteStartElement("Id");
                    writer.WriteString(info.Id);
                    writer.WriteEndElement();

                    writer.WriteStartElement("Name");
                    writer.WriteString(info.Name);
                    writer.WriteEndElement();

                    writer.WriteStartElement("Url");
                    writer.WriteString(info.Url);
                    writer.WriteEndElement();

                    writer.WriteStartElement("SearchUrl");
                    writer.WriteString(info.SearchUrl);
                    writer.WriteEndElement();

                    writer.WriteStartElement("SuggestUrl");
                    writer.WriteString(info.SuggestUrl);
                    writer.WriteEndElement();

                    writer.WriteStartElement("Encoding");
                    writer.WriteString(info.Encoding);
                    writer.WriteEndElement();
                    
                    writer.WriteStartElement("Display");
                    writer.WriteString(info.Display);
                    writer.WriteEndElement();

                    writer.WriteStartElement("Shortcut");
                    writer.WriteString(info.Shortcut.ToString());
                    writer.WriteEndElement();

                    writer.WriteEndElement();
                    info.IconImage.Save(Program.EngineIconDir + "\\" + info.Id, System.Drawing.Imaging.ImageFormat.Png);
                }
                writer.WriteEndElement();
                writer.WriteEndElement();
                writer.Flush();
                writer.Close();
            }
            if(Directory.Exists(Program.EngineIconTmpDir))
                Directory.Delete(Program.EngineIconTmpDir, true);
        }

        /// <summary>
        /// デフォルトエンジンをセットする
        /// </summary>
        private static void SetDefaultEngineList()
        {
            Program.DefaultEngineList.Clear();
            Program.DefaultEngineDic.Clear();
            Program.HotKeyEngineList.Clear();
            EngineInfo info = new EngineInfo();
            info.Id = "0001";
            info.Name = "コマンドラインランチャー";
            info.Type = 0;
            info.Shortcut = 4186; //RControl + Z
            info.IconImage = Properties.Resources.EngineCommand.ToBitmap();
            info.Display = "コマンドを入力してください";
            Program.DefaultEngineList.Add(info.Name);
            Program.DefaultEngineDic.Add(info.Name, info);
            Program.HotKeyEngineList.Add(info.Shortcut);

            info = new EngineInfo();
            info.Id = "0002";
            info.Name = "ファイル検索";
            info.Type = 1;
            info.Shortcut = 4166; //RControl + F
            info.IconImage = Properties.Resources.EngineSearch.ToBitmap();
            info.Display = "ファイル名を入力してください";
            Program.DefaultEngineList.Add(info.Name);
            Program.DefaultEngineDic.Add(info.Name, info);
            Program.HotKeyEngineList.Add(info.Shortcut);
        }
    }
}
