﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using Lugens.Utils;
using System.Runtime.InteropServices;

namespace Lugens.Passer
{
    public partial class PopupForm : Form
    {
        private const int SWP_NOSIZE = 1;
        private const int SWP_NOMOVE = 2;
        private const int SWP_NOACTIVATE = 0x10;
        private const int HWND_TOPMOST = -1;
        private const int GWL_EXSTYLE = -20;

        private const int CURSOR_CHANGE_RANGE = 4;

        private int cursorType = 0;

        private bool cursorDrag = false;

        private delegate void SetTextDelegate(string text);

        private Rectangle drawRectangle = new Rectangle();

        private Point mousePoint = new Point(0, 0);

        private Bitmap holdBitmap = Lugens.Passer.Properties.Resources.hold;

        private int borderSpace;
        public int BorderSpace
        {
            get { return borderSpace; }
            set
            {
                borderSpace = value;
            }
        }

        public override string Text
        {
            get { return base.Text; }
            set
            {
                base.Text = value;
                using (Graphics g = this.CreateGraphics())
                {
                    Size size = g.MeasureString(value, this.Font, 600).ToSize();
                    size.Width += 14;
                    size.Height += 8;
                    this.Size = size;
                }
                this.Locate();

            }
        }

        public bool Loading
        {
            get { return pictureBox_loading.Visible; }
            set { pictureBox_loading.Visible = value; }
        }

        public PopupForm()
        {
            InitializeComponent();
            this.BorderSpace = 3;
        }

        protected override void CreateHandle()
        {
            base.CreateHandle();
            Win32.SetWindowPos(new HandleRef(this, this.Handle), (IntPtr)HWND_TOPMOST, 0, 0, 0, 0, SWP_NOSIZE | SWP_NOMOVE | SWP_NOACTIVATE);
        }

        protected override CreateParams CreateParams
        {
            get
            {
                CreateParams cp = base.CreateParams;
                //cp.Style = (-2147483648) | 0x00040000;
                cp.ClassStyle |= 0x00020000;
                cp.ExStyle |= Win32.WS_EX_NOACTIVATE;
                return cp;
            }
        }

        protected override bool ShowWithoutActivation
        {
            get { return true; }
        }

        public void RequestSetText(string text)
        {
            Invoke(new SetTextDelegate(SetText), new object[]{text});
        }

        public void SetText(string text)
        {
            this.Loading = false;
            this.Text = text;
        }

        public void Locate()
        {
            Point p = this.Location;
            Rectangle workingArea = Screen.FromPoint(this.Location).WorkingArea;
            if (this.Location.X + this.Size.Width > workingArea.Width)
                p.X = workingArea.Width - this.Size.Width;
            if (this.Location.Y + this.Size.Height > workingArea.Height)
                p.Y = workingArea.Height - this.Size.Height;
            this.Location = p;
        }

        public void DoVisibleChange(bool visible)
        {
            if (visible)
            {
                Program.Status = ProgramStatus.PopupFormOpen;
                this.Visible = true;
            }
            else
            {
                Program.Status = ProgramStatus.Waiting;
                this.Close();
            }
        }

        private void DoResize(int x, int y)
        {
            int w, h;
            Point locationNew = this.Location;
            Size sizeNew = this.Size;

            if (this.cursorType == 1 || this.cursorType == 4 || this.cursorType == 7)
            {
                if (sizeNew.Width - x < this.MinimumSize.Width)
                {
                    locationNew.X += this.ClientRectangle.Width - this.MinimumSize.Width;
                    sizeNew.Width = this.MinimumSize.Width;
                }
                else
                {
                    locationNew.X += x;
                    sizeNew.Width -= x;
                }
            }

            if (this.cursorType == 7 || this.cursorType == 8 || this.cursorType == 9)
            {
                if (sizeNew.Height - y < this.MinimumSize.Height)
                {
                    locationNew.Y += this.ClientRectangle.Height - this.MinimumSize.Height;
                    sizeNew.Height = this.MinimumSize.Height;
                }
                else
                {
                    locationNew.Y += y;
                    sizeNew.Height -= y;
                }
            }

            if (this.cursorType == 3 || this.cursorType == 6 || this.cursorType == 9)
            {
                w = x + this.mousePoint.X;
                if (w < this.MinimumSize.Width)
                {
                    sizeNew.Width = this.MinimumSize.Width;
                }
                else
                {
                    sizeNew.Width = w;
                }
            }

            if (this.cursorType == 1 || this.cursorType == 2 || this.cursorType == 3)
            {
                h = y + this.mousePoint.Y;
                if (h < this.MinimumSize.Height)
                {
                    sizeNew.Height = this.MinimumSize.Height;
                }
                else
                {
                    sizeNew.Height = h;
                }
            }

            this.Location = locationNew;
            this.Size = sizeNew;
            this.Invalidate();
        }

        private void PopupForm_MouseMove(object sender, MouseEventArgs e)
        {
            if (this.cursorDrag && (e.Button & MouseButtons.Left) == MouseButtons.Left)
            {
                int x = e.X + this.mousePoint.X;
                int y = e.Y + this.mousePoint.Y;
                this.DoResize(e.X, e.Y);
                return;
            }

            if (e.X <= CURSOR_CHANGE_RANGE && e.Y <= CURSOR_CHANGE_RANGE)
            {
                this.cursorType = 7;
                this.Cursor = Cursors.SizeNWSE;
            }
            else if (e.X >= this.Size.Width - 3 && e.Y <= CURSOR_CHANGE_RANGE)
            {
                this.cursorType = 9;
                this.Cursor = Cursors.SizeNESW;
            }
            else if (e.X <= CURSOR_CHANGE_RANGE && e.Y >= this.Size.Height - CURSOR_CHANGE_RANGE)
            {
                this.cursorType = 1;
                this.Cursor = Cursors.SizeNESW;
            }
            else if (e.X >= this.Size.Width - this.holdBitmap.Width && e.Y >= this.Size.Height - this.holdBitmap.Height)
            {
                this.cursorType = 3;
                this.Cursor = Cursors.SizeNWSE;
            }
            else if (e.Y <= CURSOR_CHANGE_RANGE)
            {
                this.cursorType = 8;
                this.Cursor = Cursors.SizeNS;
            }
            else if (e.Y >= this.Size.Height - CURSOR_CHANGE_RANGE)
            {
                this.cursorType = 2;
                this.Cursor = Cursors.SizeNS;
            }
            else if (e.X <= CURSOR_CHANGE_RANGE)
            {
                this.cursorType = 4;
                this.Cursor = Cursors.SizeWE;
            }
            else if (e.X >= this.Size.Width - CURSOR_CHANGE_RANGE)
            {
                this.cursorType = 6;
                this.Cursor = Cursors.SizeWE;
            }
            else
            {
                this.cursorType = 0;
                this.Cursor = Cursors.Default;
            }
        }

        private void PopupForm_MouseUp(object sender, MouseEventArgs e)
        {
            this.cursorDrag = false;
        }

        private void PopupForm_MouseDown(object sender, MouseEventArgs e)
        {
            if (this.cursorType != 0)
            {
                this.cursorDrag = true;
                this.mousePoint.X = this.Size.Width - e.X;
                this.mousePoint.Y = this.Size.Height - e.Y;
            }
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            e.Graphics.DrawString(this.Text, this.Font, Brushes.Black, this.drawRectangle);
            using (Pen pen = new Pen(this.ForeColor, 1.0F))
            {
                e.Graphics.DrawRectangle(pen, 0, 0, this.ClientSize.Width - 1, this.ClientSize.Height - 1);
            }

            e.Graphics.DrawImage(this.holdBitmap, new Point(this.Size.Width - 17, this.Size.Height - 17));
        }

        private void PopupForm_SizeChanged(object sender, EventArgs e)
        {
            Point p = new Point(this.Size.Width >> 1, this.Size.Height >> 1);
            p.X -= this.pictureBox_loading.Size.Width >> 1;
            p.Y -= this.pictureBox_loading.Size.Height >> 1;
            this.pictureBox_loading.Location = p;
            this.drawRectangle = new Rectangle(2, 2, this.Width - 10, this.Height - 10);
        }

        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case Program.WM_APP_KHOOK:
                    if ((int)m.WParam == Win32.WM_KEYDOWN || (int)m.WParam == Win32.WM_SYSKEYDOWN)
                    {
                        this.DoVisibleChange(false);
                    }
                    break;
            }
            base.WndProc(ref m);
        }

    }
}
