
/* --------------------------------------------- */
/*  H8-3069F Serial communicate function         */
/*                                               */
/*  CPU    : Renesus H8/3069F 25MHz              */
/*  Memory : ROM 512KB, RAM 16KB E-RAM 2MB       */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "serial.h"
#include "key.h"

#define READ_BUF_SIZE 16
#define SENT_BUF_SIZE 16

// #define RECEIVE_DATA_EVENIF_ERR

// -------------------------------------------
//  Macros
// -------------------------------------------
typedef struct {
	unsigned char speed_setting_index;
	unsigned char code_setting_index;
	unsigned char param_setting_index;
	unsigned char flow_control_request;
} SCI_SETTING, *P_SCI_SETTING;

typedef struct {
	unsigned long file_size;
	unsigned long sent_bytes;
	unsigned char* p_buf;
	unsigned char tx_in_progress;
} TX_STATUS, *P_TX_STATUS;

// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
// Global
void serial_1ms_handler(void);
int  serial_initialize(void);
void  serial_process(void);
void ui_serial_progress( void );
unsigned char serial_status(REQUEST_TO_CLASS req);

//int ui_function_serial(UI_COMMAND uicmd);
void int_ERI0();
void int_RXI0();
void int_TXI0();
void int_TEI0();

unsigned char TransferStartRequest( unsigned char* pBuf, unsigned long size );
int ui_function_serial(UI_COMMAND uicmd);

// Local
static unsigned char Cnv2Iso( unsigned char ascii_char );
static unsigned char Cnv2Eia( unsigned char ascii_char );
static int ui_function_serial_nop(UI_COMMAND uicmd, int param);
static int ui_function_serial_debug(UI_COMMAND uicmd, int param);
static int ui_function_serial_config(UI_COMMAND uicmd, int param);
static int ui_function_serial_trans(UI_COMMAND uicmd, int param);
static int ui_function_serial_trans_debug(UI_COMMAND uicmd, int param);

// -------------------------------------------
//  Variables
// -------------------------------------------
// Global
// Locals
static int serial_disable_timer;
static unsigned char serial_proc;
static unsigned char serial_error;
static unsigned char ui_serial_proc;

static SCI_SETTING 	sci_setting;
static TX_STATUS	tx_status;
static unsigned char ReceiveBuf[READ_BUF_SIZE];
static unsigned char SentBuf[SENT_BUF_SIZE];
static unsigned char ReceiveCnt;
static unsigned char ReadCnt;
static unsigned int  SCIErrorDebug;


enum serial_flow_ctl_req {
	FLOW_REQ_NONE,
	FLOW_REQ_PAUSE,
	FLOW_REQ_RESUME,
};


enum serial_error_code {
	SCI_ERR_NONE,
	SCI_PROCESS_ERROR = 100,
};

enum serial_process_mode {
	SCI_INITIALIZE,
	SCI_READY_WAIT,				
	SCI_IDLE = 0x80,			// Idle
	SCI_DATA_SET,
	SCI_WAIT_TX_OK,
	
	
	SCI_TX_IN_PROGRESS,
	SCI_TX_IN_PAUSE,
	SCI_TX_IN_END,
	SCI_ERROR_STOP = 0xFF,
};

static const SPEED_SETTING speed_setting[] = {
	{	COMM_SPEED_UNKNOWN,		1,		  0,		"        ---- bps", },
	{	COMM_SPEED_1200BPS,		1,		162,		"        1200 bps", },
	{	COMM_SPEED_2400BPS,		1,		 80,		"        2400 bps", },
	{	COMM_SPEED_4800BPS,		0,		162,		"        4800 bps", },
	{	COMM_SPEED_9600BPS,		0,		 80,		"        9600 bps", },
	{	COMM_SPEED_19200BPS,	0,		 40,		"       19200 bps", },
	{	0xFF,					0,		  0,		"0123456789ABCDEF", },
};
static const CODE_SETTING code_setting[] = {
	{	COMM_CODE_UNKNOWN,	"         Unknown", },
	{	COMM_CODE_ISO,		"             ISO", },
	{	COMM_CODE_EIA,		"             EIA", },
	{	COMM_CODE_ASCII,	"           ASCII", },
	{	0xFF,				"0123456789ABCDEF", },
};

static const PARAM_SETTING param_setting[] = {
	{	COMM_PARAM_UNKNOWN,						0x00,	"  Unknown       ", },
	{	COMM_PARAM_8BIT_NONPARITY_STOP1,		0x00,	" 8bit/NonP/Stop1", },	// SMR=0000 00..   << Default
	{	COMM_PARAM_8BIT_NONPARITY_STOP2,		0x08,	" 8bit/NonP/Stop2", },	// SMR=0000 10.. 
	{	COMM_PARAM_8BIT_ODDPARITY_STOP1,		0x30,	" 8bit/OddP/Stop1", },	// SMR=0011 00.. 
	{	COMM_PARAM_8BIT_ODDPARITY_STOP2,		0x38,	" 8bit/OddP/Stop2", },	// SMR=0011 10.. 
	{	COMM_PARAM_8BIT_EVNPARITY_STOP1,		0x20,	" 8bit/EvnP/Stop1", },	// SMR=0010 00.. 
	{	COMM_PARAM_8BIT_EVNPARITY_STOP2,		0x28,	" 8bit/EvnP/Stop2", },	// SMR=0010 10.. 
	{	COMM_PARAM_7BIT_NONPARITY_STOP1,		0x40,	" 7bit/NonP/Stop1", },	// SMR=0100 00.. 
	{	COMM_PARAM_7BIT_NONPARITY_STOP2,		0x48,	" 7bit/NonP/Stop2", },	// SMR=0100 10.. 
	{	COMM_PARAM_7BIT_ODDPARITY_STOP1,		0x70,	" 7bit/OddP/Stop1", },	// SMR=0111 00.. 
	{	COMM_PARAM_7BIT_ODDPARITY_STOP2,		0x78,	" 7bit/OddP/Stop2", },	// SMR=0111 10.. 
	{	COMM_PARAM_7BIT_EVNPARITY_STOP1,		0x60,	" 7bit/EvnP/Stop1", },	// SMR=0110 00.. 
	{	COMM_PARAM_7BIT_EVNPARITY_STOP2,		0x68,	" 7bit/EvnP/Stop2", },	// SMR=0110 10.. 
	{	0xFF,									0,		"0123456789ABCDEF", },	
};

static const MODULE_MENU_TABLE serial_app_table[] = {					// interval time will be ignored.
	{	 0,		0,	ui_function_serial_debug,		"  Debug         ",	},
	{	 1,		0,	ui_function_serial_config,		"  Config        ",	},
	{	 2,		0,	ui_function_serial_trans,		"  Transfer      ",	},
	{	 3,		0,	ui_function_serial_trans_debug,	"  Transfer(Dbg) ",	},
	{	-1,		0,	ui_function_serial_nop,			"0123456789ABCDEF",	},
};

static const DEBUG_VAR_TABLE debug_var_table[] = {
	{	 0,		 1,	"Process number: ",	&serial_proc,				},
	{	 1,		 1,	"Error Code:     ",	&serial_error,				},
	{	 2,		 2,	"Int. Err Debug: ",	&SCIErrorDebug,				},
	{	 3,		 3,	"txsts.sentbytes:",	&tx_status.sent_bytes,		},
	{	 4,		 3,	"txsts.file_size:",	&tx_status.file_size,		},
	{	 5,		 4,	"Receive Buffer: ",	ReceiveBuf,					},
	{	 6,		 4,	"Sent Buffer:    ",	SentBuf,					},
	{	-1,		 1,	"0123456789ABCDEF", 0,					},
};

// Only for this table, "type" member SHOULD BE index number.
static const DEBUG_VAR_TABLE config_var_table[] = {
	{	 0,		 0,	"Speed setting : ",	&sci_setting.speed_setting_index	},
	{	 1,		 1,	"Code setting:   ",	&sci_setting.code_setting_index		},
	{	 2,		 2,	"COM Parameter:  ",	&sci_setting.param_setting_index	},
	{	-1,		 1,	"0123456789ABCDEF", 0,					},
};

// -------------------------------------------
//  Interrupt handlers (1ms)
// -------------------------------------------
void serial_1ms_handler(void) {
	if( serial_disable_timer )		serial_disable_timer--;
}

// -------------------------------------------
//  Interrupt handlers (SCI Error)
// -------------------------------------------
#pragma interrupt (int_ERI0)
void int_ERI0(void) {
	if(SCI0.SSR.BIT.FER) {
		SCI0.SSR.BIT.FER = 0;
		SCIErrorDebug++;
	}
	if(SCI0.SSR.BIT.ORER) {
		SCI0.SSR.BIT.ORER = 0;
		SCIErrorDebug += 0x10;
	}
	if(SCI0.SSR.BIT.PER) {
		SCI0.SSR.BIT.PER = 0;
		SCIErrorDebug+=0x100;
#ifdef RECEIVE_DATA_EVENIF_ERR
		ReceiveBuf[ReceiveCnt] = SCI0.RDR;		// f[^M
		// DC code was refered from http://www5a.biglobe.ne.jp/~NCPRO/pc_nc.htm.
		if( ReceiveBuf[ReceiveCnt] == 0x11
		 || ReceiveBuf[ReceiveCnt] == 0xA5
		 || ReceiveBuf[ReceiveCnt] == 0x25 ) {
		 	sci_setting.flow_control_request = FLOW_REQ_RESUME;
		} else if( ReceiveBuf[ReceiveCnt] == 0x93
		 || ReceiveBuf[ReceiveCnt] == 0x13
		 || ReceiveBuf[ReceiveCnt]== 0x14 ) {
		 	sci_setting.flow_control_request = FLOW_REQ_PAUSE;
		} else {
			sci_setting.flow_control_request = FLOW_REQ_NONE;
		}
		if( ++ReceiveCnt >= READ_BUF_SIZE ) ReceiveCnt = 0;
		if(SCI0.SSR.BIT.PER) SCI0.SSR.BIT.RDRF = 0;
#endif

	}
}

// -------------------------------------------
//  Interrupt handlers (SCI RX data ready)
// -------------------------------------------
#pragma interrupt (int_RXI0)
void int_RXI0(void) {
	if(SCI0.SSR.BIT.RDRF) {		
		ReceiveBuf[ReceiveCnt] = SCI0.RDR;		// f[^M
		
		// DC code was refered from http://www5a.biglobe.ne.jp/~NCPRO/pc_nc.htm.
		if( ReceiveBuf[ReceiveCnt] == 0x11
		 || ReceiveBuf[ReceiveCnt] == 0xA5
		 || ReceiveBuf[ReceiveCnt] == 0x25 ) {
		 	sci_setting.flow_control_request = FLOW_REQ_RESUME;
		} else if( ReceiveBuf[ReceiveCnt] == 0x93
		 || ReceiveBuf[ReceiveCnt] == 0x13
		 || ReceiveBuf[ReceiveCnt]== 0x14 ) {
		 	sci_setting.flow_control_request = FLOW_REQ_PAUSE;
		} else {
			sci_setting.flow_control_request = FLOW_REQ_NONE;
		}
		
		if( ++ReceiveCnt >= READ_BUF_SIZE ) ReceiveCnt = 0;
	}
	SCI0.SSR.BIT.RDRF = 0;
			
}

// -------------------------------------------
//  Interrupt handlers (SCI TX Buffer ready)
// -------------------------------------------
#pragma interrupt (int_TXI0)
void int_TXI0(void) {	
//	if( serial_proc == SCI_TX_IN_PROGRESS ) {
	if( serial_proc == SCI_WAIT_TX_OK ) {
		if(SCI0.SSR.BIT.TDRE) {
/*
			if( sci_setting.flow_control_request == FLOW_REQ_PAUSE ) {
				sci_setting.flow_control_request = FLOW_REQ_NONE;
				serial_proc = SCI_TX_IN_PAUSE;
				tx_status.sent_bytes++;
				SCI0.SCR.BIT.TIE = 0;		// TX empty intr. disable
			} else if( tx_status.sent_bytes < tx_status.file_size ) {
				switch( sci_setting.code_setting_index ) {
					case COMM_CODE_ISO:
						SCI0.TDR = Cnv2Iso( (unsigned char) tx_status.p_buf[tx_status.sent_bytes++]);
						break;
					case COMM_CODE_EIA:
						SCI0.TDR = Cnv2Eia( (unsigned char) tx_status.p_buf[tx_status.sent_bytes++]);
						break;
					default:
						SCI0.TDR = (unsigned char) tx_status.p_buf[tx_status.sent_bytes++];
						break;
				}
				SentBuf[tx_status.sent_bytes&0x0F] = SCI0.TDR;
			} else {
				SCI0.SCR.BIT.TIE = 0;		// TX empty intr. disable
			}
*/
			tx_status.tx_in_progress = 0;
			SCI0.SCR.BIT.TIE = 0;
		}
	} else {
		SCI0.SCR.BIT.TIE = 0;
	}

}

// -------------------------------------------
//  Interrupt handlers (SCI TX End)
// -------------------------------------------
#pragma interrupt (int_TEI0)
void int_TEI0(void) {
	if(SCI0.SSR.BIT.TEND) SCI0.SSR.BIT.TEND = 0;
}

// -------------------------------------------
//  Initialize
// -------------------------------------------
int  serial_initialize(void) {
	serial_proc =  SCI_INITIALIZE;
	serial_error = SCI_ERR_NONE;
	SCIErrorDebug = 0;
	
	// Default setting
	sci_setting.speed_setting_index		= COMM_SPEED_4800BPS;
	sci_setting.code_setting_index		= COMM_CODE_ASCII;
	sci_setting.param_setting_index		= COMM_PARAM_7BIT_EVNPARITY_STOP2;
	sci_setting.flow_control_request	= FLOW_REQ_NONE;
	// UI related
	ui_serial_proc = 0x00;
}

// -------------------------------------------
//  Query for FAT class status
// -------------------------------------------
unsigned char serial_status(REQUEST_TO_CLASS req) {
	switch( req ) {
		case CLASS_REQ_RESET:
			serial_initialize();
			return CLASS_STS_RESET_IN_PROGRESS;
			
		default:
			break;
	}
	
	// Status return
	if( serial_proc < SCI_IDLE )				return CLASS_STS_RESET_IN_PROGRESS;	
	else if( serial_proc == SCI_IDLE ) 			return CLASS_STS_READY;
	else if( serial_proc == SCI_ERROR_STOP )	return CLASS_STS_ERROR_STOP;
	else 										return CLASS_STS_NOT_READY;
}

// -------------------------------------------
//  Transfer Start Request
// -------------------------------------------
unsigned char TransferStartRequest( unsigned char* pBuf, unsigned long size ) {
	if( serial_proc != SCI_IDLE && serial_proc != SCI_TX_IN_PAUSE )	return CLASS_REQ_BUSY;
	
	if( size != 0 ) {					// Start request
		tx_status.file_size  = size;
		tx_status.sent_bytes = 0;
		tx_status.p_buf      = pBuf;
		tx_status.tx_in_progress = 0;
		serial_proc = SCI_DATA_SET;
		sci_setting.flow_control_request = FLOW_REQ_PAUSE;
#ifdef SCI_DEBUG_ON
		printf("[SCI] TX Req Set\r\n");
#endif
	}
/*
	 else {							// Continue request
		serial_proc = SCI_TX_IN_PROGRESS;
		// Transfer start
		switch( sci_setting.code_setting_index ) {
			case COMM_CODE_ISO:
				SCI0.TDR = Cnv2Iso( (unsigned char) tx_status.p_buf[tx_status.sent_bytes++]);
				break;
			case COMM_CODE_EIA:
				SCI0.TDR = Cnv2Eia( (unsigned char) tx_status.p_buf[tx_status.sent_bytes++]);
				break;
			default:
				SCI0.TDR = (unsigned char) tx_status.p_buf[tx_status.sent_bytes++];
				break;
		}
		SentBuf[tx_status.sent_bytes&0x0F] = SCI0.TDR;
		
		if(SCI0.SSR.BIT.TDRE) SCI0.SSR.BIT.TDRE = 0;		// TX empty intr. clear
		SCI0.SCR.BIT.TIE = 1;								// TX empty intr. enable
	}
*/
}

// -------------------------------------------
//  Main process
// -------------------------------------------
void serial_process( void ) {
	int i;
	if( serial_disable_timer ) return;
	
	switch( serial_proc ) {
		case SCI_INITIALIZE:
		 	SCI0.SCR.BYTE	= 0x00;			// Clear firstly. RX/TX stop
		 	
			SCI0.SMR.BYTE	= param_setting[sci_setting.param_setting_index].smr_setting 
			               |  speed_setting[sci_setting.speed_setting_index].cks_setting;
			SCI0.BRR			= speed_setting[sci_setting.speed_setting_index].brr_setting;
			
			for(i=0; i<READ_BUF_SIZE; i++) ReceiveBuf[i] = 0x00;
			for(i=0; i<SENT_BUF_SIZE; i++) SentBuf[i] = 0x00;
			ReceiveCnt = 0;
			ReadCnt = 0;
			serial_proc = SCI_READY_WAIT;
			serial_disable_timer = 10;
			break;
			
		case SCI_READY_WAIT:
			SCI0.SCR.BIT.TE  = 1;					//TX OK
			SCI0.SCR.BIT.RE  = 1;					//RX OK
			SCI0.SCR.BIT.TIE = 1;					//TX Interrupt enable
			SCI0.SCR.BIT.RIE = 1;					//RX Interrupt enable
			serial_proc = SCI_IDLE;
			break;
			
		case SCI_IDLE:
			break;
			
		case SCI_DATA_SET:
			if( sci_setting.flow_control_request != FLOW_REQ_PAUSE ) {
				// 1 byte Transfer start
				switch( sci_setting.code_setting_index ) {
					case COMM_CODE_ISO:
						SCI0.TDR = Cnv2Iso( (unsigned char) tx_status.p_buf[tx_status.sent_bytes++]);
						break;
					case COMM_CODE_EIA:
						SCI0.TDR = Cnv2Eia( (unsigned char) tx_status.p_buf[tx_status.sent_bytes++]);
						break;
					default:
						SCI0.TDR = (unsigned char) tx_status.p_buf[tx_status.sent_bytes++];
						break;
				}
				serial_proc = SCI_WAIT_TX_OK;
				serial_disable_timer = 2;
				SentBuf[tx_status.sent_bytes&0x0F] = SCI0.TDR;
				tx_status.tx_in_progress = 1;
				
				if(SCI0.SSR.BIT.TDRE) SCI0.SSR.BIT.TDRE = 0;		// TX empty intr. clear
				SCI0.SCR.BIT.TIE = 1;								// TX empty intr. enable
			}
			break;
			
		case SCI_WAIT_TX_OK:
			if( tx_status.tx_in_progress ) break;
			if( tx_status.sent_bytes >= tx_status.file_size ) {
				SCI0.SCR.BIT.TIE = 0;					//TX empty Interrupt disable
				serial_proc = SCI_TX_IN_END;
				serial_disable_timer = 1000;
			} else {
				serial_proc = SCI_DATA_SET;
			}
			break;
			
		case SCI_TX_IN_PROGRESS:
			if( tx_status.sent_bytes >= tx_status.file_size ) {
				SCI0.SCR.BIT.TIE = 0;					//TX empty Interrupt disable
				serial_proc = SCI_TX_IN_END;
				serial_disable_timer = 1000;
			}
			break;
			
		case SCI_TX_IN_PAUSE:
			if( sci_setting.flow_control_request == FLOW_REQ_RESUME ) {
				sci_setting.flow_control_request = FLOW_REQ_NONE;
				TransferStartRequest(0, 0);
			}
			break;

		case SCI_TX_IN_END:
#ifdef SCI_DEBUG_ON
			printf("[SCI] TX Complete\r\n");
#endif
			serial_proc = SCI_IDLE;
			break;

		case SCI_ERROR_STOP:
			break;
	}
}

// -------------------------------------------
//  Convert to ISO code
// -------------------------------------------
//                          0 1 2 3 4 5 6 7 8 9 A B C D E F
unsigned char bitnum[16] = {0,1,1,2,1,2,2,3,1,2,2,3,2,3,3,4};
static unsigned char Cnv2Iso( unsigned char ascii_char ) {
	unsigned char num;
	
//	num = bitnum[(ascii_char>>4)&0x0F] + bitnum[ascii_char&0x0F];

	num = 0;
	if( ascii_char & 0x80 ) num++;
	if( ascii_char & 0x40 ) num++;
	if( ascii_char & 0x20 ) num++;
	if( ascii_char & 0x10 ) num++;
	if( ascii_char & 0x08 ) num++;
	if( ascii_char & 0x04 ) num++;
	if( ascii_char & 0x02 ) num++;
	if( ascii_char & 0x01 ) num++;

	if( num&0x01 ) 
		return (0x80|ascii_char);
	else
		return ascii_char;
}

// -------------------------------------------
//  Convert to EIA code
// -------------------------------------------
unsigned char table_EIA[256] = 
	{	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x2A,0x3E,0x80,0x0B,0x0C,0x0D,0x0E,0x0F,	// 0x00 
		0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,	// 0x10 
		0x10,0x21,0x22,0x23,0x94,0x0B,0x0E,0x27,0x1A,0x4A,0x98,0x70,0x3B,0x40,0x6B,0x31,	// 0x20 
		0x20,0x01,0x02,0x13,0x04,0x15,0x16,0x07,0x08,0x19,0x3A,0x3B,0x3C,0x8C,0x3E,0x9E,	// 0x30 0-9(39)
		0x8F,0x61,0x62,0x73,0x64,0x75,0x76,0x67,0x68,0x79,0x51,0x52,0x43,0x54,0x45,0x46,	// 0x40 A(41)-O
		0x57,0x58,0x49,0x32,0x23,0x34,0x25,0x26,0x37,0x38,0x29,0xB0,0x5C,0xA2,0x5E,0x5F,	// 0x50 P-Z(5A)
		0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6A,0x6B,0x6C,0x6D,0x6E,0x6F,	// 0x60 
		0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,0x7B,0x7C,0x7D,0x7E,0x7F,	// 0x70 
		0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8A,0x8B,0x8C,0x8D,0x8E,0x8F,	// 0x80 
		0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99,0x9A,0x9B,0x9C,0x9D,0x9E,0x9F,	// 0x90 
		0xA0,0xA1,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,0xA9,0xAA,0xAB,0xAC,0xAD,0xAE,0xAF,	// 0xA0 
		0xB0,0xB1,0xB2,0xB3,0xB4,0xB5,0xB6,0xB7,0xB8,0xB9,0xBA,0xBB,0xBC,0xBD,0xBE,0xBF,	// 0xB0 
		0xC0,0xC1,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,0xCB,0xCC,0xCD,0xCE,0xCF,	// 0xC0 
		0xD0,0xD1,0xD2,0xD3,0xD4,0xD5,0xD6,0xD7,0xD8,0xD9,0xDA,0xDB,0xDC,0xDD,0xDE,0xDF,	// 0xD0 
		0xE0,0xE1,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,0xE8,0xE9,0xEA,0xEB,0xEC,0xED,0xEE,0xEF,	// 0xE0 
		0xF0,0xF1,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,0xF8,0xF9,0xFA,0xFB,0xFC,0xFD,0xFE,0xFF };	// 0xF0 
static unsigned char Cnv2Eia( unsigned char ascii_char ) {
	return table_EIA[ascii_char];
}


// -------------------------------------------
//  UI Function - Serial function
// -------------------------------------------
int ui_function_serial(UI_COMMAND uicmd) {
	static unsigned int current_index;
	static unsigned int cursol_position;
	int ret_val;
	UI_COMMAND ui_cmd_sub;

	ret_val = UI_RET_READY;
	
	switch( ui_serial_proc ) {
		case 0x00:
			// Event check
			switch( uicmd.cmd ) {
				case UI_CMD_NOP:
				case UI_CMD_INTEVAL:
					break;
					
				case UI_CMD_STARTED:
					current_index = 0;
					cursol_position = 0;
					break;
					
				case UI_CMD_KEY_PRESS_BACK:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					ret_val = UI_RET_QUIT;
					break;
					
				case UI_CMD_KEY_PRESS_UP:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					if( cursol_position == 1 ) {
						cursol_position = 0;
						current_index--;
					} else {
						if( current_index != 0 ) current_index--;
					}
					break;
				case UI_CMD_KEY_PRESS_DOWN:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					if( cursol_position == 0 ) {
						cursol_position = 1;
						current_index++;
					} else {
						if( serial_app_table[current_index+1].num != -1 ) current_index++;
					}
					break;
				case UI_CMD_KEY_PRESS_OK:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					ui_cmd_sub.cmd = UI_CMD_STARTED;
					serial_app_table[current_index].pExecFunc(ui_cmd_sub, serial_app_table[current_index].param);	// Initialize
					ui_serial_proc = 0x01;
					break;
			}
			
			// Menu view
			if( cursol_position == 0) { // cursol in first line
				sc1602_set_buffer_cursol( 0, serial_app_table[current_index  ].display );
				sc1602_set_buffer       ( 1, serial_app_table[current_index+1].display );
			} else {
				sc1602_set_buffer       ( 0, serial_app_table[current_index-1].display );
				sc1602_set_buffer_cursol( 1, serial_app_table[current_index  ].display );
			}
			break;
		
		case 0x01:
			switch( serial_app_table[current_index].pExecFunc(uicmd, serial_app_table[current_index].param) ) {
				case UI_RET_QUIT:
					ui_serial_proc = 0x00;
					break;
				default:
					break;
			}
			break;
		default:
			break;
	}
	return ret_val;
}


// -------------------------------------------
//  UI Function - SERIAL debug
// -------------------------------------------
static int ui_function_serial_debug(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	static unsigned char ok_press;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_KEY_PRESS_OK:
		if( uicmd.param == OFF_EDGE )	ok_press = 0;
		else							ok_press = 1;
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, debug_var_table[current_index].display );
		switch(debug_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_variable3 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 4:
				sc1602_set_buffer_dump (1, debug_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		ok_press = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( ok_press == 0 ) {
			if( debug_var_table[current_index+1].num !=  -1 ) current_index++;
		} else {
			// TODO
		}
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}

// -------------------------------------------
//  UI Function - SERIAL config
// -------------------------------------------
static int ui_function_serial_config(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_KEY_PRESS_OK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		switch( current_index ) {
			case 0:
				sci_setting.speed_setting_index++;
				if( speed_setting[sci_setting.speed_setting_index].index == 0xFF )
					sci_setting.speed_setting_index = 0x01;
				break;
			case 1:
				sci_setting.code_setting_index++;
				if( code_setting[sci_setting.code_setting_index].index == 0xFF )
					sci_setting.code_setting_index = 0x01;
				break;
			case 2:
				sci_setting.param_setting_index++;
				if( param_setting[sci_setting.param_setting_index].index == 0xFF )
					sci_setting.param_setting_index = 0x01;
				break;
			default:
				break;
		}
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, config_var_table[current_index].display );
		switch( current_index ) {
			case 0:
				sc1602_set_buffer ( 1, speed_setting[sci_setting.speed_setting_index].Name );
				break;
			case 1:
				sc1602_set_buffer ( 1, code_setting[sci_setting.code_setting_index].Name );
				break;
			case 2:
				sc1602_set_buffer ( 1, param_setting[sci_setting.param_setting_index].Name );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( config_var_table[current_index+1].num !=  -1 ) current_index++;
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}

// -------------------------------------------
//  UI Function - SERIAL trans
// -------------------------------------------
static int ui_function_serial_trans(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	static unsigned char ok_press;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_KEY_PRESS_OK:
		if( uicmd.param == OFF_EDGE )	ok_press = 0;
		else							ok_press = 1;
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		switch( serial_proc ) {
			case SCI_INITIALIZE:
			case SCI_ERROR_STOP:
			case SCI_READY_WAIT:
				sc1602_set_buffer( 0, "SCI is not ready" );
				break;
				
			case SCI_IDLE:
				sc1602_set_buffer( 0, "SCI is ready.   " );
				break;
				
			case SCI_DATA_SET:
			case SCI_WAIT_TX_OK:
				if( sci_setting.flow_control_request == FLOW_REQ_PAUSE ) 	sc1602_set_buffer( 0, "Pause....       " );
				else 														sc1602_set_buffer( 0, "Sending...      " );
				break;
				
			case SCI_TX_IN_PROGRESS:
				sc1602_set_buffer( 0, "Sending...      " );
				break;
				
			case SCI_TX_IN_PAUSE:
				sc1602_set_buffer( 0, "Pause....       " );
				break;

			case SCI_TX_IN_END:
				sc1602_set_buffer( 0, "Complete!       " );
				break;
		}
		sc1602_set_buffer_progress_kb(1, tx_status.sent_bytes, tx_status.file_size);
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		ok_press = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( ok_press == 0 ) {
			if( debug_var_table[current_index+1].num !=  -1 ) current_index++;
		} else {
			// TODO
		}
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}

static const char hex2ascii[] = "0123456789ABCDEF";
// -------------------------------------------
//  UI Function - SERIAL trans (debug)
// -------------------------------------------
static int ui_function_serial_trans_debug(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	static unsigned char ok_press;
	unsigned char *data;
	int ret_val;
	unsigned int uint_val;
	char tempbuf[2][16];
	// 0123456789ABCDEF
	// ----------------
	// R[?]:........|**
	// S..C..P..   ...%
	tempbuf[0][0x0]='R';	tempbuf[0][0x1]='[';	tempbuf[0][0x2]=' ';	tempbuf[0][0x3]=']';
	tempbuf[0][0x4]=' ';	tempbuf[0][0x5]=' ';	tempbuf[0][0x6]=' ';	tempbuf[0][0x7]=' ';
	tempbuf[0][0x8]=' ';	tempbuf[0][0x9]=' ';	tempbuf[0][0xA]=' ';	tempbuf[0][0xB]=' ';
	tempbuf[0][0xC]=' ';	tempbuf[0][0xD]='|';	tempbuf[0][0xE]=' ';	tempbuf[0][0xF]=' ';

	tempbuf[1][0x0]='S';	tempbuf[1][0x1]=' ';	tempbuf[1][0x2]=' ';	tempbuf[1][0x3]='C';
	tempbuf[1][0x4]=' ';	tempbuf[1][0x5]=' ';	tempbuf[1][0x6]='P';	tempbuf[1][0x7]=' ';
	tempbuf[1][0x8]=' ';	tempbuf[1][0x9]=' ';	tempbuf[1][0xA]=' ';	tempbuf[1][0xB]=' ';
	tempbuf[1][0xC]=' ';	tempbuf[1][0xD]=' ';	tempbuf[1][0xE]=' ';	tempbuf[1][0xF]='%';


	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
		break;
		
	case UI_CMD_KEY_PRESS_OK:
		if( uicmd.param == OFF_EDGE )	ok_press = 0;
		else							ok_press = 1;
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		switch( serial_proc ) {
			case SCI_INITIALIZE:
			case SCI_ERROR_STOP:
			case SCI_READY_WAIT:
				//sc1602_set_buffer( 0, "SCI is not ready" );
				tempbuf[0][0xE]='N'; tempbuf[0][0xF]='R';		// NR / Not ready
				break;
				
			case SCI_IDLE:
				//sc1602_set_buffer( 0, "SCI is ready.   " );
				tempbuf[0][0xE]='R'; tempbuf[0][0xF]='D';		// RD / Ready
				break;
				
			case SCI_DATA_SET:
			case SCI_WAIT_TX_OK:
				if( sci_setting.flow_control_request == FLOW_REQ_PAUSE ) 
					/*sc1602_set_buffer( 0, "Pause....       " );*/ {tempbuf[0][0xE]='P'; tempbuf[0][0xF]='S';}		// PS / Pause
				else 													
					/*sc1602_set_buffer( 0, "Sending...      " );*/ {tempbuf[0][0xE]='S'; tempbuf[0][0xF]='D';}		// SD / Sending
				break;
				
			case SCI_TX_IN_PROGRESS:
				//sc1602_set_buffer( 0, "Sending...      " );
				tempbuf[0][0xE]='S'; tempbuf[0][0xF]='D';		// SD / Sending
				break;
				
			case SCI_TX_IN_PAUSE:
				//sc1602_set_buffer( 0, "Pause....       " );
				tempbuf[0][0xE]='P'; tempbuf[0][0xF]='S';		// PS / Pause
				break;

			case SCI_TX_IN_END:
				//sc1602_set_buffer( 0, "Complete!       " );
				tempbuf[0][0xE]='P'; tempbuf[0][0xF]='S';		// DN / Complete
				break;
		}
		
		// Read buf dump
		data = &ReceiveBuf[(ReceiveCnt&0x03)];
		tempbuf[0][0x2] =  hex2ascii[ReceiveCnt & 0x0F ];
		tempbuf[0][0x5] =  hex2ascii[(*(data + 0) >> 4) & 0x0F ];
		tempbuf[0][0x6] =  hex2ascii[(*(data + 0)     ) & 0x0F ];
		tempbuf[0][0x7] =  hex2ascii[(*(data + 1) >> 4) & 0x0F ];
		tempbuf[0][0x8] =  hex2ascii[(*(data + 1)     ) & 0x0F ];
		tempbuf[0][0x9] =  hex2ascii[(*(data + 2) >> 4) & 0x0F ];
		tempbuf[0][0xA] =  hex2ascii[(*(data + 2)     ) & 0x0F ];
		tempbuf[0][0xB] =  hex2ascii[(*(data + 3) >> 4) & 0x0F ];
		tempbuf[0][0xC] =  hex2ascii[(*(data + 3)     ) & 0x0F ];
		
		// Registers
		tempbuf[1][0x1] =  hex2ascii[(SCI0.SSR.BYTE >> 4) & 0x0F ];
		tempbuf[1][0x2] =  hex2ascii[(SCI0.SSR.BYTE     ) & 0x0F ];
		tempbuf[1][0x4] =  hex2ascii[(SCI0.SCR.BYTE >> 4) & 0x0F ];
		tempbuf[1][0x5] =  hex2ascii[(SCI0.SCR.BYTE     ) & 0x0F ];
		tempbuf[1][0x7] =  hex2ascii[(serial_proc >> 4) & 0x0F ];
		tempbuf[1][0x8] =  hex2ascii[(serial_proc     ) & 0x0F ];
		
		// Percentage
		uint_val = (unsigned int) (tx_status.sent_bytes/tx_status.file_size)*100;
		tempbuf[1][0xC] = hex2ascii[(uint_val/100)  % 10 ];
		tempbuf[1][0xD] = hex2ascii[(uint_val/10)   % 10 ];
		tempbuf[1][0xE] = hex2ascii[(uint_val)      % 10 ];
		
		sc1602_set_buffer(0,tempbuf[0]);
		sc1602_set_buffer(1,tempbuf[1]);
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		ok_press = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
		
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( ok_press == 0 ) {
			if( debug_var_table[current_index+1].num !=  -1 ) current_index++;
		} else {
			// TODO
		}
		break;
		
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}

// -------------------------------------------
//  UI Sub Function - SERIAL nop
// -------------------------------------------
static int ui_function_serial_nop(UI_COMMAND uicmd, int param) {
	return UI_RET_QUIT;
}


// -------------------------------------------
//  UI service Function - SERIAL progress
// -------------------------------------------
void ui_serial_progress( void ) {
	// Title
	switch( serial_proc ) {
		case SCI_INITIALIZE:
		case SCI_ERROR_STOP:
		case SCI_READY_WAIT:
			sc1602_set_buffer( 0, "SCI is not ready" );
			break;
			
		case SCI_IDLE:
			sc1602_set_buffer( 0, "SCI is ready.   " );
			break;
			
		case SCI_DATA_SET:
		case SCI_WAIT_TX_OK:
			if( sci_setting.flow_control_request == FLOW_REQ_PAUSE ) 	sc1602_set_buffer( 0, "Pause....       " );
			else 														sc1602_set_buffer( 0, "Sending...      " );
			break;
			
		case SCI_TX_IN_PROGRESS:
			sc1602_set_buffer( 0, "Sending...      " );
			break;
			
		case SCI_TX_IN_PAUSE:
			sc1602_set_buffer( 0, "Pause....       " );
			break;

		case SCI_TX_IN_END:
			sc1602_set_buffer( 0, "Complete!       " );
			break;
	}
	sc1602_set_buffer_progress_kb(1, tx_status.sent_bytes, tx_status.file_size);
}
