require 'getoptlong'
require 'logger'
require 'pp'

module OSDN
  module CLI
    class Runner
      def initialize
        @logger = Logger.new(STDERR)
        @logger.level = Logger::INFO
        @logger.formatter = proc { |severity, time, progname, msg|
          "[%s] %s\n" % [severity, msg]
        }
        OSDNClient.configure do |config|
          ENV['OSDN_API_OVERRIDE_HOST'] and
            config.host = ENV['OSDN_API_OVERRIDE_HOST']
          #ENV['OSDN_API_SKIP_SSL_VERIFY'].to_s =~ /^(1|t(rue)?|y(es)?)$/i and
          #  config.verify_ssl = false
          config.verify_ssl = false
        end
      end
      attr_reader :logger
      
      def parse_opt
        opts = GetoptLong.new(
          [ '--help', '-h', GetoptLong::NO_ARGUMENT ],
          [ '--verbose', '-v', GetoptLong::NO_ARGUMENT ],
          [ '--quiet', '-q', GetoptLong::NO_ARGUMENT ],
        )
        opts.ordering = GetoptLong::REQUIRE_ORDER
        opts.each do |opt, arg|
          case opt
          when '--help'
            help
            exit 0
          when '--verbose'
            if logger.level == Logger::DEBUG
              OSDNClient.configure do |config|
                config.debugging = true
              end
            end
            logger.level > Logger::DEBUG and
              logger.level -= 1
          when '--quiet'
            logger.level < Logger::UNKNOWN and
              logger.level += 1
          when '--help'
            help
            exit
          end
        end
        logger.debug "Loglevel is #{logger.level}"
      end

      def get_command_class(command_name)
        class_name = command_name.to_s.split('_').map(&:capitalize).join
        begin
          return self.class.const_get("OSDN::CLI::Command::#{class_name}")
        rescue NameError => e
          logger.fatal "Invalid command name '#{command_name}'. Use 'help' to list commands."
          exit
        end
        false
      end

      def run
        parse_opt

        command_name = ARGV.shift
        unless command_name
          help
          exit 1
        end

        if command_name == 'help'
          help
          exit
        end
        
        command = get_command_class(command_name).new(logger)
        logger.debug "Run command #{command_name}"
        begin
          command.run
        rescue OSDNClient::ApiError => e
          begin
            err = JSON.parse(e.response_body)
            if err["message"]
              logger.fatal "#{err["status"]}: #{err["message"]}"
            elsif err["error_description"]
              logger.fatal err["error_description"]
            else
              logger.fatal "Command failed by ApiError: #{e.response_body}"
            end
          rescue
            logger.fatal "Command failed: #{e.inspect}"
          end
        end
      end

      def help
        command_name = ARGV.shift
        if command_name
          get_command_class(command_name).new(logger).help
        else
          puts "#{$0} [global-options] <command> [command-options] [args]"
          puts "#{$0} help <command>"
          puts "Global Options:"
          puts "  -h --help      Show help message. use 'help <command>' for specific command. "
          puts "  -v --verbose   Increase log level (multiple)"
          puts "  -q --quiet     Decrease log level (multiple)"
          puts "Avaiable Commands:"
          puts "  help"
          OSDN::CLI::Command.constants.each do |c|
            c = c.to_s.split(/(?=[A-Z])/).join('_').downcase
            c == 'base' and next
            puts "  %-14s %s" % [c, get_command_class(c).description]
          end
        end
      end
    end
  end
end
