/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.chart.ui;

import java.util.Calendar;
import java.util.Date;

import jp.sf.orangesignal.chart.util.StringManager;

/**
 * 期間を表す列挙型を提供します。
 * 
 * @author 杉澤 浩二
 */
public enum PeriodType {

	/**
	 * 3ヶ月を表します。
	 */
	THREE_MONTH {
		@Override public int getPeriod(final DatasetType unit) {
			switch (unit) {
				case DAILY:		return 65;
				case WEEKLY:	return 13;
				case MONTHLY:	return 3;
				default:		throw new IllegalArgumentException();
			}
		}
		@Override public String toString() { return StringManager.getString("3month"); }
	},

	/**
	 * 6ヶ月を表します。
	 */
	SIX_MONTH {
		@Override public int getPeriod(final DatasetType unit) {
			switch (unit) {
				case DAILY:		return 127;
				case WEEKLY:	return 26;
				case MONTHLY:	return 6;
				default:		throw new IllegalArgumentException();
			}
		}
		@Override public String toString() { return StringManager.getString("6month"); }
	},

	/**
	 * 1年を表します。
	 */
	ONE_YEAR {
		@Override public int getPeriod(final DatasetType unit) {
			switch (unit) {
				case DAILY:		return 246;
				case WEEKLY:	return 52;
				case MONTHLY:	return 12;
				default:		throw new IllegalArgumentException();
			}
		}
		@Override public String toString() { return StringManager.getString("1year"); }
	},

	/**
	 * 2年を表します。
	 */
	TWO_YEAR {
		@Override public int getPeriod(final DatasetType unit) {
			switch (unit) {
				case DAILY:		return 246 * 2;
				case WEEKLY:	return 52 * 2;
				case MONTHLY:	return 12 * 2;
				default:		throw new IllegalArgumentException();
			}
		}
		@Override public String toString() { return StringManager.getString("2year"); }
	},

	/**
	 * 3年を表します。
	 */
	THREE_YEAR {
		@Override public int getPeriod(final DatasetType unit) {
			switch (unit) {
				case DAILY:		return 246 * 3;
				case WEEKLY:	return 52 * 3;
				case MONTHLY:	return 12 * 3;
				default:		throw new IllegalArgumentException();
			}
		}
		@Override public String toString() { return StringManager.getString("3year"); }
	},

	/**
	 * 4年を表します。
	 */
	FOUR_YEAR {
		@Override public int getPeriod(final DatasetType unit) {
			switch (unit) {
				case DAILY:		return 246 * 4;
				case WEEKLY:	return 52 * 4;
				case MONTHLY:	return 12 * 4;
				default:		throw new IllegalArgumentException();
			}
		}
		@Override public String toString() { return StringManager.getString("4year"); }
	},

	/**
	 * 5年を表します。
	 */
	FIVE_YEAR {
		@Override public int getPeriod(final DatasetType unit) {
			switch (unit) {
				case DAILY:		return 246 * 5;
				case WEEKLY:	return 52 * 5;
				case MONTHLY:	return 12 * 5;
				default:		throw new IllegalArgumentException();
			}
		}
		@Override public String toString() { return StringManager.getString("5year"); }
	},

	/**
	 * 10年を表します。
	 */
	TEN_YEAR {
		@Override public int getPeriod(final DatasetType unit) {
			switch (unit) {
				case DAILY:		return 246 * 10;
				case WEEKLY:	return 52 * 10;
				case MONTHLY:	return 12 * 10;
				default:		throw new IllegalArgumentException();
			}
		}
		@Override public String toString() { return StringManager.getString("10year"); }
	},

	/**
	 * すべてを表します。
	 */
	ALL {
		@Override public int getPeriod(final DatasetType unit) {
			switch (unit) {
				case DAILY:
				case WEEKLY:
				case MONTHLY:
					return -1;
				default:
					throw new IllegalArgumentException();
			}
		}
		@Override public String toString() { return StringManager.getString("all"); }
	};

	/**
	 * 指定された足単位に対応した期間を返します。
	 * 
	 * @param unit 足単位
	 * @return 足単位に対応した期間
	 */
	public abstract int getPeriod(DatasetType unit);

	/**
	 * 指定された日付から、この期間分前の日付を返します。<p>
	 * この期間がすべてを表す場合、常に <code>null</code> を返します。
	 * 
	 * @param date 日付
	 * @return この期間分前の日付。または <code>null</code>
	 */
	public Date getPreviousDate(final Date date) {
		if (this == ALL || date == null) {
			return null;
		}

		final Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.add(Calendar.MONTH, -getPeriod(DatasetType.MONTHLY));
		return c.getTime();
	}

}
