/*
 * Copyright (c) 2009-2010 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;
import java.text.Format;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;
import jp.sf.orangesignal.csv.filters.CsvNamedValueFilter;
import jp.sf.orangesignal.csv.manager.CsvColumnNameMappingBeanManager;

/**
 * 区切り文字形式データの項目名を基準として Java プログラム要素のリストと区切り文字形式データアクセスを行うハンドラを提供します。
 * 
 * @author 杉澤 浩二
 * @see CsvColumnNameMappingBeanManager
 */
public class ColumnNameMappingBeanListHandler<T> extends BeanListHandlerSupport<T, ColumnNameMappingBeanListHandler<T>> {

	/**
	 * 項目名と Java プログラム要素のフィールド名のマップを保持します。
	 */
	private LinkedHashMap<String, String> columnMapping = new LinkedHashMap<String, String>();

	/**
	 * 区切り文字形式データフィルタを保持します。
	 */
	private CsvNamedValueFilter valueFilter;

	// ------------------------------------------------------------------------

	/**
	 * コンストラクタです。
	 * 
	 * @param type Java プログラム要素の型
	 * @throws IllegalArgumentException <code>type</code> が <code>null</code> の場合
	 */
	public ColumnNameMappingBeanListHandler(final Class<T> type) {
		super(type);
	}

	// ------------------------------------------------------------------------

	/**
	 * 指定された項目名と Java プログラム要素のフィールド名をマップへ追加します。
	 * 
	 * @param column 項目名
	 * @param field Java プログラム要素のフィールド名
	 * @return このオブジェクトへの参照
	 */
	public ColumnNameMappingBeanListHandler<T> addColumn(final String column, final String field) {
		return addColumn(column, field, null);
	}

	/**
	 * 指定された項目名と Java プログラム要素のフィールド名をマップへ追加します。
	 * 
	 * @param column 項目名
	 * @param field Java プログラム要素のフィールド名
	 * @param format フィールドを処理するフォーマットオブジェクト (オプション)
	 * @return このオブジェクトへの参照
	 * @since 1.2
	 */
	public ColumnNameMappingBeanListHandler<T> addColumn(final String column, final String field, final Format format) {
		columnMapping.put(column, field);
		if (format != null) {
			setValueParser(field, format);
			setValueFormatter(column, format);
		}
		return this;
	}

	/**
	 * 項目名と Java プログラム要素のフィールド名のマップを設定します。
	 * 
	 * @param columnMapping 項目名と Java プログラム要素のフィールド名のマップ
	 * @throws IllegalArgumentException <code>columnMapping</code> が <code>null</code> の場合
	 * @since 1.2.4
	 */
	public void setColumnMapping(final Map<String, String> columnMapping) {
		if (columnMapping == null) {
			throw new IllegalArgumentException("Column mapping must not be null");
		}
		this.columnMapping = new LinkedHashMap<String, String>(columnMapping);
	}

	/**
	 * 項目名と Java プログラム要素のフィールド名のマップを設定します。
	 * 
	 * @param columnMapping 項目名と Java プログラム要素のフィールド名のマップ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>columnMapping</code> が <code>null</code> の場合
	 */
	public ColumnNameMappingBeanListHandler<T> columnMapping(final Map<String, String> columnMapping) {
		setColumnMapping(columnMapping);
		return this;
	}

	/**
	 * 区切り文字形式データフィルタを設定します。
	 * 
	 * @param filter 区切り文字形式データフィルタ
	 * @return このオブジェクトへの参照
	 * @since 1.2.3
	 */
	public ColumnNameMappingBeanListHandler<T> filter(final CsvNamedValueFilter filter) {
		this.valueFilter = filter;
		return this;
	}

	// ------------------------------------------------------------------------

	@Override
	public List<T> load(final CsvReader reader) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		prepare();

		// ヘッダ部を処理します。
		final List<String> columnNames = reader.readValues();
		if (columnNames == null) {
			// ヘッダがない場合は例外をスローします。
			throw new IOException("No header is available");
		}

		// データ部を処理します。
		final List<T> results = new ArrayList<T>();
		final Field[] fields = getType().getDeclaredFields();
		final Map<String, Object[]> fieldColumnsMap = createFieldAndColumnsMap(columnMapping);
		int offset = 0;
		List<String> values;
		while ((values = reader.readValues()) != null && (limit <= 0 || results.size() < limit)) {
			if (valueFilter != null && !valueFilter.accept(columnNames, values)) {
				continue;
			}
			if (beanFilter == null && offset < this.offset) {
				offset++;
				continue;
			}
			final T bean = createBean();
			for (final Field f : fields) {
				final Object[] columns = fieldColumnsMap.get(f.getName());
				final int count = columns == null ? 0 : columns.length;

				Object o = null;
				if (count == 1) {
					final int pos = columnNames.indexOf(columns[0]);
					if (pos != -1) {
						o = stringToObject(f, values.get(pos));
					}
				} else if (count > 1) {
					final StringBuilder sb = new StringBuilder();
					for (final Object column : columns) {
						final int pos = columnNames.indexOf(column);
						if (pos != -1) {
							final String s = values.get(pos);
							if (s != null) {
								sb.append(s);
							}
						}
					}
					o = stringToObject(f, sb.toString());
				}
				if (o != null) {
					setFieldValue(bean, f, o);
				}
			}
			if (beanFilter != null) {
				if (!beanFilter.accept(bean)) {
					continue;
				}
				if (offset < this.offset) {
					offset++;
					continue;
				}
			}
			results.add(bean);
		}

		// 並び替え条件が指定されている場合は並び替えを行います。
		if (orders != null && !orders.isEmpty()) {
			BeanOrderComparator.sort(results, orders);
		}

		return results;
	}

	@Override
	public void save(final List<T> list, final CsvWriter writer) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		prepare();

		// ヘッダ部を処理します。
		final List<String> columnNames = new ArrayList<String>();
		for (final Map.Entry<String, String> entry : columnMapping.entrySet()) {
			columnNames.add(entry.getKey());
		}
		writer.writeValues(columnNames);

		// データ部を処理します。
		final int columnCount = columnNames.size();
		for (final T bean : list) {
			// 要素が null の場合は null 出力します。
			if (bean == null) {
				writer.writeValues(null);
				continue;
			} else if (beanFilter != null && !beanFilter.accept(bean)) {
				continue;
			}

			final String[] values = new String[columnCount];
			for (int i = 0; i < columnCount; i++) {
				final String columnName = columnNames.get(i);
				if (columnName == null) {
					continue;
				}
				final String fieldName = columnMapping.get(columnName);
				if (fieldName == null) {
					continue;
				}
				final Field f = getField(bean.getClass(), fieldName);
				values[i] = objectToString(columnName, getFieldValue(bean, f));
			}
			final List<String> _values = Arrays.asList(values);
			if (valueFilter != null && !valueFilter.accept(columnNames, _values)) {
				continue;
			}
			writer.writeValues(_values);
		}
	}

	/**
	 * 項目名とフィールド名のマップが指定されていない場合、フィールド名からマップを作成して準備します。
	 */
	private void prepare() {
		if (columnMapping.size() == 0) {
			for (final Field f : getType().getDeclaredFields()) {
				final String name = f.getName();
				addColumn(name, name);
			}
		}
	}

}
