/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 *
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 *
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.manager;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

import jp.sf.orangesignal.csv.CsvConfig;
import jp.sf.orangesignal.csv.handlers.SampleBean;

import org.junit.Before;
import org.junit.Test;

/**
 * {@link CsvColumnPositionMappingBeanManager} クラスの単体テストです。
 *
 * @author 杉澤 浩二
 */
public class CsvColumnPositionMappingBeanManagerTest {

	private CsvConfig cfg;

	@Before
	public void setUp() throws Exception {
		cfg = new CsvConfig(',');
		cfg.setEscapeDisabled(false);
		cfg.setNullString("NULL");
		cfg.setIgnoreTrailingWhitespaces(true);
		cfg.setIgnoreLeadingWhitespaces(true);
		cfg.setIgnoreEmptyLines(true);
	}

	@Test
	public void testCsvColumnPositionMappingBeanManager() {
		new CsvColumnPositionMappingBeanManager();
	}

	@Test
	public void testCsvColumnPositionMappingBeanManagerCsvConfig() {
		new CsvColumnPositionMappingBeanManager(new CsvConfig());
	}

	@Test
	public void testConfig() {
		new CsvColumnPositionMappingBeanManager().config(new CsvConfig());
	}

	@Test
	public void testLoad() throws IOException {
		cfg.setSkipLines(0);

		final List<SampleBean> list = new CsvColumnPositionMappingBeanManager(cfg)
				.load(SampleBean.class)
				.column(0, "symbol")
				.column(1, "name")
				.column(2, "price", new DecimalFormat("#,##0"))
				.column(3, "volume")
				.column(4, "date", new SimpleDateFormat("yyyy/MM/dd"))
				.column(5, "date", new SimpleDateFormat("HH:mm:ss"))
				.from(new StringReader("AAAA,aaa,10\\,000,10,2008/10/28,10:24:00\r\nBBBB,bbb,NULL,0,NULL,NULL"));

		assertThat(list.size(), is(2));
		final SampleBean o1 = list.get(0);
		assertThat(o1.symbol, is("AAAA"));
		assertThat(o1.name, is("aaa"));
		assertThat(o1.price.longValue(), is(10000L));
		assertThat(o1.volume.longValue(), is(10L));
		assertThat(new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(o1.date), is("2008/10/28 10:24:00"));
		final SampleBean o2 = list.get(1);
		assertThat(o2.symbol, is("BBBB"));
		assertThat(o2.name, is("bbb"));
		assertNull(o2.price);
		assertThat(o2.volume.longValue(), is(0L));
		assertNull(o2.date);
	}

	@Test
	public void testSave() throws Exception {
		final List<SampleBean> list = new ArrayList<SampleBean>();
		list.add(new SampleBean("AAAA", "aaa", 10000, 10, new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").parse("2008/10/28 10:24:00")));
		list.add(new SampleBean("BBBB", "bbb", null, 0, null));
		list.add(new SampleBean("CCCC", "ccc", 20000, 100, new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").parse("2008/10/26 14:20:10")));

		final StringWriter sw = new StringWriter();
		new CsvColumnPositionMappingBeanManager(cfg)
			.save(list, SampleBean.class)
			.column("symbol")
			.column("price", new DecimalFormat("#,##0"))
			.column("volume")
			.column("date", new SimpleDateFormat("yyyy/MM/dd"))
			.column("date", new SimpleDateFormat("HH:mm:ss"))
			.to(sw);

		assertThat(sw.getBuffer().toString(), is("symbol,price,volume,date,date\r\nAAAA,10\\,000,10,2008/10/28,10:24:00\r\nBBBB,NULL,0,NULL,NULL\r\nCCCC,20\\,000,100,2008/10/26,14:20:10\r\n"));
	}

}
