/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 *
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 *
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import jp.sf.orangesignal.csv.CsvConfig;
import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;

import org.junit.BeforeClass;
import org.junit.Test;

/**
 * {@link CsvEntityListHandler} クラスの単体テストです。
 *
 * @author 杉澤 浩二
 */
public class CsvEntityListHandlerTest {

	private static CsvConfig cfg;

	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		cfg = new CsvConfig(',', '"', '\\');
		cfg.setNullString("NULL");
		cfg.setBreakString("\n");
		cfg.setIgnoreTrailingWhitespaces(true);
		cfg.setIgnoreLeadingWhitespaces(true);
		cfg.setIgnoreEmptyLines(true);
		cfg.setIgnoreLinePatterns(Pattern.compile("^#.*$"));
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCsvEntityListHandlerIllegalArgumentException1() {
		new CsvEntityListHandler<SampleEntity1>(null);
	}

	@Test(expected = IllegalArgumentException.class)
	public void testCsvEntityListHandlerIllegalArgumentException2() {
		new CsvEntityListHandler<SampleBean>(SampleBean.class);
	}

	@Test
	public void testLoad() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader("# text/tab-separated-values   \r\n aaa , \"bbb\" , \"ccc\" \r\n zzz , yyy , NULL \r\n# Copyright 2009 OrangeSignal.   "), cfg);
		try {
			final List<SampleEntity1> list = new CsvEntityListHandler<SampleEntity1>(SampleEntity1.class).load(reader);
			assertThat(list.size(), is(1));
			final SampleEntity1 o = list.get(0);
			assertThat(o.aaa, is("zzz"));
			assertThat(o.bbb, is("yyy"));
			assertNull(o.ccc);
		} finally {
			reader.close();
		}
	}

	@Test
	public void testSave() throws IOException {
		final List<SampleEntity1> list = new ArrayList<SampleEntity1>();
		list.add(new SampleEntity1("aaa", "b\nb\\\\b", "c\"cc"));
		list.add(new SampleEntity1("zzz", "yyy", null));

		final StringWriter sw = new StringWriter();
		final CsvWriter writer = new CsvWriter(sw, cfg);
		try {
			new CsvEntityListHandler<SampleEntity1>(SampleEntity1.class).save(list, writer);
			writer.flush();
			assertThat(sw.getBuffer().toString(), is("\"日付\",\"始値\",\"高値\"\r\n\"aaa\",\"b\nb\\\\b\",\"c\\\"cc\"\r\n\"zzz\",\"yyy\",NULL\r\n"));
		} finally {
			writer.close();
		}
	}

}
